/*
 *  eqtl/gene.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_eqtl_genome.h>
#include <gdl/gdl_eqtl_bayes.h>


gdl_eqtl_bayes_gene *
gdl_eqtl_bayes_gene_alloc (const gdl_string * name,
                           const long start,
                           const long end,
                           const unsigned char strand,
                           const size_t nsig)
{
	gdl_eqtl_bayes_gene * candidate = GDL_CALLOC (gdl_eqtl_bayes_gene, 1);
			
	candidate->name  = gdl_string_clone (name);
	candidate->start = start;
	candidate->end   = end;
	candidate->strand= strand;
	candidate->nsig  = nsig;
	candidate->snps  = GDL_MALLOC (gdl_eqtl_bayes_snp_record *, nsig);
	candidate->proba = GDL_CALLOC (double, nsig);
	candidate->z     = GDL_CALLOC (size_t, nsig);
	
	return candidate;
}

void
gdl_eqtl_bayes_gene_free (gdl_eqtl_bayes_gene * g)
{
	if (g)
	{
			
	}	
}                

gdl_eqtl_bayes_gene_set *
gdl_eqtl_bayes_gene_set_alloc (const gdl_eqtl_genome * eqtl,
                               const long window_size,
                               const size_t max_probe,
                               const long gene_size_min,
                               const long gene_size_max,
                               const gdl_boolean qnorm_flag,
                               const double bf_threshold,
                               const size_t significant_max,
                               gdl_eqtl_bayes_workspace * w)
{
	size_t i, j, k, l, p, g, nfunc=0;
	double pos;
	gdl_hashtable * candidate_table;
	
	candidate_table = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	for (i = 0; i < eqtl->nchrom; i++)
	{
		gdl_eqtl_chromosome * chrom;
		gdl_snp ** snps;
		
		chrom = gdl_eqtl_genome_get (eqtl, i);
		snps  = chrom->snp_data->chrom->snps;
		
		gdl_eqtl_chromosome_anchor_gene (chrom);
		
		gdl_eqtl_chromosome_init_window (chrom, 0, window_size);
		
		w->state = gdl_eqtl_bayes_alloc (chrom->nindiv, chrom->npop);
		
		for (j = 0; j < chrom->ngene; j++)
		{
			gdl_eqtl_gene * gene = chrom->genes[j];
			
			gdl_eqtl_block * block = gene->blocks[0];
				
			if (block->size > max_probe) continue;
				
			long gene_size = block->end-block->start;
				
			if (gene_size < gene_size_min || gene_size >= gene_size_max) continue;
				
			gdl_vector_uint * sidx = gdl_genex_block_get_record_snp (block, 0);
				
			if (!sidx) continue;
			
			size_t nsig = 0, * sigidx = GDL_MALLOC (size_t, sidx->size);
			double x, * bf = GDL_MALLOC (double, sidx->size);
			
			gdl_eqtl_bayes_gene_init (w->state, chrom, block, qnorm_flag);
				
			for (l = 0; l < sidx->size; l++)
			{
				gdl_eqtl_bayes_snp_init (w->state, chrom, gdl_vector_uint_get (sidx, l), 0);
				for (x = 0, g = 0; g < w->grid->size; g++)
				{
					x += w->grid->weight[g]*gdl_eqtl_bayes_compute_factor2 (w->state, w->grid->sigmaa[g], 0.25*w->grid->sigmaa[g], w->grid->sigmap);
				}
				if (x > bf_threshold)
				{
					bf[nsig]     = x;
					sigidx[nsig] = gdl_vector_uint_get (sidx, l);
					nsig++;
				}
			}	
			
			if (!nsig || nsig > significant_max)
			{
				gdl_vector_uint_free (sidx);
				continue;
			}
			
			gdl_eqtl_bayes_gene * candidate = gdl_eqtl_bayes_gene_alloc (gene->name,
			                                                             block->start,
			                                                             block->end,
			                                                             block->strand,
			                                                             nsig);
			
			// Store the gene information
			gdl_hashtable_add (candidate_table, candidate->name, candidate, 0);
			// loop on all the SNP of the region
			// store the significant SNP
			// and update bins
			for (p = 0, l = block->snp_up; l <= block->snp_down; l++)
			{
				if (snps[l]->ignore == 'y') continue;
				
				gdl_eqtl_bayes_bin * bin = gdl_eqtl_bayes_workspace_get_bin (w, block, snps[block->snp_up], snps[block->snp_down], snps[l], &pos);
				
				bin->nsnp++;
				
				if (p < nsig && sigidx[p] == l)
				{
					candidate->snps[p] = gdl_eqtl_bayes_bin_add_functional_snp (bin, w->grid->size, i, l, snps[l]);
					candidate->snps[p]->position = pos;
					candidate->snps[p]->bf = bf[p];
					p++;
				}
				else
				{
					// null dico
				}
			}
			
			GDL_FREE (bf);
			GDL_FREE (sigidx);
			gdl_vector_uint_free (sidx);		
			
		}
		
		gdl_eqtl_bayes_free (w->state);
		
		gdl_eqtl_chromosome_free (chrom);
	}
	
	gdl_eqtl_bayes_gene_set * set = 0;
	
	if (gdl_hashtable_size (candidate_table))
	{
		set = GDL_MALLOC (gdl_eqtl_bayes_gene_set, 1);
		
		set->size = gdl_hashtable_size (candidate_table);
		
		gdl_eqtl_bayes_gene ** genes  = GDL_MALLOC (gdl_eqtl_bayes_gene *, set->size);
		 
		gdl_hashtable_itr * itr = gdl_hashtable_iterator (candidate_table);
		i=0;
		do
		{
			double x;
			genes[i] = (gdl_eqtl_bayes_gene *) gdl_hashtable_iterator_value (itr);
			for (x = j = 0; j < genes[i]->nsig; x += genes[i]->snps[j]->bf, j++);
			for (j = 0; j < genes[i]->nsig; j++)
			{
				genes[i]->proba[j]=genes[i]->snps[j]->bf/x;
				genes[i]->snps[j]->gamma = genes[i]->proba[j];
			}
			i++;
		}
		while(gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
		
		set->genes = genes;
	}
	
	return set;
}
