/*
 *  eqtl/result.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_sort_double.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_bayesian_regression.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>
#include <gdl/gdl_eqtl_bayes.h>

gdl_eqtl_bayes_snp_result *
gdl_eqtl_bayes_snp_result_alloc (const size_t grid_size)
{
	gdl_eqtl_bayes_snp_result * r;
	
	r = GDL_CALLOC (gdl_eqtl_bayes_snp_result, 1);
	
	r->bf = GDL_CALLOC (double, grid_size);
	
	return r;	
}

void
gdl_eqtl_bayes_snp_result_free (gdl_eqtl_bayes_snp_result * r)
{
	if (r)
	{
		GDL_FREE (r->bf);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_snp_result *
gdl_eqtl_bayes_snp_result_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		gdl_eqtl_bayes_snp_result * r;
		
		r = gdl_eqtl_bayes_snp_result_alloc (grid_size);
		
		status = fread (r->bf, sizeof(double), grid_size, stream);
		GDL_FREAD_STATUS (status, grid_size);
		status = fread (&r->pval, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return r;	
	}
	return 0;
}

int
gdl_eqtl_bayes_snp_result_fwrite (FILE * stream, const gdl_eqtl_bayes_snp_result * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		
		status = fwrite (r->bf, sizeof(double), grid_size, stream);
		GDL_FWRITE_STATUS (status, grid_size);
		status = fwrite (&r->pval, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/**
 * 
 * Probes
 * 
 */

gdl_eqtl_bayes_probe_result *
gdl_eqtl_bayes_probe_result_alloc (const size_t size)
{
	gdl_eqtl_bayes_probe_result * r;
	
	r = GDL_MALLOC (gdl_eqtl_bayes_probe_result, 1);
	
	r->size = size;
	r->snps = GDL_CALLOC (gdl_eqtl_bayes_snp_result *, size);
	
	return r;
}

void
gdl_eqtl_bayes_probe_result_free (gdl_eqtl_bayes_probe_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_bayes_snp_result_free (r->snps[i]);
		GDL_FREE (r->snps);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_probe_result *
gdl_eqtl_bayes_probe_result_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		gdl_eqtl_bayes_probe_result * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_bayes_probe_result_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			r->snps[i] = gdl_eqtl_bayes_snp_result_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->snps[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_bayes_probe_result_fwrite (FILE * stream, const gdl_eqtl_bayes_probe_result * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_bayes_snp_result_fwrite (stream, r->snps[i], grid_size);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/**
 * 
 * Blocks
 * 
 */

gdl_eqtl_bayes_block_result *
gdl_eqtl_bayes_block_result_alloc (const size_t size, const long snp_from, const long snp_to)
{
	gdl_eqtl_bayes_block_result * r;
	
	r = GDL_MALLOC (gdl_eqtl_bayes_block_result, 1);
	
	r->size     = size;
	r->snp_from = snp_from;
	r->snp_to   = snp_to;
	r->probes   = GDL_CALLOC (gdl_eqtl_bayes_probe_result *, size);
	
	return r;
}

void
gdl_eqtl_bayes_block_result_free (gdl_eqtl_bayes_block_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_bayes_probe_result_free (r->probes[i]);
		GDL_FREE (r->probes);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_block_result *
gdl_eqtl_bayes_block_result_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		long from, to;
		gdl_eqtl_bayes_block_result * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&from, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&to, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_bayes_block_result_alloc (size, from, to);
		
		for (i = 0; i < size; i++)
		{
			r->probes[i] = gdl_eqtl_bayes_probe_result_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->probes[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_bayes_block_result_fwrite (FILE * stream, const gdl_eqtl_bayes_block_result * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->snp_from, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->snp_to, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_bayes_probe_result_fwrite (stream, r->probes[i], grid_size);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/**
 * 
 * Genes
 * 
 */

gdl_eqtl_bayes_gene_result *
gdl_eqtl_bayes_gene_result_alloc (const size_t size)
{
	gdl_eqtl_bayes_gene_result * r;
	
	r = GDL_MALLOC (gdl_eqtl_bayes_gene_result, 1);
	
	r->size = size;
	r->blocks = GDL_CALLOC (gdl_eqtl_bayes_block_result *, size);
	
	return r;
}

void
gdl_eqtl_bayes_gene_result_free (gdl_eqtl_bayes_gene_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_bayes_block_result_free (r->blocks[i]);
		GDL_FREE (r->blocks);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_gene_result *
gdl_eqtl_bayes_gene_result_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		gdl_eqtl_bayes_gene_result * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_bayes_gene_result_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			r->blocks[i] = gdl_eqtl_bayes_block_result_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->blocks[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_bayes_gene_result_fwrite (FILE * stream, const gdl_eqtl_bayes_gene_result * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_bayes_block_result_fwrite (stream, r->blocks[i], grid_size);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/*
 * Chromosome
 */
 
gdl_eqtl_bayes_chromosome_result *
gdl_eqtl_bayes_chromosome_result_alloc (const size_t size)
{
	gdl_eqtl_bayes_chromosome_result * r;
	
	r = GDL_MALLOC (gdl_eqtl_bayes_chromosome_result, 1);
	
	r->size = size;
	r->genes = GDL_CALLOC (gdl_eqtl_bayes_gene_result *, size);
	
	return r;
}

void
gdl_eqtl_bayes_chromosome_result_free (gdl_eqtl_bayes_chromosome_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_bayes_gene_result_free (r->genes[i]);
		GDL_FREE (r->genes);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_chromosome_result *
gdl_eqtl_bayes_chromosome_result_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		gdl_eqtl_bayes_chromosome_result * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_bayes_chromosome_result_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			r->genes[i] = gdl_eqtl_bayes_gene_result_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->genes[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_bayes_chromosome_result_fwrite (FILE * stream, const gdl_eqtl_bayes_chromosome_result * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_bayes_gene_result_fwrite (stream, r->genes[i], grid_size);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/*
 * Genome
 */
 
gdl_eqtl_bayes_genome_result *
gdl_eqtl_bayes_genome_result_alloc (const gdl_string * dir, const size_t size, const gdl_bayreg_model * model)
{
	gdl_eqtl_bayes_genome_result * r;
	
	r = GDL_MALLOC (gdl_eqtl_bayes_genome_result, 1);
	
	r->dir    = gdl_string_clone (dir);
	r->size   = size;
	r->chroms = GDL_CALLOC (gdl_string *, size);
	r->model  = gdl_bayreg_model_clone (model);
	
	return r;
}

void
gdl_eqtl_bayes_genome_result_free (gdl_eqtl_bayes_genome_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_string_free (r->chroms[i]);
		GDL_FREE (r->chroms);
		gdl_bayreg_model_free (r->model);
		gdl_string_free (r->dir);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_genome_result *
gdl_eqtl_bayes_genome_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_eqtl_bayes_genome_result * r;
		
		r = GDL_CALLOC (gdl_eqtl_bayes_genome_result, 1);
		
		r->dir = gdl_string_fread (stream);
		GDL_FREAD_STATUS (r->dir!=0, 1);
		
		status = fread (&r->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r->chroms = GDL_CALLOC (gdl_string *, r->size);
		
		for (i = 0; i < r->size; i++)
		{
			r->chroms[i] = gdl_string_fread (stream);
			GDL_FREAD_STATUS (r->chroms[i]!=0, 1);
		}
		
		r->model = gdl_bayreg_model_fread (stream);
		GDL_FREAD_STATUS (r->model!=0, 1);
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_bayes_genome_result_fwrite (FILE * stream, const gdl_eqtl_bayes_genome_result * r)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = gdl_string_fwrite (stream, r->dir);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_string_fwrite (stream, r->chroms[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		status = gdl_bayreg_model_fwrite (stream, r->model);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

int
gdl_eqtl_bayes_genome_result_set (gdl_eqtl_bayes_genome_result * g, size_t i, const gdl_string * name, const gdl_eqtl_bayes_chromosome_result * chrom)
{
	FILE * stream;
	gdl_string * filename = gdl_string_sprintf ("%s/%s.res", g->dir, name);
	
	gdl_string_free (g->chroms[i]);
	
	g->chroms[i] = gdl_string_clone (name);
	
	stream = gdl_fileopen (filename, "w");
	
	gdl_bayreg_model_bf_storage_size (g->model);
	
	gdl_eqtl_bayes_chromosome_result_fwrite (stream, chrom, g->model->size);
	
	gdl_fileclose (filename, stream);
	
	gdl_string_free (filename);
}

gdl_eqtl_bayes_chromosome_result *
gdl_eqtl_bayes_genome_result_get (const gdl_eqtl_bayes_genome_result * g, size_t i)
{
	FILE * stream;
	gdl_string * filename = gdl_string_sprintf ("%s/%s.res", g->dir, g->chroms[i]);
	gdl_eqtl_bayes_chromosome_result * chrom;
	
	stream = gdl_fileopen (filename, "r");
	
	gdl_bayreg_model_bf_storage_size (g->model);
	
	chrom = gdl_eqtl_bayes_chromosome_result_fread (stream, g->model->size);
	
	gdl_fileclose (filename, stream);
	
	gdl_string_free (filename);
	
	return chrom;	
}


