/*
 *  eqtl/chi.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>
#include <gdl/gdl_eqtl_genome.h>

void gdl_eqtl_chromosome_get_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq);
double ** gdl_eqtl_chromosome_cis_get_block_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, double ** var, size_t ** pidx, size_t * np);
double gdl_eqtl_chromosome_do_LRT (gdl_eqtl_chromosome * chrom, const double * x, const double * y, const double * popf, double * a, double * r2, double * p);

void
gdl_eqtl_chromosome_trans_snp_test (gdl_eqtl_chromosome * chrom, gdl_eqtl_chromosome * chrom2, const double pmin, const long cis_window)
{
	const size_t G    = chrom2->ngene;
	const size_t P    = chrom2->ploidy;
	const size_t N    = chrom2->nindiv;
	const size_t POP  = chrom2->npop;
	const size_t NSNP = chrom->snp_data->chrom->size;
	size_t s, i, j, k, p, l, * pidx, np, * minl;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double * popf, f, chi, t, pv, r2, * x, ** Y, * var;
	double * minpv;
	gdl_string * kind;
	
	snps = chrom->snp_data->chrom->snps;
	
	x = GDL_MALLOC (double, N);
	popf = GDL_MALLOC (double, POP);
	
	for (s = 0; s < NSNP; s++)
	{ 
		if (snps[s]->ignore == 'y') continue;
				
		gdl_eqtl_chromosome_get_snp_vector (chrom, s, x, popf, &f);
					
		for (i = 0; i < G; i++)
		{
			gene = chrom2->genes[i];
			
			if (gene->ignore == 'y') continue;
			
			for (j = 0; j < gene->size; j++)
			{
				block = gene->blocks[j];
				
				if (block->ignore == 'y') continue;
				
				Y = gdl_eqtl_chromosome_cis_get_block_signal (chrom2, block, &var, &pidx, &np);
				
				kind = (snps[s]->position >= block->start-cis_window && snps[s]->position <= block->end+cis_window) ? "cis" : "trans";
					
				for (p = 0; p < np; p++)
				{
					
					chi = gdl_eqtl_chromosome_do_LRT (chrom2, x, Y[p], popf, &t, &r2, &pv);
					
					if (pv < pmin)
					{
						fprintf (chrom->logger, "%s %s %s %d %s %s %d %d %d %d %c %s %d %d %1.5f",kind,chrom->name,snps[s]->rs,snps[s]->position,chrom2->name,gene->name,gene->id,j+1,block->start,block->end,block->strand,block->probes[pidx[p]]->name, block->probes[pidx[p]]->start, block->probes[pidx[p]]->end, f/(double)P);
						for (l = 0; l < chrom->npop; l++)
						{
							fprintf (chrom->logger, " %1.5f", popf[l]/(double)P);
						}
						fprintf (chrom->logger, " %g %g %g %e", chi, t, r2, pv);
						fprintf(chrom->logger, "\n");
					}
				}
				
				GDL_MATRIX_FREE (Y, np);
				GDL_FREE (var);
				GDL_FREE (pidx);
			}
		}
	}

}

void
gdl_eqtl_chromosome_trans_test (gdl_eqtl_genome * genome, gdl_eqtl_chromosome * chrom, const size_t c, const double pmin, const long cis_window)
{
	size_t i;
	
	for (i = 0; i < genome->nchrom; i++)
	{
		gdl_eqtl_chromosome * chrom2;
		
		if (i != c)
			chrom2 = gdl_eqtl_genome_get (genome, i);
		else
			chrom2 = chrom;
		
		gdl_eqtl_chromosome_trans_snp_test (chrom, chrom2, pmin, cis_window);
		
		if (i != c)
			gdl_eqtl_chromosome_free (chrom2);
	}
}
