/*
 *  eqtl/util.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>

void
gdl_eqtl_chromosome_cis_get_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq)
{
	size_t i, j, k, p;
	*freq = 0;
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(pop_freq[i] = 0, j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			switch(gdl_snp_chromosome_get_genotype (chrom->snp_data, i, j, snp))
			{
				case 2:
					x[k]=2;
					pop_freq[i]+=2;
					break;
				case 1:
					x[k]=1;
					pop_freq[i]+=1;
					break;
				case 0:
					x[k]=0;
					break;
			}
		}
		*freq += pop_freq[i];
		pop_freq[i] /= chrom->pop_sizes[i];
	}
	*freq /= chrom->nindiv;
}

void
gdl_eqtl_chromosome_get_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq)
{
	size_t i, j, k, p;
	
	*freq = 0;
	for (k = i = 0; i < chrom->npop; i++)
	{
		pop_freq[i] = 0;
		if (!gdl_snp_chromosome_is_polymorphic (chrom->snp_data, i, snp))
		{
			continue;
		}
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			switch(gdl_snp_chromosome_get_genotype (chrom->snp_data, i, j, snp))
			{
				case 2:
					x[k]=2;
					pop_freq[i]+=2;
					break;
				case 1:
					x[k]=1;
					pop_freq[i]+=1;
					break;
				case 0:
					x[k]=0;
					break;
			}
		}
		*freq += pop_freq[i];
		pop_freq[i] /= (double)chrom->pop_sizes[i];
	}
	*freq /= (double)k;
}

double ** 
gdl_eqtl_chromosome_cis_get_block_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, double ** var, size_t ** pidx, size_t * np)
{
	size_t i, j, k, p, pp;
	double ** Y, s, m;
	
	for (*np = p = 0; p < block->size; p++)
	{
	   if (block->probes[p]->ignore=='n') (*np)++;
	}
	
	Y     = GDL_MATRIX_ALLOC (double, *np, chrom->nindiv+1);
	*pidx = GDL_MALLOC (size_t, *np);
	*var  = GDL_MALLOC (double, *np);
	
	for (pp = p = 0; p < block->size; p++)
	{
		if (block->probes[p]->ignore == 'y') continue;
		(*pidx)[pp] = p;
		(*var)[pp]  = 0;
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
			{
				Y[pp][k]  = block->probes[p]->data[i][j];
				Y[pp][k] -= block->probes[p]->mean[i];
				(*var)[pp]  += Y[pp][k]*Y[pp][k];
			}
			//var[p] += (chrom->pop_sizes[i]-1)*block->probes[p]->var[i]/chrom->nindiv;
		}
		(*var)[pp] /= chrom->nindiv; 
		pp++;
	}
	
	return Y;
}

double
gdl_eqtl_chromosome_do_LRT (gdl_eqtl_chromosome * chrom, const double * x, const double * y, const double * popf, double * a, double * r2, double * p)
{
	size_t i, j, kx, ky;
	double l0, l1, chi, e, sig0=0, sig1=0, xx=0, xy=0;
	
	for (kx = ky = i = 0; i < chrom->npop; i++)
	{
		if (!popf[i]) 
		{
			ky += chrom->pop_sizes[i];
			continue;
		}
		for(j = 0; j < chrom->pop_sizes[i]; j++, kx++, ky++)
		{
			xx += (x[kx]-popf[i])*(x[kx]-popf[i]);
			xy += (x[kx]-popf[i])*y[ky];
			sig0 += y[ky]*y[ky];
		}
	}
	*a=xy/xx;
	for (kx = ky = i = 0; i < chrom->npop; i++)
	{
		if (!popf[i])
		{
			ky += chrom->pop_sizes[i];
			continue;
		}
		for(j = 0; j < chrom->pop_sizes[i]; j++, kx++, ky++)
		{
			 e = y[ky]-(*a)*(x[kx]-popf[i]);
			 sig1 += e*e;
		}
	}
	chi = (log(sig0)-log(sig1))*chrom->nindiv;
	chi = (chi >= 0) ? chi : 0; // in case of some numerical errors if sig0=sig1...
	*p=gdl_sf_gamma_inc_Q (0.5,0.5*chi);
	*r2=1-sig1/sig0;
	return chi;
}

double
gdl_eqtl_chromosome_gene_LRT (gdl_eqtl_chromosome * chrom, const double * y1, const double * y2, const double sig0, double * a, double * r2, double * p)
{
	size_t i, j, k;
	double l0, l1, chi, e, sig1=0, xx=0, xy=0;
	
	for (k = i = 0; i < chrom->nindiv; i++)
	{
		xx += y2[i]*y2[i];
		xy += y1[i]*y2[i];
	}
	*a=xy/xx;
	for (k = i = 0; i < chrom->nindiv; i++)
	{
		e = y1[i]-(*a)*y2[i];
		sig1 += e*e;
	}
	sig1 /= chrom->nindiv;
	chi = (log(sig0)-log(sig1))*chrom->nindiv;
	chi = (chi >= 0) ? chi : 0; // in case of some numerical errors if sig0=sig1...
	*p=gdl_sf_gamma_inc_Q (0.5,0.5*chi);
	*r2=1-sig1/sig0;
	return chi;
}
