/*
 *  eqtl/region.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_eqtl_bayes.h>

gdl_eqtl_bayes_workspace *
gdl_eqtl_bayes_workspace_alloc (const long window, const long bin)
{
	size_t i;
	gdl_eqtl_bayes_workspace * r;
	
	r = GDL_CALLOC (gdl_eqtl_bayes_workspace, 1);
	
	r->window = window;
	r->bin = bin;
	r->size = 2*window/bin+1;
	r->bins = GDL_MALLOC (gdl_eqtl_bayes_bin *, r->size);
	for (i = 0; i < r->size; i++)
	{
		r->bins[i] = gdl_eqtl_bayes_bin_alloc (i);
		if (i != (r->size-1)/2) r->bins[i]->length = bin;
		else r->bins[i]->gene = gdl_true;
	}
	r->grid = gdl_bayreg_grid_alloc ();
	
	return r;
}

void
gdl_eqtl_bayes_workspace_free (gdl_eqtl_bayes_workspace * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
		{
			gdl_eqtl_bayes_bin_free (r->bins[i]);
		}
		GDL_FREE (r->bins);
		gdl_bayreg_grid_free (r->grid);
		GDL_FREE (r);
	}
}

gdl_eqtl_bayes_bin *
gdl_eqtl_bayes_workspace_get_bin (const gdl_eqtl_bayes_workspace * r, const gdl_eqtl_block * block, const gdl_snp * snp_start, const gdl_snp * snp_end, const gdl_snp * snp, double * pos)
{
	long q, i;
	
	if (snp->position >= block->start && snp->position <= block->end)
	{
		i = (r->size-1)/2;
		if (block->strand == '-') *pos = block->end-snp->position;
		else *pos = snp->position-block->start;
		*pos /= (block->end-block->start);
	}
	else if (snp->position < block->start)
	{
		i = q = (snp->position-snp_start->position)/r->bin;
		if (block->strand == '-') i = r->size - q - 1;
		*pos = ((double)i)-(r->size-1)/2;
		*pos += ((double)((snp->position-snp_start->position)-r->bin*q))/(double)r->bin;
	}
	else 
	{
		i = r->size - (snp_end->position - snp->position)/r->bin - 1;
		if (block->strand == '-') i = r->size - i - 1;
		*pos = ((double)i)-(r->size-1)/2;
		q = (snp_end->position-snp->position)/r->bin;
	   *pos += ((double)((snp_end->position-snp->position)-r->bin*q))/(double)r->bin;
	}
	
	return r->bins[i];
}

gdl_eqtl_bayes_workspace *
gdl_eqtl_bayes_workspace_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		long window, bin;
		gdl_eqtl_bayes_workspace * w;
		
		status = fread (&window, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&bin, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		w = gdl_eqtl_bayes_workspace_alloc (window, bin);
		
		for (i = 0; i < w->size; i++)
		{
			status = gdl_eqtl_bayes_bin_fread (stream, w->bins[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS); 
		}
		
		w->null_dico = gdl_snp_annot_dico_fread (stream);
		GDL_FREAD_STATUS (w->null_dico!=0, 1);
		
		return w;
	}
	
	return 0;
}

int
gdl_eqtl_bayes_workspace_fwrite (FILE * stream, const gdl_eqtl_bayes_workspace * w)
{
	if (stream && w)
	{
		int status;
		size_t i;
		
		status = fwrite (&w->window, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&w->bin, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < w->size; i++)
		{
			status = gdl_eqtl_bayes_bin_fwrite (stream, w->bins[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		status = gdl_snp_annot_dico_fwrite (stream, w->null_dico);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}
