/*  
 * 	sheet/iterator.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_fsheet.h>

struct _gdl_fsheet_itr 
{
	int ridx;
	int cidx;
	size_t rnew;
	size_t cnew;
	long * rftell;
	char mode;
	char ** names;
	char * nbuff;
	char * cbuff;
	gdl_fsheet * fs;
};

gdl_fsheet_itr *
gdl_fsheet_itr_alloc (void)
{
	gdl_fsheet_itr * itr;
	
	itr = GDL_MALLOC (gdl_fsheet_itr, 1);
	itr->names  = NULL;
	itr->nbuff  = NULL;
	itr->cbuff  = NULL;
	itr->rftell = NULL;
	itr->rnew   = 1;
	itr->cnew   = 1;	
	return itr;	
}

void
gdl_fsheet_iterator_free (gdl_fsheet_itr * itr)
{
	if (itr != NULL && itr->fs != NULL)
	{
		if (itr->names)
		{
			size_t i;
			size_t n;
			switch (itr->mode)
			{
				case 'r':
					n = itr->fs->nc;
					break;
				case 'c':
					n = itr->fs->nr;
					break;
			}
			for (i = 0; i < n; i++)
				GDL_FREE (itr->names[i]);
			GDL_FREE (itr->names);
		}
		GDL_FREE (itr->nbuff);
		GDL_FREE (itr->cbuff);
		GDL_FREE (itr->rftell);
		GDL_FREE (itr);
	}
}

gdl_fsheet_itr *
gdl_fsheet_iterator_c (gdl_fsheet * fs)
{
	if (fs == 0)
	{
		return NULL;
	}
	else
	{
		size_t i;
		gdl_fsheet_itr * itr = gdl_fsheet_itr_alloc ();
		
		gdl_fsheet_rewind (fs);
		
		itr->fs   = fs;
		itr->ridx = 0;
		itr->cidx = 0;
		itr->mode = 'c';
		
		itr->rftell = GDL_MALLOC (long, fs->nr);
		
		if (fs->type->rname)
		{
			itr->names = GDL_CALLOC (char *, fs->nr);	
		}
		
		if (fs->type->cname)
		{
			gdl_fsheet_parse_header (fs);
			gdl_fsheet_next_field (fs);
			itr->nbuff = gdl_string_clone (fs->_buffer);
		}
		
		gdl_fsheet_jump_header (fs);
			
		i = 0;
			
		do
		{
			gdl_fsheet_parse_cell (fs);
			
			if (fs->type->rname)
			{
				gdl_fsheet_next_field (fs);
				itr->names[i]  = gdl_string_clone (fs->_buffer);
			}
			
			itr->rftell[i] = gdl_fsheet_ftell (fs);
			
			if (!i)
			{
				gdl_fsheet_next_field (fs);
				itr->cbuff     = gdl_string_clone (itr->fs->_buffer);
				itr->rftell[i] = gdl_fsheet_ftell (fs);
			}
			
			i++;
		}
		while (gdl_fsheet_next_row (fs));
		
		gdl_fsheet_rewind (itr->fs);
				
		return itr;
	}
}

gdl_fsheet_itr *
gdl_fsheet_iterator_r (gdl_fsheet * fs)
{
	if (fs == 0)
	{
		return NULL;
	}
	else
	{
		gdl_fsheet_itr * itr = gdl_fsheet_itr_alloc ();
		
		gdl_fsheet_rewind (fs);
		
		itr->fs   = fs;
		itr->ridx = 0;
		itr->cidx = 0;
		itr->mode = 'r';
		
		if (fs->type->cname)
		{
			size_t i;
			
			itr->names = GDL_CALLOC (char *, fs->nc);
			
			gdl_fsheet_parse_header (fs);
			
			i = 0;
			
			while (gdl_fsheet_next_field (fs))
			{
				itr->names[i] = gdl_string_clone (fs->_buffer);				
				i++;
			};
			itr->names[i] = gdl_string_clone (fs->_buffer);			
			
		}
		
		gdl_fsheet_parse_cell (fs);
			
		if (fs->type->rname)
		{	
			
			gdl_fsheet_next_field (fs);
			
			itr->nbuff = gdl_string_clone (fs->_buffer);
			
		}
		
		gdl_fsheet_next_field (fs);
		
		itr->cbuff = gdl_string_clone (itr->fs->_buffer);
		
		return itr;
	}
}

char *
gdl_fsheet_iterator_column (gdl_fsheet_itr * itr)
{
	switch (itr->mode)
	{
		case 'r' :
			return (itr->names == NULL) ? NULL : itr->names[itr->cidx];
		case 'c' :
			return (itr->nbuff == NULL) ? NULL : itr->nbuff;
	}
}

char *
gdl_fsheet_iterator_row (gdl_fsheet_itr * itr)
{
	switch (itr->mode)
	{
		case 'c' :
			return (itr->names == NULL) ? NULL : itr->names[itr->ridx];
		case 'r' :
			return (itr->nbuff == NULL) ? NULL : itr->nbuff;
	}
}

char *
gdl_fsheet_iterator_cell (gdl_fsheet_itr * itr)
{
	return itr->cbuff;
}

static int
gdl_fsheet_iterator_next_r (gdl_fsheet_itr * itr)
{
	int status;

#define NEXT_R { status = gdl_fsheet_next_field (itr->fs); \
		if (status < 0) {return 0;} \
		GDL_FREE (itr->cbuff); \
		itr->cbuff = gdl_string_clone(itr->fs->_buffer);}
	
	if (itr->mode == 'r')
	{
		if (itr->cidx == itr->fs->nc - 1)
		{
			itr->cidx = 0;
			itr->ridx++;
			if (itr->ridx == itr->fs->nr)
			{
				return 0;
			}
			
			gdl_fsheet_parse_cell (itr->fs);
			
			if (itr->fs->type->rname)
			{
				GDL_FREE (itr->nbuff);
				gdl_fsheet_next_field (itr->fs);
				itr->nbuff = gdl_string_clone (itr->fs->_buffer);
			}
			
			NEXT_R
			itr->rnew = 1;
			return 1;			
		}
		
		NEXT_R
		itr->cidx++;
		if (itr->rnew)
			itr->rnew = 0;
		return 1;				
	}
	else
	{
		return 0;	
	}

#undef NEXT_R	

}

static int
gdl_fsheet_iterator_next_c (gdl_fsheet_itr * itr)
{
	if (itr->mode == 'c')
	{
		if (itr->ridx == itr->fs->nr - 1)
		{
			itr->ridx = 0;
			itr->cidx++;
			
			if (itr->cidx == itr->fs->nc)
			{
				return 0;
			}
			
			gdl_fsheet_rewind (itr->fs);
			
			// get the column name
			if (itr->fs->type->cname)
			{
				size_t c = 0;
				gdl_fsheet_parse_header (itr->fs);
				do {
					gdl_fsheet_next_field (itr->fs);
					c++;
				} while ( c <= itr->cidx);
				GDL_FREE (itr->nbuff);
				itr->nbuff = gdl_string_clone (itr->fs->_buffer);
				gdl_fsheet_parse_cell_2 (itr->fs);
			}
			
			gdl_fsheet_jump_header (itr->fs);
			
			itr->cnew = 1;
		}
		else
		{
			itr->ridx++;
			if (itr->cnew)
				itr->cnew = 0;			
		}
		gdl_fsheet_fseek (itr->fs, itr->rftell[itr->ridx]);
		gdl_fsheet_next_field (itr->fs);
		GDL_FREE (itr->cbuff);
		itr->cbuff = gdl_string_clone (itr->fs->_buffer);
		itr->rftell[itr->ridx] = gdl_fsheet_ftell (itr->fs);		
		return 1;
	}
	else
	{
		return 0;	
	}
}

int
gdl_fsheet_iterator_next (gdl_fsheet_itr * itr)
{
	switch (itr->mode)
	{
		case 'c' :
			return gdl_fsheet_iterator_next_c (itr);
		case 'r' :
			return gdl_fsheet_iterator_next_r (itr);
	}	
}

size_t
gdl_fsheet_iterator_column_idx (gdl_fsheet_itr * itr)
{
	return itr->cidx;	
}

size_t
gdl_fsheet_iterator_row_idx (gdl_fsheet_itr * itr)
{
	return itr->ridx;
}

size_t
gdl_fsheet_iterator_is_new_column (gdl_fsheet_itr * itr)
{
	return itr->cnew;
}

size_t
gdl_fsheet_iterator_is_new_row (gdl_fsheet_itr * itr)
{
	return itr->rnew;	
}
