/*  
 * 	fuzzy/gview.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_fuzzy.h>


typedef struct
{
  	int start;
  	size_t K;
  	size_t n;
  	size_t p;
  	size_t l;
  	size_t nres;
  	size_t * na;
  	double abs_res;
  	double sq_res;
  	gdl_hnblock  * f;
  	gdl_hnblock  * uf;
  	gdl_block    * q;
  	gdl_block    * uq;
  	gdl_block    * z;
	gdl_gvalues_get   * gbuf;
	const gdl_fuzzy_param * P;
} gdl_fuzzy_gview_t;

static double
_gdl_fuzzy_gview_get_q (const gdl_fuzzy_gview_t * state, size_t k, size_t i)
{
	return gdl_block_get (state->q, k, i);
}

static double
_gdl_fuzzy_gview_get_f (const gdl_fuzzy_gview_t * state, size_t k, size_t l, size_t a)
{
	return gdl_hnblock_get (state->f, k, l, a);
}


static int
gdl_fuzzy_gview_update_fallele (gdl_fuzzy_gview_t * state,
                                                         size_t k,
                                                         size_t l, 
                                                         size_t a,
                                                         double z,
                                                         double x)
{
	double p = x*z;
	double f = gdl_hnblock_get (state->uf, k, l, a);
	gdl_hnblock_set (state->uf, k, l, a, f + p);
	return 0;
}

static int 
gdl_fuzzy_gview_update_qz(gdl_fuzzy_gview_t * state,
                                                    size_t k,
                                                    size_t i, 
                                                    double z)
{
	double q = gdl_block_get (state->uq, k, i);
	gdl_block_set (state->uq, k, i, q + z);
	return 0;
}                                              

static double
gdl_fuzzy_gview_get_pr (gdl_fuzzy_gview_t * state,
                                                    size_t k,             
                                                    size_t i,
                                                    size_t j,
                                                    size_t l) 
{
	size_t ii, a, na;
	double f, q, pr;
	const gdl_gvalues * x;
	
	pr = 0.;
	
	ii = gdl_clustering_clust_idx (state->P->gclust, i);
	
	GDL_GVIEW_GET_ALLELE (state->P->gview, state->P->gmask, ii, l, j, state->gbuf);
	 
	x = gdl_gvalues_get_gvalues (state->gbuf);
		
	if ( x != 0 ) 
	{
		for ( a = 0; a < x->size; a++)
		{
			f   = _gdl_fuzzy_gview_get_f (state, k, l, x->values[a]->idx);
			pr += f*x->values[a]->value;
		}
		
		return pr;
	}	
	
	return 1.0;
}

static void
gdl_fuzzy_gview_set_f (gdl_fuzzy_gview_t * state,
                                                    size_t k,            
                                                    size_t l,
                                                    size_t a,
                                                    double f)
{
	gdl_hnblock_set (state->f, k, l, a, f);
}  

static void
gdl_fuzzy_gview_set_q (gdl_fuzzy_gview_t * state,
                                                    size_t k,             
                                                    size_t i,
                                                    double q) 
{
	gdl_block_set (state->q, k, i, q);	
}                                                    

static double
gdl_fuzzy_gview_get_z (const gdl_fuzzy_gview_t * state,
                                                    size_t k,          
                                                    size_t i,
                                                    size_t j,
                                                    size_t l) 
{
	return gdl_block_get (state->z, k, i, j, l);
}

static void
gdl_fuzzy_gview_set_z (gdl_fuzzy_gview_t * state,
                                                    size_t k,          
                                                    size_t i,
                                                    size_t j,
                                                    size_t l,
                                                    double z) 
{
	gdl_block_set (state->z, k, i, j, l, z);
}


static int
gdl_fuzzy_gview_iterate_start (gdl_fuzzy_gview_t * state)
{
	size_t i, j, k, l;
	
	// Clean the buffers
	for ( k = 0; k < state->K; k++)
	{
		for ( i = 0; i < state->n; i++)
		{
			for ( j = 0; j < state->p; j++)
			{
				gdl_block_set (state->uq, k, i, j, 0.);
			}
		}	
	}
	for ( k = 0; k < state->K; k++)
	{
		for ( i = 0; i < state->l; i++)
		{
			
			for ( j = 0; j < state->na[i]; j++)
			{
				gdl_hnblock_set (state->uf, k, i, j, 0.);
			}	
		}
	}
	
	state->abs_res = 0.;
	state->sq_res  = 0.;
	state->nres    = 0;
	
	return 0;
}

static int
gdl_fuzzy_gview_update_z (gdl_fuzzy_gview_t * state,
                                                    size_t i,
                                                    size_t j,
                                                    size_t l)
{
	size_t k;
	double pr, q, z, s = 0;
	
	for (k = 0; k < state->K; k++)
	{
		pr = gdl_fuzzy_gview_get_pr (state, k, i, j, l);
		q  = _gdl_fuzzy_gview_get_q (state, k, i);
		z  = q*pr;
		s += z;
		gdl_fuzzy_gview_set_z (state, k, i, j, l, z);
	}
	for (k = 0; k < state->K; k++)
	{
		z = gdl_fuzzy_gview_get_z (state, k, i, j, l);
		
		gdl_fuzzy_gview_set_z (state, k, i, j, l, z/s);
	}
	
	return 0;	
}

static int
gdl_fuzzy_gview_update_f (gdl_fuzzy_gview_t * state,
                                                    size_t i,
                                                    size_t j,
                                                    size_t l)
{
	size_t ii, k, a, na, ni;
	double z;
	const gdl_gvalues * x;
	
	ii  = gdl_clustering_clust_idx (state->P->gclust, i);
	ni  = gdl_clustering_clust_size (state->P->gclust, i);
	
	for (k = 0; k < state->K; k++)
	{
		
		z   = gdl_fuzzy_gview_get_z (state, k, i, j, l);
		
		GDL_GVIEW_GET_ALLELE (state->P->gview, state->P->gmask, ii, l, j, state->gbuf);
		
		x = gdl_gvalues_get_gvalues (state->gbuf);
		
		if ( x != 0 ) 
		{
			for ( a = 0; a < x->size; a++)
			{
				gdl_fuzzy_gview_update_fallele(state, k, l, x->values[a]->idx, z, ni*x->values[a]->value);
			}
		}
				
	}
	
	return 0;
}

static int
gdl_fuzzy_gview_update_q (gdl_fuzzy_gview_t * state,
                                                    size_t i,
                                                    size_t j,
                                                    size_t l)
{
	size_t k;
	
	for (k = 0; k < state->K; k++)
	{
		
		double z = gdl_fuzzy_gview_get_z (state, k, i, j, l);
		
		//printf ("UPDATE Q [%d][%d] + %g\n", k, i, z);
		
		gdl_fuzzy_gview_update_qz (state, k, i, z);
		
	}
	
	return 0;
}

static int
gdl_fuzzy_gview_iterate_end_swap (gdl_fuzzy_gview_t * state)
{
	gdl_hnblock  * ftmp;
  	gdl_block    * qtmp;
		
	ftmp      = state->f;
	state->f  = state->uf;
	state->uf = ftmp;
	
	qtmp      = state->q;
	state->q  = state->uq;
	state->uq = qtmp;
	
	return 0;
}

static int
gdl_fuzzy_gview_iterate_end_q (gdl_fuzzy_gview_t * state)
{
	double s, q, e, t, u;
	size_t i, k;
	
	for ( i = 0; i < state->n; i++)
	{
		s = 0;
		for ( k = 0; k < state->K; k++)
		{
			s += _gdl_fuzzy_gview_get_q (state, k, i);
			//printf (">>Q[%d][%d] %g\n", k, i, gdl_fuzzy_gview_get_q (state, k, i));
		}
		if (s)
		{
			for ( k = 0; k < state->K; k++)
			{
				q = _gdl_fuzzy_gview_get_q (state, k, i);
				t = q/s;
				if ( t < 1.e-20)
				{
					t = q = 0.;
				}
				//printf ("Q[%d][%d] %g\n", k, i, q/s);
				gdl_fuzzy_gview_set_q (state, k, i, t);
				u = gdl_block_get (state->uq, k, i);
				e = fabs(t - u);
				state->abs_res += e;
				state->sq_res  += e*e;
				(state->nres)++;
			}
		}		
	}
	
	return 0;
}

static int
gdl_fuzzy_gview_iterate_end_f (gdl_fuzzy_gview_t * state)
{
	double s, f, e, t, u;
	size_t i, j, k;
	
	for ( k = 0; k < state->K; k++)
	{
		for ( i = 0; i < state->l; i++)
		{
			s = 0;
			for ( j = 0; j < state->na[i]; j++)
			{
				s += _gdl_fuzzy_gview_get_f (state, k, i, j);
			}
			if (s)
			{
				for ( j = 0; j < state->na[i]; j++)
				{
					f = _gdl_fuzzy_gview_get_f (state, k, i, j);
					t = f/s;
					if ( t < 1.e-20) 
					{
						t = f = 0.;
					}
					gdl_fuzzy_gview_set_f (state, k, i, j, t);
					//printf ("F[%d][%d][%d] %g\n", k, i, j, f/s);
					u = gdl_hnblock_get (state->uf, k, i, j);
					e = fabs(t - u);	
					state->abs_res += e;
					state->sq_res  += e*e;
					(state->nres)++;
				}
			}
		}
	}
	
	return 0;
}

static int
gdl_fuzzy_gview_iterate_end (gdl_fuzzy_gview_t * state)
{
	int status;
	
	// Swap here the buffer
	
	status  = gdl_fuzzy_gview_iterate_end_swap (state);
	
	status |= gdl_fuzzy_gview_iterate_end_q (state);
	
	status |= gdl_fuzzy_gview_iterate_end_f (state);
	
	state->start = 1;
	
	return status;
}


int
gdl_fuzzy_gview_alloc (void * vstate, const gdl_fuzzy_param * P)
{
	size_t i, * tmp;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
  
	state->start = 1;
	state->K     = P->k;
	state->P     = P;
	state->n     = gdl_clustering_nclust (P->gclust);
	state->p     = gdl_gview_ploidy (P->gview);
	state->gbuf  = GDL_GVIEW_GET_NEW (P->gview, P->gmask);
	state->l     = GDL_GVIEW_LOCUS_SIZE (P->gview, P->gmask);
	state->na    = GDL_CALLOC (size_t, state->l);
	for (i = 0; i < state->l; i++)
	{
		gdl_locus * locus = GDL_GVIEW_GET_LOCUS (P->gview, P->gmask, i);
		state->na[i]      = gdl_locus_allele (locus);
	}
	tmp          = GDL_MALLOC (size_t, 2);
	tmp[0]       = state->K;
	tmp[1]       = state->l;
	state->f     = gdl_hnblock_alloc (1, 1, tmp, &state->na);
	state->q     = gdl_block_alloc2 (2, state->K, state->n);
	state->uf    = gdl_hnblock_alloc (1, 1, tmp, &state->na);
	state->uq    = gdl_block_alloc2 (2, state->K, state->n);
	state->z     = gdl_block_alloc2 (4, state->K, state->n, state->p, state->l);
	GDL_FREE (tmp);
	
    return GDL_SUCCESS;
}

int
gdl_fuzzy_gview_free (void * vstate)
{
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	if (state == 0) return;
	
	gdl_hnblock_free (state->f);
	gdl_block_free (state->q);
	gdl_hnblock_free (state->uf);
	gdl_block_free (state->uq);	
	gdl_block_free (state->z);
	GDL_FREE (state->na);
	gdl_gvalues_get_free (state->gbuf);
	
	return GDL_SUCCESS;
}

int
gdl_fuzzy_gview_init (void * vstate)
{
	int status=0;
	size_t i, j, l, k;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	// Randomly init z
	for (i = 0; i < state->n; i++)
	{
		for (j = 0; j < state->p; j++)
		{
			for (l = 0; l < state->l; l++)
			{
				double z, s = 0.;
				
				for (k = 0; k < state->K; k++)
				{
					z = gdl_rng_uniform (state->P->rng);
					gdl_fuzzy_gview_set_z (state, k, i, j, l, z);
					s += z;
				}
				for (k = 0; k < state->K; k++)
				{
					z = gdl_fuzzy_gview_get_z (state, k, i, j, l);
					gdl_fuzzy_gview_set_z (state, k, i, j, l, z/s);
				}
				status |= gdl_fuzzy_gview_update_f (state, i, j, l);
				status |= gdl_fuzzy_gview_update_q (state, i, j, l);
			}
		}
	}
	
	status |= gdl_fuzzy_gview_iterate_end (state);
	
	return status;
}

int
gdl_fuzzy_gview_iterate (void * vstate)
{
	size_t i, j, l;
	int status;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	status = gdl_fuzzy_gview_iterate_start (state);
	
	for (i = 0; i < state->n; i++)
	{
		for (j = 0; j < state->p; j++)
		{
			for (l = 0; l < state->l; l++)
			{
				status |= gdl_fuzzy_gview_update_z (state, i, j, l);
				status |= gdl_fuzzy_gview_update_f (state, i, j, l);
				status |= gdl_fuzzy_gview_update_q (state, i, j, l);				
			}
		}
	}
	
	status |= gdl_fuzzy_gview_iterate_end (state);
	
	return status;
}

double
gdl_fuzzy_gview_loglikelihood (const void * vstate)
{
	size_t i, j, k, l;
	double s, p, q, v = 0.;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	for (i = 0; i < state->n; i++)
	{
		for (j = 0; j < state->p; j++)
		{
			for (l = 0; l < state->l; l++)
			{
				s = 0;
				for (k = 0; k < state->K; k++)
				{
					p  = gdl_fuzzy_gview_get_pr (state, k, i, j, l);
					q  = _gdl_fuzzy_gview_get_q (state, k, i);
					s += p*q;
				}
				v += log (s);			
			}
		}
	}
	
	return v;
}

double
gdl_fuzzy_gview_residual_sq (const void * vstate)
{
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	return (state->sq_res)/(double)state->nres;
}

double
gdl_fuzzy_gview_residual_abs (const void * vstate)
{
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	return (state->abs_res)/(double)state->nres;
}

double
gdl_fuzzy_gview_get_q (const void * vstate, size_t k, size_t i)
{
	size_t ii;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	ii = gdl_clustering_cluster (state->P->gclust, i);
	return _gdl_fuzzy_gview_get_q (state, k, ii);
}

double
gdl_fuzzy_gview_get_f (const void * vstate, size_t k, size_t l, size_t a)
{
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	return _gdl_fuzzy_gview_get_f (state, k, l, a);
}

size_t
gdl_fuzzy_gview_get_f_max (const void * vstate, size_t pop, size_t loc)
{
	double x, max  = GDL_NEGINF;
	size_t a, amax = 0;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	for (a = 0; a < state->na[loc]; a++)
	{
		x = _gdl_fuzzy_gview_get_f (state, pop, loc, a);
		if (x > max)
		{
			max  = x;
			amax = a;
		}
	}
	
	return amax;
}

size_t
gdl_fuzzy_gview_get_f_size (const void * vstate, size_t loc)
{
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	return state->na[loc];
}

int
gdl_fuzzy_gview_fread (FILE * stream, void * vstate)
{
	int status;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	status = fread (&state->K, sizeof (size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&state->n, sizeof (size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&state->p, sizeof (size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&state->l, sizeof (size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&state->abs_res, sizeof (double), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&state->sq_res, sizeof (double), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	state->na = GDL_ARRAY_FREAD (size_t, state->l, stream);
	GDL_FREAD_STATUS (state->na==0, 0);
	state->f  = gdl_hnblock_fread (stream);
	GDL_FREAD_STATUS (state->f == 0, 0);
	state->uf = gdl_hnblock_fread (stream);
	GDL_FREAD_STATUS (state->uf == 0, 0);
	state->q  = gdl_block_fread (stream);
	GDL_FREAD_STATUS (state->q == 0, 0);
	state->uq = gdl_block_fread (stream);
	GDL_FREAD_STATUS (state->uq == 0, 0);
	state->z  = gdl_block_fread (stream);
	GDL_FREAD_STATUS (state->z == 0, 0);
	
	return GDL_SUCCESS;
}

int
gdl_fuzzy_gview_fwrite (FILE * stream, const void * vstate)
{
	int status;
	gdl_fuzzy_gview_t * state = (gdl_fuzzy_gview_t *) vstate;
	
	status = fwrite (&state->K, sizeof (size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&state->n, sizeof (size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&state->p, sizeof (size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&state->l, sizeof (size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&state->abs_res, sizeof (double), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&state->sq_res, sizeof (double), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = GDL_ARRAY_FWRITE (state->na, size_t, state->l, stream);
	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	status = gdl_hnblock_fwrite (stream, state->f);
	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	status = gdl_hnblock_fwrite (stream, state->uf);
	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	status  = gdl_block_fwrite (stream, state->q);
	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	status = gdl_block_fwrite (stream, state->uq);
	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	status  = gdl_block_fwrite (stream, state->z);
	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	
	return GDL_SUCCESS;	
}


static const gdl_fuzzy_workspace_type _gview =
{
	"gdl_fuzzy_gview",
	sizeof (gdl_fuzzy_gview_t),
	&gdl_fuzzy_gview_alloc,
	&gdl_fuzzy_gview_free,
	&gdl_fuzzy_gview_init,
	&gdl_fuzzy_gview_iterate,
	&gdl_fuzzy_gview_loglikelihood,
	&gdl_fuzzy_gview_residual_abs,
	&gdl_fuzzy_gview_residual_sq,
	&gdl_fuzzy_gview_get_q,
	&gdl_fuzzy_gview_get_f,
	&gdl_fuzzy_gview_get_f_max,
	&gdl_fuzzy_gview_get_f_size,
	&gdl_fuzzy_gview_fread,
	&gdl_fuzzy_gview_fwrite
};

const gdl_fuzzy_workspace_type * gdl_fuzzy_gview = &_gview;
