/*  
 *  fview/rmix.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <stdio.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_fsheet.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_reader.h>

static const gdl_fsheet_type STANDARD =
{
	1,
	1,
	"",
	"\t",
	"\n",
	"\t",
	"\t"	
};

static gdl_fsheet_type *
_gdl_fview_reader_mixture_fsheet_type (const gdl_fview_reader_type * type)
{
	gdl_fsheet_type * ftype = gdl_fsheet_type_default ();
	
   ftype->rname  = type->has_acc_name;
	ftype->cname  = type->has_fac_header;
	gdl_string_free (ftype->fsep);
	ftype->fsep   = gdl_string_clone (type->sep);
	gdl_string_free (ftype->rnfsep);
	ftype->rnfsep = gdl_string_clone (type->sep);
	gdl_string_free (ftype->cnfsep);
	ftype->cnfsep = gdl_string_clone (type->sep);
	gdl_string_free (ftype->tsep);
	ftype->tsep   = gdl_string_clone (STANDARD.tsep);
	gdl_string_free (ftype->rsep);
	ftype->rsep   = gdl_string_clone (STANDARD.rsep);
	
	return ftype;
}

static gdl_fsheet *
_gdl_fview_reader_mixture_fsheet (gdl_fview_reader * reader)
{
	gdl_fsheet_type * ftype = _gdl_fview_reader_mixture_fsheet_type (reader->type);
	gdl_fsheet * fs;
	
	fs = gdl_fsheet_alloc (ftype);
	
	gdl_fsheet_init (fs, reader->stream);
	
	return fs;
}

static int
_gdl_fview_reader_mixture_init (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	size_t n, f;
	
	n = gdl_fsheet_nrow (fs);
	f = gdl_fsheet_ncolumn (fs);
	
	if (!n || !f)
	{
		return GDL_EINVAL;
	}
	
	reader->fview = gdl_fview_alloc (gdl_fview_standard);
	
	return GDL_SUCCESS;
}

static int
_gdl_fview_reader_mixture_check_factor (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status;
	size_t nc, i = 0, j, fidx = 0, lidx = 0;
	gdl_string * col, * factor, * level, * old;
	gdl_factor * vf;
	
	nc   = gdl_fsheet_ncolumn (fs);
	old  = 0;
	fidx = 0;
	lidx = 0;
	
	gdl_fsheet_itr * itr = gdl_fsheet_iterator_r (fs);
	
	do 
	{
		col = gdl_fsheet_iterator_column (itr);
		
		if (i == nc)
		{
			break;
		}
		
		if (gdl_fview_reader_split_mixture_column_name (col, &factor, &level)!=GDL_SUCCESS)
		{
			gdl_fview_reader_error (reader, "Not Valid Mixture Factor Name %s", col);
			gdl_fsheet_iterator_free (itr);
			gdl_string_free (level);
			gdl_string_free (factor);
			return (1);
		}
		
		if (old==0 || strcmp(factor, old))
		{
			if (lidx == 1)
			{
				gdl_fview_reader_error (reader, "Factor %s has only 1 mixture level !", factor);
				gdl_fsheet_iterator_free (itr);
				gdl_string_free (level);
				gdl_string_free (factor);
				return (1);
			}
			vf = gdl_fview_reader_add_factor (reader, gdl_factor_mixture, factor, fidx);
			fidx++;
			lidx=0;
			gdl_string_free (old);
			old = factor;
		}
		
		j = gdl_fview_reader_update_factor (vf, level);
		
		if (j != lidx)
		{
			gdl_fview_reader_error (reader, "Factor Level Name %s is not unique", level);
			gdl_fsheet_iterator_free (itr);
			return (1);
		}
		
		lidx++;
			
		gdl_string_free (level);
		
		i++;
	}
	while (gdl_fsheet_iterator_next (itr));
	
	gdl_fsheet_iterator_free (itr);
	
	return GDL_SUCCESS;	
}

static int
_gdl_fview_reader_mixture_check_accession (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status;
	char * cell, * row, * col;
	size_t ridx, cidx, first = 1;
	gdl_accession  * va;
	gdl_factor      * vl;
	gdl_fview_reader_cell  * pcell;
	
	gdl_fsheet_itr * itr = gdl_fsheet_iterator_c (fs);
	
	do 
	{
		row  = gdl_fsheet_iterator_row (itr);
		col  = gdl_fsheet_iterator_column (itr);
		cell = gdl_fsheet_iterator_cell (itr);
		ridx = gdl_fsheet_iterator_row_idx (itr);
		cidx = gdl_fsheet_iterator_column_idx (itr);
		
		va = gdl_fview_reader_add_accession (reader, row, ridx);
		
		if (first && va->idx != ridx)
		{
			gdl_fview_reader_error (reader, "Accession %s is not unique", row);
			gdl_fsheet_iterator_free (itr);
			return (1);
		}
		
	}
	while (gdl_fsheet_iterator_next (itr));
	
	gdl_fsheet_iterator_free (itr);
	
	return GDL_SUCCESS;	
}

static void
gdl_fview_reader_mixture_parse_cell (gdl_fview_reader * reader, gdl_fview_reader_cell * pcell, size_t lidx, gdl_string * cell)
{
	if (strcmp (cell, reader->type->missing))
	{
		pcell->point->values[lidx]->value = (double)atof(cell);
	}
	else
	{
		gdl_gvalues_free (pcell->point);
		pcell->point = NULL;
	}
}

static gdl_fview_reader_cell *
_gdl_fview_reader_mixture_new_pcell (const gdl_factor * f)
{
	if (f)
	{
		size_t i, nl;
		gdl_fview_reader_cell * p;
		
		p = gdl_fview_reader_cell_alloc ();
		
		nl = gdl_factor_size (f);
		
		p->point = gdl_gvalues_alloc (nl);
		
		for (i = 0; i < nl; i++)
		{
			p->point->values[i]        = gdl_gvalue_alloc ();
			p->point->values[i]->idx   = i;
			p->point->values[i]->value = 0.0;
		}
		
		return p;
	}
	
	return NULL;
}

static int
_gdl_fview_reader_mixture_perform (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status, lidx;
	char * cell, * row, * col;
	size_t ridx, cidx, fidx, first = 1;
	gdl_accession  * va;
	gdl_factor      * vl = NULL;
	gdl_fview_reader_cell  * pcell;
	
	fidx = 0;
	lidx = -1;
	
	vl    = gdl_fview_get_factor (reader->fview, 0);
	pcell = _gdl_fview_reader_mixture_new_pcell (vl);
	
	gdl_fsheet_itr * itr = gdl_fsheet_iterator_r (fs);
	
	do 
	{
		row  = gdl_fsheet_iterator_row (itr);
		col  = gdl_fsheet_iterator_column (itr);
		cell = gdl_fsheet_iterator_cell (itr);
		ridx = gdl_fsheet_iterator_row_idx (itr);
		cidx = gdl_fsheet_iterator_column_idx (itr);
		
		if (gdl_fsheet_iterator_is_new_column (itr))
		{
			lidx++;
			if (lidx == gdl_factor_size (vl))
			{
				fidx++;
				if (fidx == gdl_fview_factor_size (reader->fview))
				{
					fidx=0;
				}
				vl = gdl_fview_get_factor (reader->fview, fidx);
				gdl_fview_reader_cell_free (pcell);
				pcell = _gdl_fview_reader_mixture_new_pcell (vl);
				lidx  = 0;
			}
		}
		
		va = gdl_fview_get_accession (reader->fview, ridx);
		
		gdl_fview_reader_mixture_parse_cell (reader, pcell, lidx, cell);
		
		gdl_fview_reader_set_fdatapoint (reader, va, vl, pcell);
		
	}
	while (gdl_fsheet_iterator_next (itr));
	
	gdl_fsheet_iterator_free (itr);
	
	return GDL_SUCCESS;
}

static int
_gdl_fview_reader_mixture_parse (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status;
	
	if (_gdl_fview_reader_mixture_init (reader, fs) != GDL_SUCCESS)
	{
		return GDL_EINVAL;
	}
	
	status = _gdl_fview_reader_mixture_check_factor (reader, fs);
	
	if (status != GDL_SUCCESS)
	{
		gdl_fview_free (reader->fview);
		GDL_ERROR_VAL (reader->error, GDL_EINVAL, 0);
	}
	
	status = _gdl_fview_reader_mixture_check_accession (reader, fs);
	
	if (status != GDL_SUCCESS)
	{
		gdl_fview_free (reader->fview);
		GDL_ERROR_VAL (reader->error, GDL_EINVAL, 0);
	}
	
	status = _gdl_fview_reader_mixture_perform (reader, fs);
	
	if (status != GDL_SUCCESS)
	{
		gdl_fview_free (reader->fview);
		GDL_ERROR_VAL (reader->error, GDL_EINVAL, 0);
	}
	
	return GDL_SUCCESS;
}

static int
_gdl_fview_reader_mixture_read (gdl_fview_reader * reader)
{
	int status;
	
	gdl_fsheet * fs = _gdl_fview_reader_mixture_fsheet (reader);
	
	if (fs == NULL)
	{
		return GDL_EINVAL;
	}
		
	status = _gdl_fview_reader_mixture_parse (reader, fs);
	
	gdl_fsheet_free (fs);
	
	return status;
}

static gdl_fview_reader_type _gdl_fview_reader_mixture = 
{
	"?",
	"\t",
	1,
	1,
	&_gdl_fview_reader_mixture_read
};

const gdl_fview_reader_type * gdl_fview_reader_mixture = &_gdl_fview_reader_mixture;
