/*  
 *  fview/standard.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_factor.h>
#include <gdl/gdl_accession.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_check_range.h>

typedef struct
{
	size_t nf;
	size_t na;
	gdl_hashtable   * factor;
	gdl_hashtable   * accession;
	gdl_fdatapoints * fdata;
	gdl_factor       ** _factor;
	gdl_accession    ** _accession;
} gdl_fview_standard_t;

static int
_gdl_fview_standard_grow_factor (gdl_fview_standard_t * v, size_t n)
{
	if (v->_factor == NULL)
	{
		v->_factor = GDL_MALLOC (gdl_factor *, n);
	}
	else
	{
		gdl_factor ** new = GDL_MALLOC (gdl_factor *, n);
		memcpy (new, v->_factor, sizeof(gdl_factor *)*v->nf);
		GDL_FREE (v->_factor);
		v->_factor = new;
	}
	v->nf = n;
	return (0);
}

static int
_gdl_fview_standard_grow_accession (gdl_fview_standard_t * v, size_t n)
{
	if (v->_accession == NULL)
	{
		v->_accession = GDL_MALLOC (gdl_accession *, n);
	}
	else
	{
		gdl_accession ** new = GDL_MALLOC (gdl_accession *, n);
		memcpy (new, v->_accession, sizeof(gdl_accession *)*v->na);
		GDL_FREE (v->_accession);
		v->_accession = new;
	}
	v->na = n;
	return (0);
}

static int
_gdl_fview_standard_get (const gdl_fview_standard_t * v, size_t i, size_t j, gdl_gvalues_get * get)
{	
	gdl_fdatapoint * f = gdl_fdatapoints_get (v->fdata, i, j);
	
	if (f == NULL)
	{
		get->na = gdl_true;
	}
	else 
	{
		get->na      = gdl_false;
		get->x->size = f->size;
		gdl_gvalues_copy (get->x, f);
	}
	
	return GDL_SUCCESS;
}

int
gdl_fview_standard_alloc (void * vg)
{
	gdl_fview_standard_t * g = (gdl_fview_standard_t *) vg;
	
	g->na = g->nf = 0;
	g->fdata      = NULL;
	g->_factor     = NULL;
	g->_accession = NULL;
	
	g->factor     = gdl_hashtable_alloc (gdl_entity_interface, 0);
	g->accession  = gdl_hashtable_alloc (gdl_entity_interface, 0);

	return GDL_SUCCESS;
}

void
gdl_fview_standard_free (void * vg)
{
	if (vg)
	{
		gdl_fview_standard_t * g = (gdl_fview_standard_t *) vg;
		
		gdl_hashtable_free (g->factor);
		gdl_hashtable_free (g->accession);
		gdl_fdatapoints_free (g->fdata);
		GDL_FREE (g->_factor);
		GDL_FREE (g->_accession);
	}	
}

size_t
gdl_fview_standard_factor_size (const void * vg)
{
	return ((gdl_fview_standard_t *)vg)->nf;
}

size_t
gdl_fview_standard_accession_size (const void * vg)
{
	return ((gdl_fview_standard_t *)vg)->na;
}

gdl_factor *
gdl_fview_standard_add_factor (void * vg, const gdl_factor_type * T, const gdl_string * name)
{
	if (vg == 0 || name == 0)
	{
		return NULL;
	}
	
	gdl_fview_standard_t * v = (gdl_fview_standard_t *) vg;
	
	if (v->fdata)
	{
		GDL_ERROR_VAL ("Attempt to add a factor after setting the fview",
		               GDL_FAILURE,
		               0);
	}
	else
	{
		gdl_factor * vl = 
			(gdl_factor *) gdl_hashtable_lookup (v->factor, name);
		if (vl)
		{
			return vl;
		}
		else
		{
			vl      = gdl_factor_new (T, name);
			vl->idx = gdl_hashtable_size (v->factor);
		    gdl_hashtable_add (v->factor, name, vl, 1);
		    _gdl_fview_standard_grow_factor (v, vl->idx+1);
		    v->_factor[vl->idx] = vl;
		    return vl;
		}
	}
}

gdl_accession *
gdl_fview_standard_add_accession (void * vg, const gdl_string * name)
{
	if (vg == 0 || name == 0)
	{
		return NULL;
	}
	
	gdl_fview_standard_t * v = (gdl_fview_standard_t *) vg;
	
	if (v->fdata)
	{
		GDL_ERROR_VAL ("Attempt to add an accession after setting the fview",
		               GDL_FAILURE,
		               0);
	}
	else
	{
		
		gdl_accession * va = 
			(gdl_accession *) gdl_hashtable_lookup (v->accession, name);
		if (va)
		{
			return va;
		}
		else
		{
			va      = gdl_accession_new (name);
			va->idx = gdl_hashtable_size (v->accession);
		    gdl_hashtable_add (v->accession, name, va, 1);
		    _gdl_fview_standard_grow_accession (v, va->idx+1);
		    v->_accession[va->idx] = va;
		    return va;
		}
	}
}

gdl_factor *
gdl_fview_standard_get_factor (const void * vg, size_t idx)
{
	if (gdl_check_range)
	{
		if (idx >= ((gdl_fview_standard_t *)vg)->nf)
		{
		    GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
		}
	}
	return ((gdl_fview_standard_t *)vg)->_factor[idx];
}

gdl_accession *
gdl_fview_standard_get_accession (const void * vg, size_t idx)
{
	if (gdl_check_range)
	{
		if (idx >= ((gdl_fview_standard_t *)vg)->na)
		{
		    GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
		}
	}
	return ((gdl_fview_standard_t *)vg)->_accession[idx];
}

gdl_factor *
gdl_fview_standard_search_factor (const void * vg, const char * name)
{
	return gdl_hashtable_lookup (((gdl_fview_standard_t *)vg)->factor, name);
}

gdl_accession *
gdl_fview_standard_search_accession (const void * vg, const char * name)
{
	return gdl_hashtable_lookup (((gdl_fview_standard_t *)vg)->accession, name);
}

gdl_gvalues_get *
gdl_fview_standard_get_new (const void * gv)
{
	gdl_gvalues_get * get;
	size_t i, j, max=0;
	gdl_fview_standard_t * g = (gdl_fview_standard_t *) gv;
	
	for (i = 0; i < g->nf; i++)
	{
		gdl_factor * vl = gdl_fview_standard_get_factor (g, i);
		j               = gdl_factor_size (vl);
		if (j > max)
		{
			max = j;
		}
	}
	return gdl_gvalues_get_alloc (max);
}

int
gdl_fview_standard_set_fdatapoint (void * vg, const gdl_accession * va, const gdl_factor * vl, gdl_fdatapoint * gd)
{
	return gdl_fview_standard_set_fdatapoint_f (vg, va->idx, vl->idx, gd);
}

int
gdl_fview_standard_get (const void * gv, const gdl_accession * va, const gdl_factor * vl, gdl_gvalues_get * get)
{
	return
		_gdl_fview_standard_get ((gdl_fview_standard_t *) gv, va->idx, vl->idx, get);
}

gdl_boolean
gdl_fview_standard_is_missing (const void * gv, const gdl_accession * va, const gdl_factor * vl)
{
	return
		gdl_fdatapoints_is_missing (((gdl_fview_standard_t *)gv)->fdata, va->idx, vl->idx);	
}

int
gdl_fview_standard_set_fdatapoint_f (void * vg, size_t i, size_t j, gdl_fdatapoint * gd)
{
	gdl_fview_standard_t * f = (gdl_fview_standard_t *)vg;
	
	if (f->fdata == 0)
	{
		f->fdata = gdl_fdatapoints_alloc (f->na, f->nf, f->_factor);
	}
	
	return gdl_fdatapoints_set (f->fdata, i, j, gd);
}

int
gdl_fview_standard_get_f (const void * gv, size_t i, size_t j, gdl_gvalues_get * get)
{
	return _gdl_fview_standard_get ((gdl_fview_standard_t *)gv, i, j, get);
}

gdl_boolean
gdl_fview_standard_is_missing_f (const void * gv, size_t i, size_t j)
{
	return gdl_fdatapoints_is_missing (((gdl_fview_standard_t *)gv)->fdata, i, j);
}

void *
gdl_fview_standard_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		int status;
		size_t n, l;
		gdl_fview_standard_t * v;
		
		v = GDL_CALLOC (gdl_fview_standard_t, 1);
		
		gdl_fview_standard_alloc (v);
		
		status = gdl_hashtable_fread (stream, v->factor);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		status = gdl_hashtable_fread (stream, v->accession);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		v->fdata = gdl_fdatapoints_fread (stream);
		GDL_FREAD_STATUS (v->fdata != NULL, 1);
		
		v->nf = gdl_hashtable_size (v->factor);
		
		if (v->nf)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (v->factor);
			
			_gdl_fview_standard_grow_factor (v, v->nf);
			
			do
			{
				gdl_factor * l = (gdl_factor *)
					gdl_hashtable_iterator_value (itr);
				v->_factor[l->idx] = l;
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		v->na = gdl_hashtable_size (v->accession);
		
		if (v->na)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (v->accession);
			
			_gdl_fview_standard_grow_accession (v, v->na);
			
			do
			{
				gdl_accession * a = (gdl_accession *)
					gdl_hashtable_iterator_value (itr);
				v->_accession[a->idx] = a; 
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
				
		return v;
	}	
}

int
gdl_fview_standard_fwrite (FILE * stream, void * gv)
{
	if (stream == 0 || gv == 0)
		return GDL_EINVAL;
	else
	{
		int status;
		gdl_fview_standard_t * v = (gdl_fview_standard_t *) gv;
		
		status = gdl_hashtable_fwrite (stream, v->factor);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_hashtable_fwrite (stream, v->accession);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_fdatapoints_fwrite (stream, v->fdata);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}	
}

static const gdl_fview_type _standard = 
{
	"gdl_fview_standard",
	sizeof (gdl_fview_standard_t),
	&gdl_fview_standard_alloc,
	&gdl_fview_standard_free,
	&gdl_fview_standard_add_factor,
	&gdl_fview_standard_add_accession,
	&gdl_fview_standard_factor_size,
	&gdl_fview_standard_accession_size,
	&gdl_fview_standard_get_factor,
	&gdl_fview_standard_get_accession,
	&gdl_fview_standard_search_factor,
	&gdl_fview_standard_search_accession,
	&gdl_fview_standard_get_new,
	&gdl_fview_standard_set_fdatapoint,
	&gdl_fview_standard_get,
	&gdl_fview_standard_is_missing,
	&gdl_fview_standard_set_fdatapoint_f,
	&gdl_fview_standard_get_f,
	&gdl_fview_standard_is_missing_f,
	&gdl_fview_standard_fread,
	&gdl_fview_standard_fwrite,
	NULL
};

const gdl_fview_type * gdl_fview_standard = &_standard;
