/*  
 *  fview/wrapper.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_mask.h>
#include <gdl/gdl_fview_wrapper.h>

struct _gdl_fview_wrapper_type
{
	gdl_string * name;
};

struct _gdl_fview_wrapper
{
	const gdl_fview_wrapper_type * type;
	const gdl_fview     * fview;
	const gdl_mask      * mask;
	//gdl_fview_collector * gmiss;
	gdl_accession        ** accession;
	gdl_factor           ** factor;
	gdl_factor           ** template;
};

static void
_alloc_template (gdl_fview_wrapper * gw, size_t j)
{
	size_t i, k, na, * tmp;
	gdl_factor      * factor;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	gdl_boolean has;
	
	gw->template[j] = gdl_entity_clone (GDL_FVIEW_GET_FACTOR (gw->fview, gw->mask, j));
	
	if (gdl_factor_get_type (gw->template[j]) == gdl_factor_categorial)
	{
		gbuf = gdl_gvalues_get_alloc (gdl_factor_size (gw->template[j]));
		na   = GDL_FVIEW_ACCESSION_SIZE (gw->fview, gw->mask);
		for (i = 0; i < na; i++)
		{
			GDL_FVIEW_GET_VALUE (gw->fview, gw->mask, i, j, gbuf);
			gv = gdl_gvalues_get_gvalues (gbuf);
			if (gv)
			{
				for (k = 0; k < gv->size; k++)
				{
					gdl_factor_level * level = gdl_factor_get (gw->template[j], gv->values[k]->idx);
					tmp = (size_t *)level->extra;
					if (!tmp)
					{
						tmp = GDL_CALLOC (size_t, 1);
						level->extra = tmp;
					}
					if (gv->values[k]->value == 1.0)
					{
						(*tmp)++;	
					}
				}
			}
			else
			{
				GDL_ERROR_VOID ("Unexpected null pointer", GDL_EFAULT);
			}
		}
		gdl_gvalues_get_free (gbuf);
	}
}

static void
_alloc_factor (gdl_fview_wrapper * gw, size_t j)
{
	size_t i, k, na, ng, * tmp;
	gdl_factor    * template;
	gdl_factor    * factor;
	gdl_factor_level * g1, * g2;
	
	gw->factor[j] = gdl_factor_new (gdl_factor_get_type (gw->template[j]), gdl_entity_get_name (gw->template[j]));
	
	gdl_entity_set_idx (gw->factor[j], j);
	
	if (gdl_factor_get_type (gw->factor[j]) == gdl_factor_categorial)
	{
		template = gw->template[j];
		factor   = gw->factor[j];
		
		ng = gdl_factor_size (template);
		for (i = 0; i < ng; i++)
		{
			g1   = gdl_factor_get (template, i);
			tmp  = (size_t *) g1->extra;
			if (tmp)
			{	
				if ((gw->type == gdl_fview_wrapper_local && *tmp)
				    || gw->type == gdl_fview_wrapper_global)
				{
					//printf (" KEEP\n");
					g2 = gdl_entity_clone (g1);
					gdl_factor_add (factor, &g2, 1);
					*tmp = g2->idx;
				}
			}
		}
	}
	else if (gdl_factor_get_type (gw->factor[j]) == gdl_factor_mixture)
	{
		template = gw->template[j];
		factor   = gw->factor[j];
		ng = gdl_factor_size (template);
		for (i = 0; i < ng; i++)
		{
			g1 = gdl_factor_get (template, i);
			g2 = gdl_entity_clone (g1);
			gdl_factor_add (factor, &g2, 1);
		}
	}
}

static void
_gdl_fview_wrapper_init_factor (gdl_fview_wrapper * gw)
{
	size_t j, nl;
	
	nl = GDL_FVIEW_FACTOR_SIZE (gw->fview, gw->mask);
	
	gw->template  = GDL_MALLOC (gdl_factor *, nl);
	gw->factor    = GDL_MALLOC (gdl_factor *, nl);
	
	for (j = 0; j < nl; j++)
	{
		_alloc_template (gw, j);
		_alloc_factor (gw, j);
	}
}

static void
_gdl_fview_wrapper_init_accession (gdl_fview_wrapper * gw)
{
	size_t j, na;
	
	na = GDL_FVIEW_ACCESSION_SIZE (gw->fview, gw->mask);
	
	gw->accession = GDL_MALLOC (gdl_accession *, na);
	
	for (j = 0; j < na; j++)
	{
		gw->accession[j] = gdl_entity_clone (GDL_FVIEW_GET_ACCESSION (gw->fview, gw->mask, j));
		gdl_entity_set_idx (gw->accession[j], j);
	}
}

gdl_fview_wrapper *
gdl_fview_wrapper_alloc (const gdl_fview_wrapper_type * T, 
                         const gdl_fview * fview,
                         const gdl_mask * mask)
{
	gdl_fview_wrapper * gw;
	
	gw = GDL_MALLOC (gdl_fview_wrapper, 1);
	
	gw->type   = T;
	gw->fview  = fview;
	gw->mask   = mask;
	//gw->gmiss  = gdl_fview_collector_alloc (gdl_fview_collector_missing);
	//gdl_fview_collector_perform (gw->gmiss, fview, mask, gw->gclust);
	
	_gdl_fview_wrapper_init_factor (gw);
	
	_gdl_fview_wrapper_init_accession (gw);
	
	return gw;
}

void
gdl_fview_wrapper_free (gdl_fview_wrapper * gw)
{
	if (gw)
	{
		size_t i, nl = gdl_fview_wrapper_factor_size (gw);
		
		//gdl_fview_collector_free (gw->gmiss);
		
		for (i = 0; i < nl; i++)
		{
			gdl_factor_free (gw->factor[i]);
			gdl_factor_free (gw->template[i]);
		}
		GDL_FREE (gw->template);
		GDL_FREE (gw->factor);
		
		nl = gdl_fview_wrapper_accession_size (gw);
		for (i = 0; i < nl; i++)
		{
			gdl_factor_free (gw->accession[i]);
		}
		GDL_FREE (gw->accession);
		
		GDL_FREE (gw);
	}	
}

const
gdl_fview * gdl_fview_wrapper_fview (const gdl_fview_wrapper * gw)
{
	return gw->fview;
}

const
gdl_mask * gdl_fview_wrapper_mask (const gdl_fview_wrapper * gw)
{
	return gw->mask;
}

//const gdl_fview_collector *
//gdl_fview_wrapper_collector (const gdl_fview_wrapper * gw)
//{
//	return gw->gmiss;
//}

size_t
gdl_fview_wrapper_accession_size (const gdl_fview_wrapper * gw)
{
	return GDL_FVIEW_ACCESSION_SIZE (gw->fview, gw->mask);
}

size_t
gdl_fview_wrapper_factor_size (const gdl_fview_wrapper * gw)
{
	return GDL_FVIEW_FACTOR_SIZE (gw->fview, gw->mask);
}

size_t
gdl_fview_wrapper_factor_level_size (const gdl_fview_wrapper * gw, size_t i)
{
	return gdl_factor_size (gw->factor[i]);
}

gdl_accession *
gdl_fview_wrapper_get_accession (const gdl_fview_wrapper * gw, size_t i)
{
	return gw->accession[i];
}

gdl_factor *
gdl_fview_wrapper_get_factor (const gdl_fview_wrapper * gw, size_t i)
{
	return gw->factor[i];
}

int
gdl_fview_wrapper_get_value (const gdl_fview_wrapper * gw, size_t i, size_t j, gdl_gvalues_get * gbuf)
{
	gdl_gvalues * gv = NULL;
	gdl_factor   * temp = gw->template[j];
	
	if (!GDL_FVIEW_IS_MISSING (gw->fview, gw->mask, i, j))
	{
		GDL_FVIEW_GET_VALUE (gw->fview, gw->mask, i, j, gbuf);
		
		gv = gbuf->x;
		
		if (gdl_factor_get_type (gw->factor[j]) == gdl_factor_categorial)
		{
			for (i = 0; i < gv->size; i++)
			{
				gdl_factor_level * level = gdl_factor_get (temp, gv->values[i]->idx);
				if (level->extra)
				{
					gv->values[i]->idx = *((size_t*)level->extra);
				}
				else
				{
					(gv->size)--;
					if (!gv->size)
					{
						gbuf->na = gdl_true;
						break;
					}
					i--;
				}
			}
		}
	}
	else
	{
		gbuf->na = gdl_true;
//		gv = (gdl_gvalues * ) gdl_fview_collector_get_idx (gw->gmiss, ic, j);
//		if (gv)
//		{
//			gbuf->na      = gdl_false;
//			gbuf->x->size = gv->size;
//			gdl_gvalues_copy (gbuf->x, gv);
//		}
	}
	
	return GDL_SUCCESS;
}

gdl_boolean
gdl_fview_wrapper_is_missing (const gdl_fview_wrapper * gw, size_t i, size_t j)
{
	if (GDL_FVIEW_IS_MISSING (gw->fview, gw->mask, i, j))
	{
//		gdl_gvalues * gv = (gdl_gvalues * ) gdl_fview_collector_get_idx (gw->gmiss, i, j);
//		if (gv)
//		{
//			return gdl_false;
//		}
//		else
//		{
//			return gdl_true;
//		}
		return gdl_true;
	}
	else
	{
		return gdl_false;
	}
}

gdl_gvalues_get *
gdl_fview_wrapper_get_new (const gdl_fview_wrapper * gw)
{
	gdl_gvalues_get * get;
	size_t i, size, nl;
	
	nl = GDL_FVIEW_FACTOR_SIZE (gw->fview, gw->mask);
	
	for (size = i = 0; i < nl; i++)
	{
		gdl_factor * vl = gw->factor[i];
		if (gdl_factor_size (vl) > size)
		{
			size = gdl_factor_size (vl);	
		}
	}
	
	return gdl_gvalues_get_alloc (size);
}

static gdl_mask *
gdl_fview_wrapper_mask_not_informative_factor (const gdl_fview_wrapper * fw, const gdl_mask * mask)
{
	size_t i, nl, ng;
	gdl_factor * factor;
	const gdl_factor_type * T;
	gdl_mask * u = NULL;
	
	nl = gdl_mask_size (mask, GDL_FACTOR);
	
	if (!nl)
	{
		nl = gdl_fview_wrapper_factor_size (fw);
	}	
	
	for (i = 0; i < nl; i++)
	{
		factor = gdl_fview_wrapper_get_factor (fw, gdl_mask_get_idx (mask, GDL_FACTOR, i));
		T = gdl_factor_get_type (factor);
		
		ng = gdl_factor_size (factor);
		
		if (T == gdl_factor_continuous || ng >= 2)
		{
			// keep it
			if (u == 0)
			{
				u = gdl_mask_alloc ();
				if (mask != 0)
				{
					gdl_mask_set (u, GDL_ACCESSION, gdl_mask_get_clone (mask, GDL_ACCESSION), 1);
				}
			}
			gdl_mask_add (u, factor);
		}
	}
	
	return u;	
}

static gdl_mask *
gdl_fview_wrapper_mask_not_informative_accession (const gdl_fview_wrapper * fw, const gdl_mask * mask)
{
	size_t i, j;
	size_t na, nl, ni;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues * x;
	gdl_mask * u = NULL;
	
	na = gdl_mask_size (mask, GDL_ACCESSION);
	if (!na)
	{
		na = gdl_fview_wrapper_accession_size (fw);
	}
	nl = gdl_mask_size (mask, GDL_FACTOR);
	if (!nl)
	{
		na = gdl_fview_wrapper_factor_size (fw);
	}
	
	gbuf = gdl_fview_wrapper_get_new (fw);
	
	for (i = 0; i < na; i++)
	{
		for (ni = j = 0; j < nl; j++)
		{
			gdl_fview_wrapper_get_value (fw, gdl_mask_get_idx (mask, GDL_ACCESSION, i), gdl_mask_get_idx (mask, GDL_FACTOR, j), gbuf);
			
			x = gdl_gvalues_get_gvalues (gbuf);
			
			if (x==0)
			{
				ni++;	
			}
		}
		if (ni < nl)
		{
			// keep it
			gdl_accession * a = gdl_fview_wrapper_get_accession (fw, gdl_mask_get_idx (mask, GDL_ACCESSION, i));
			if (u == 0)
			{
				u = gdl_mask_alloc ();
				if (mask)
				{
					gdl_mask_set (u, GDL_FACTOR, gdl_mask_get_clone (mask, GDL_FACTOR), 1);
				}
			}
			gdl_mask_add (u, a);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return u;
}

gdl_mask *
gdl_fview_wrapper_mask_not_informative (const gdl_fview_wrapper * fw, const gdl_mask * mask, const gdl_entity_type * T)
{
	if (T == GDL_ACCESSION)
	{
		return gdl_fview_wrapper_mask_not_informative_accession (fw, mask);
	}
	else if (T == GDL_FACTOR)
	{
		return gdl_fview_wrapper_mask_not_informative_factor (fw, mask);
	}
	return NULL;
}

//size_t
//gdl_fview_wrapper_missing_size (const gdl_fview_wrapper * gw)
//{
//	return gdl_fview_collector_size (gw->gmiss);
//}
//
//size_t
//gdl_fview_wrapper_missing_accession_size (const gdl_fview_wrapper * gw)
//{
//	return gdl_fview_collector_accession_size (gw->gmiss);	
//}
//
//size_t
//gdl_fview_wrapper_missing_gfactor_size (const gdl_fview_wrapper * gw, size_t r)
//{
//	return gdl_fview_collector_gfactor_size (gw->gmiss, r);	
//}
//
//size_t
//gdl_fview_wrapper_missing_hfactor_size (const gdl_fview_wrapper * gw, size_t r, size_t h)
//{
//	return gdl_fview_collector_hfactor_size (gw->gmiss, r, h);
//}
//
//size_t
//gdl_fview_wrapper_missing_accession_idx_c (const gdl_fview_wrapper * gw, size_t r)
//{
//	return gdl_fview_collector_accession_idx (gw->gmiss, r);	
//}
//
//const size_t *
//gdl_fview_wrapper_missing_gfactor_idx (const gdl_fview_wrapper * gw, size_t r)
//{
//	return gdl_fview_collector_gfactor_idx (gw->gmiss, r);	
//}
//
//const size_t *
//gdl_fview_wrapper_missing_hfactor_idx (const gdl_fview_wrapper * gw, size_t r, size_t h)
//{
//	return gdl_fview_collector_hfactor_idx (gw->gmiss, r, h);
//}
//
//static gdl_gvalues *
//_init_missing_genotype (const gdl_fview_wrapper * gw, size_t ic, size_t i, size_t l)
//{
//	gdl_gvalues_get * gbuf = gdl_fview_wrapper_get_new (gw);
//	gdl_gvalues * x;
//	const gdl_gvalues * y;
//	
//	GDL_FVIEW_GET_GENOTYPE_TEMPLATE (gw->fview, gw->mask, i, gw->factor[l], gbuf);
//	
//	y = gdl_gvalues_get_gvalues (gbuf);
//	
//	x = gdl_gvalues_clone (y);
//	
//	gdl_fview_collector_gset_idx (gw->gmiss, ic, l, x);
//	
//	gdl_gvalues_get_free (gbuf);
//	
//	return x;
//}
//
//static gdl_gvalues *
//_init_missing_allele (const gdl_fview_wrapper * gw, size_t ic, size_t i, size_t l, size_t p)
//{
//	size_t j;
//	gdl_hashtable     * hash;
//	gdl_hashtable_itr * itr;
//	gdl_gvalues_get     * gbuf = gdl_fview_wrapper_get_new (gw);
//	gdl_gvalues * x;
//	gdl_gvalue  * ax;
//	const gdl_gvalues * y;
//	
//	GDL_FVIEW_GET_GENOTYPE_TEMPLATE (gw->fview, gw->mask, i, gw->factor[l], gbuf);
//	
//	y = gdl_gvalues_get_gvalues (gbuf);
//	
//	x = _get_allele_gvalues_from_genotype (gw, l, p, y);
//	
//	gdl_fview_collector_hset_idx (gw->gmiss, ic, l, p, x);
//	
//	gdl_gvalues_get_free (gbuf);
//	
//	return x;
//}
//
//gdl_gvalues *
//gdl_fview_wrapper_missing_gget (const gdl_fview_wrapper * gw, size_t r, size_t l)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_gget (gw->gmiss, r, l);
//	
//	if (!x)
//	{
//		const size_t * j = gdl_fview_collector_gfactor_idx (gw->gmiss, r);
//		size_t ic = gdl_fview_collector_accession_idx (gw->gmiss, r);
//		size_t i  = gdl_clustering_clust_idx (gw->gclust, ic);
//		
//		x = _init_missing_genotype (gw, ic, i, j[l]);
//	}
//	
//	return x;
//}
//
//gdl_gvalues *
//gdl_fview_wrapper_missing_hget (const gdl_fview_wrapper * gw, size_t r, size_t h, size_t l)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_hget (gw->gmiss, r, h, l);
//	
//	if (!x)
//	{
//		const size_t * j = gdl_fview_collector_hfactor_idx (gw->gmiss, r, h);
//		size_t ic = gdl_fview_collector_accession_idx (gw->gmiss, r);
//		size_t i  = gdl_clustering_clust_idx (gw->gclust, ic);
//		
//		x = _init_missing_allele (gw, ic, i, j[l], h);
//	}
//	
//	return x;
//}
//
//gdl_gvalues *
//gdl_fview_wrapper_missing_gget_c (const gdl_fview_wrapper * gw, size_t ridx, size_t lidx)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_gget_idx (gw->gmiss, ridx, lidx);
//	
//	if (!x)
//	{
//		size_t i = gdl_clustering_clust_idx (gw->gclust, ridx);
//		
//		x = _init_missing_genotype (gw, ridx, i, lidx);
//	}
//	
//	return x;
//}
//
//gdl_gvalues *
//gdl_fview_wrapper_missing_hget_c (const gdl_fview_wrapper * gw, size_t ridx, size_t lidx, size_t p)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_hget_idx (gw->gmiss, ridx, lidx, p);
//	
//	if (!x)
//	{
//		size_t i = gdl_clustering_clust_idx (gw->gclust, ridx);
//		
//		x = _init_missing_allele (gw, ridx, i, lidx, p);
//	}
//	
//	return x;
//}
//
//void
//gdl_fview_wrapper_missing_greset (const gdl_fview_wrapper * gw, size_t r, size_t l)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_gget (gw->gmiss, r, l);
//	if (x)
//	{
//		gdl_gvalues_free (x);
//	}
//	gdl_fview_collector_gset (gw->gmiss, r, l, NULL);
//}
//
//void
//gdl_fview_wrapper_missing_hreset (const gdl_fview_wrapper * gw, size_t r, size_t h, size_t l)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_hget (gw->gmiss, r, h, l);
//	if (x)
//	{
//		gdl_gvalues_free (x);
//	}
//	gdl_fview_collector_hset (gw->gmiss, r, h, l, NULL);
//}
//
//void
//gdl_fview_wrapper_missing_greset_c (const gdl_fview_wrapper * gw, size_t ridx, size_t lidx)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_gget_idx (gw->gmiss, ridx, lidx);
//	if (x)
//	{
//		gdl_gvalues_free (x);
//	}
//	gdl_fview_collector_gset_idx (gw->gmiss, ridx, lidx, NULL);
//}
//
//void
//gdl_fview_wrapper_missing_hreset_c (const gdl_fview_wrapper * gw, size_t ridx, size_t lidx, size_t p)
//{
//	gdl_gvalues * x = (gdl_gvalues *) gdl_fview_collector_hget_idx (gw->gmiss, ridx, lidx, p);
//	if (x)
//	{
//		gdl_gvalues_free (x);
//	}
//	gdl_fview_collector_hset_idx (gw->gmiss, ridx, lidx, p, NULL);
//}
//
//gdl_mask *
//gdl_fview_wrapper_mask_c (const gdl_fview_wrapper * gw)
//{
//	gdl_mask * cmask = NULL;
//	
//	if (gw)
//	{
//		size_t ic, nc;
//		
//		cmask = gdl_mask_alloc ();
//		
//		gdl_mask_set (cmask, GDL_FACTOR, gdl_mask_get_clone (gw->mask, GDL_FACTOR), 1);
//		
//		nc   = gdl_clustering_nclust (gw->gclust);
//		
//		for (ic = 0; ic < nc; ic++)
//		{
//			gdl_accession * a = gdl_fview_wrapper_get_accession_c (gw, ic);
//			gdl_mask_add (cmask, a);
//		}
//	}
//	
//	return cmask;
//}

static const gdl_fview_wrapper_type _local =
{
	"gdl_fview_wrapper_local"
};

static const gdl_fview_wrapper_type _global =
{
	"gdl_fview_wrapper_global"
};

const gdl_fview_wrapper_type * gdl_fview_wrapper_local = &_local;
const gdl_fview_wrapper_type * gdl_fview_wrapper_global = &_global;
