/*  
 * 	gblock/block.c 
 *  
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:05 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

static size_t FUNCTION (gdl_block, get_size)  (const size_t p, const size_t *dim);
static size_t FUNCTION (gdl_block, get_index) (const TYPE (gdl_block) *b, ATOMIC * x, va_list ap);

TYPE (gdl_block) *
FUNCTION (gdl_block, alloc) (const size_t p, const size_t *dim)
{
	size_t i,j;
	TYPE (gdl_block) * b;
	
	if (p == 0)
    {
      GDL_ERROR_VAL ("block variables 'p' must be positive integer",
                      GDL_EINVAL, 0);
    }
    if (dim == 0) 
    {
    	GDL_ERROR_VAL ("block dimensions 'dim' must be a not NULL size_t array",
                      GDL_EINVAL, 0);
    }
    
	b = GDL_MALLOC (TYPE (gdl_block), 1);
	
    if (b == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for block struct",
                        GDL_ENOMEM, 0);
    }
    
    b->p    = p;
	
	b->dim  = GDL_MALLOC(size_t, p);
	
	if (b->dim == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for block dimensions",
                        GDL_ENOMEM, 0);
    }
    
    b->_ssize = GDL_MALLOC(size_t, p);
    
    if (b->_ssize == 0)
    {
      GDL_ERROR_VAL ("failed to allocate extra private space for block structure",
                        GDL_ENOMEM, 0);
    }
    
    for(i=0; i<p; i++)
    {
    	b->dim[i]    = dim[i];
    	b->_ssize[i] = 1;
    	for(j=1;j<=p-i-1;j++)
    	{
    		size_t k = p-j;
    		b->_ssize[i] *= dim[k];
    	}
    }
	
	b->size = FUNCTION (gdl_block, get_size) (p, dim);
	
	b->data = GDL_CALLOC (ATOMIC, b->size);
	
	if (b->data == 0)
	{
	    free (b);         /* exception in constructor, avoid memory leak */
	
	    GDL_ERROR_VAL ("failed to allocate space for block data",
	                        GDL_ENOMEM, 0);
	}

	return b;	
}
/**
 * 
 * 
 */
TYPE (gdl_block) *
FUNCTION (gdl_block, alloc2) (const size_t p, ...)
{
  va_list ap;
  size_t i,*dim;
  TYPE (gdl_block) * b;  

  dim = GDL_MALLOC(size_t, p);

  va_start (ap, p);                          /* Initialize the argument list. */

  for (i = 0; i < p; i++)
  {
  	size_t j = va_arg (ap, size_t);
  	dim[i] = j;
  	
  }
  
  va_end (ap);
  
  b = FUNCTION (gdl_block, alloc) (p, dim);
    
  GDL_FREE(dim);
  
  return b;
}

void
FUNCTION (gdl_block, free) (TYPE (gdl_block) * b)
{
  if (b)
  {
	  GDL_FREE (b->dim);
	  GDL_FREE (b->_ssize);
	  GDL_FREE (b->data);
	  GDL_FREE (b);
  }
}

BASE
FUNCTION (gdl_block, get) (const TYPE (gdl_block) * b, ...)
{
  va_list ap;
  size_t i, idx=0;

  va_start (ap, b);

  idx = FUNCTION (gdl_block, get_index) (b, NULL, ap);
  
  va_end (ap);
  
  return b->data[idx];
}

void
FUNCTION (gdl_block, set) (TYPE (gdl_block) * b, ...)
{
  va_list ap;
  size_t i, idx=0;
  BASE x;

  va_start (ap, b);

  idx = FUNCTION (gdl_block, get_index) (b, &x, ap);
  
  va_end (ap);
  
  b->data[idx] = x;
}

static size_t
FUNCTION (gdl_block, get_size) (const size_t p, const size_t * dim)
{
	size_t i,n=1;
	
	for (i = 0; i < p; i++) 
		n *= dim[i];
		
	return n;
}

static size_t
FUNCTION (gdl_block, get_index) (const TYPE (gdl_block) *b, ATOMIC *x, va_list ap)
{
	size_t i, j, idx=0;
	
	for (i = 0; i < b->p; i++)
	{
		j = va_arg (ap, size_t);
		idx += j*b->_ssize[i]; /* Get the next argument value. */  	
	}
	
	if (x != 0)
	{
		*x = va_arg (ap, VARG);
	}
	
	return idx;
}

TYPE (gdl_block) *
FUNCTION (gdl_block, fread) (FILE * stream)
{
  TYPE (gdl_block) *b;
  size_t n,p,*dim,*_ssize,status;
  
  b = GDL_MALLOC(TYPE (gdl_block), 1);
  
  status = fread (&b->size, sizeof (size_t), 1, stream);
  GDL_FREAD_STATUS(status, 1);
  status = fread (&b->p, sizeof (size_t), 1, stream);
  GDL_FREAD_STATUS(status, 1);
  b->dim = GDL_MALLOC(size_t, b->p);
  status = fread (b->dim, sizeof (size_t), b->p, stream);
  GDL_FREAD_STATUS(status, b->p);
  b->_ssize = GDL_MALLOC(size_t, b->p);
  status = fread (b->_ssize, sizeof (size_t), b->p, stream);
  GDL_FREAD_STATUS(status, b->p);
  b->data = GDL_MALLOC(BASE, b->size);
  status = fread (b->data, sizeof (ATOMIC), b->size, stream);
  GDL_FREAD_STATUS(status, b->size);
  
  return b;
}

int
FUNCTION(gdl_block, fwrite) (FILE * stream, const TYPE (gdl_block) * b)
{
  size_t status; 	
	
  status = fwrite (&b->size, sizeof (size_t), 1, stream);
  GDL_FWRITE_STATUS(status, 1);
  status = fwrite (&b->p, sizeof (size_t), 1, stream);
  GDL_FWRITE_STATUS(status, 1);
  status = fwrite (b->dim, sizeof (size_t), b->p, stream);
  GDL_FWRITE_STATUS(status, b->p);
  status = fwrite (b->_ssize, sizeof (size_t), b->p, stream);
  GDL_FWRITE_STATUS(status, b->p);
  status = fwrite (b->data, sizeof (ATOMIC), b->size, stream);
  GDL_FWRITE_STATUS(status, b->size);
  
  return GDL_SUCCESS;  
}
