/*  
 * 	gblock/locus.c 
 *  
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_locus_block.h>

gdl_locus_block *
gdl_locus_block_alloc  (const size_t n, const size_t l, const size_t p)
{
	size_t i;
	gdl_locus_block * f;
	
	f = GDL_MALLOC (gdl_locus_block, 1);
	f->n   = n;
	f->l   = l;
	f->p   = p; 
	f->data = GDL_CALLOC (double, f->p*f->n*f->l);
	
	return f;
}

void
gdl_locus_block_free (gdl_locus_block * f)
{
	if (f)
	{
		GDL_FREE (f->data);
		GDL_FREE (f);
	}	
}

double
gdl_locus_block_get (const gdl_locus_block * f, size_t a, size_t l, size_t p)
{
	return f->data[a*(f->l*f->p)+p*(f->l)+l];
}

void
gdl_locus_block_set (const gdl_locus_block * f, size_t a, size_t l, size_t p, double x)
{
	f->data[a*(f->l*f->p)+p*(f->l)+l]=x;	
}

gdl_locus_block *
gdl_locus_block_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_locus_block * b;
		
		b = GDL_MALLOC (gdl_locus_block, 1);
		
		status=fread (&b->n, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status=fread (&b->l, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status=fread (&b->p, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		b->data = GDL_MALLOC (double, b->n*b->l*b->p);
		
		status=fread (b->data, sizeof(double), b->l*b->n*b->p, stream);
		GDL_FREAD_STATUS (status, b->l*b->n*b->p);
		
		return b;
	}
	else
	{
		return 0;	
	}
}

int
gdl_locus_block_fwrite (FILE * stream, const gdl_locus_block * b)
{
	if (stream && b)
	{
		int status;
		
		status=fwrite (&b->n, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status=fwrite (&b->l, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status=fwrite (&b->p, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status=fwrite (b->data, sizeof(double), b->l*b->n*b->p, stream);
		GDL_FWRITE_STATUS (status, b->l*b->n*b->p);
		
		return GDL_SUCCESS;
	}
	else
	{
		return GDL_EINVAL;	
	}	
}


gdl_loci_block *
gdl_loci_block_alloc  (const size_t k, const size_t n, const size_t l, const size_t p)
{
	size_t i;
	gdl_loci_block * b;
	
	b = GDL_MALLOC (gdl_loci_block, 1);
	
	b->k = k;
	b->blocks = GDL_MALLOC (gdl_locus_block *, k);
	for (i = 0; i < k; i++)
	{
		b->blocks[i] = gdl_locus_block_alloc (n, l, p);
	}
	
	return b;
}

void
gdl_loci_block_free (gdl_loci_block * b)
{
	if (b)
	{
		size_t i;
		for (i = 0; i < b->k; i++)
		{
			gdl_locus_block_free (b->blocks[i]);
		}
		GDL_FREE (b->blocks);
		GDL_FREE (b);
	}	
}

double
gdl_loci_block_get (const gdl_loci_block * b, size_t k, size_t a, size_t l, size_t j)
{
	return gdl_locus_block_get (b->blocks[k], a, l, j);	
}

void
gdl_loci_block_set (const gdl_loci_block * b, size_t k, size_t a, size_t l, size_t j, double x)
{
	gdl_locus_block_set (b->blocks[k], a, l, j, x);
}

gdl_loci_block *
gdl_loci_block_fread (FILE * stream)
{
	if (stream)
	{
		size_t i;
		int status;
		gdl_loci_block * b;
	
		b = GDL_MALLOC (gdl_loci_block, 1);
	
		status = fread (&(b->k), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		b->blocks = GDL_MALLOC (gdl_locus_block *, b->k);
		for (i = 0; i < b->k; i++)
		{
			b->blocks[i] = gdl_locus_block_fread (stream);
			GDL_FREAD_STATUS (b->blocks[i]!=0, 1);
		}
		
		return b;
	}
	else
	{
		return 0;	
	}	
}

int
gdl_loci_block_fwrite (FILE * stream, const gdl_loci_block * b)
{
	if (stream && b)
	{
		size_t i;
		int status;
		
		status = fwrite (&(b->k), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < b->k; i++)
		{
			status = gdl_locus_block_fwrite (stream, b->blocks[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		return GDL_SUCCESS;	
	}
	else
	{
		return GDL_EINVAL;	
	}	
}

