/*  
 * 	gblock/nblock_source.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */


static size_t FUNCTION (gdl_nblock, get_index) (const TYPE(gdl_nblock) * b, ATOMIC *x, va_list ap);

TYPE (gdl_nblock) *
FUNCTION (gdl_nblock, alloc) (const size_t p, const size_t *dim, size_t **nested)
{
	size_t i,j,k;
	TYPE (gdl_nblock) *b;
	
	if (p == 0)
    {
      GDL_ERROR_VAL ("block variables 'p' must be positive integer",
                      GDL_EINVAL, 0);
    }
    if (dim == 0) 
    {
    	GDL_ERROR_VAL ("block dimensions 'dim' must be a not NULL size_t array",
                      GDL_EINVAL, 0);
    }
    if (nested == 0) 
    {
    	GDL_ERROR_VAL ("block nested 'nested' must be a not NULL size_t array",
                      GDL_EINVAL, 0);
    }
    
	b = GDL_MALLOC( TYPE(gdl_nblock) , 1);
	
    b->p    = p;
    
    b->size = 1;
	
	b->dim     = GDL_MALLOC(size_t, p);
	
	b->nested  = GDL_MALLOC(size_t *, p);
	
	b->_nssize = GDL_MALLOC(size_t *, p);
    
    b->_sizes  = GDL_MALLOC(size_t, p);
    
    for(i = 0 ; i < p; i++)
    {
    	b->dim[i] = dim[i];
    	
    	b->nested[i]  = GDL_CALLOC(size_t, dim[i]);
    	
    	b->_nssize[i] = GDL_CALLOC(size_t, dim[i]);
    	
    	b->_sizes[i]  = 0;
    
    	for(j = 0; j < dim[i]; j++) 
	    {
	    	
	    	b->nested[i][j] = nested[i][j];
	    	b->_sizes[i]   += nested[i][j];
	    	
	    	if (j > 0)
	    	{
	    		b->_nssize[i][j] = b->_nssize[i][j-1] + nested[i][j-1];
	    	}
	    	
	    }
	    
	    b->size *= b->_sizes[i];
    }
    
    b->_ssize = GDL_MALLOC(size_t, p);
    
    for(i = 0 ; i < p; i++)
    {
    	b->_ssize[i] = 1;
    	for(j = i+1; j < p; j++)
    	{
    		b->_ssize[i] *= b->_sizes[j];
    	}
    }
    
    b->data = GDL_CALLOC(BASE, b->size);
	
	return b;	
}

void
FUNCTION (gdl_nblock, free) ( TYPE (gdl_nblock) *b)
{
  size_t i;
  GDL_FREE (b->dim);
  GDL_FREE (b->_sizes);
  for( i = 0; i < b->p; i++)
  { 
  	GDL_FREE (b->nested[i]);  	
  	GDL_FREE (b->_nssize[i]);
  }
  GDL_FREE (b->nested);
  GDL_FREE (b->_nssize);
  GDL_FREE (b->_ssize);
  GDL_FREE (b->data);
  GDL_FREE (b);
}

BASE
FUNCTION (gdl_nblock, get) (const TYPE (gdl_nblock) *b, ...)
{
  va_list ap;
  size_t i, j, k, idx=0;

  va_start (ap, b);              

  idx = FUNCTION (gdl_nblock, get_index) (b, NULL, ap);  
  
  va_end (ap);               
  
  return b->data[idx];
}

void
FUNCTION (gdl_nblock, set) ( TYPE (gdl_nblock) *b, ...)
{
  va_list ap;
  int i, idx=0;
  BASE x = 0;

  va_start (ap, b);     

  idx = FUNCTION (gdl_nblock, get_index) (b, &x, ap);
  
  va_end (ap);
    
  b->data[idx] = x;
}

static size_t
FUNCTION (gdl_nblock, get_index) (const TYPE (gdl_nblock) *b, ATOMIC *x, va_list ap)
{
  size_t i, j, jj, k=0, idx=0;
  	
  for (i = 0; i < 2*b->p; i++)
  {
  	
  	j = va_arg (ap, size_t);
  	
  	if (i % 2 != 0)
  	{
  		idx += (b->_nssize[k][jj] + j)*b->_ssize[k];
  		k++;
  	}
  	else
  	{
  		jj = j;
  	}
  }
  
  if (x != 0)
  {
  	*x = va_arg (ap, VARG);
  }
  
  return idx;
}

TYPE (gdl_nblock) *
FUNCTION (gdl_nblock, fread) (FILE * stream)
{
  size_t status, i;
  TYPE (gdl_nblock) * b;
  
  b = GDL_MALLOC(TYPE (gdl_nblock), 1);
  
  status = fread (&b->size, sizeof (size_t), 1, stream);
  GDL_FREAD_STATUS(status, 1);
  status = fread (&b->p, sizeof (size_t), 1, stream);
  GDL_FREAD_STATUS(status, 1);
  b->dim = GDL_MALLOC(size_t, b->p);
  status = fread (b->dim, sizeof (size_t), b->p, stream);
  GDL_FREAD_STATUS(status, b->p);
  b->_ssize = GDL_MALLOC(size_t, b->p);
  status = fread (b->_ssize, sizeof (size_t), b->p, stream);
  GDL_FREAD_STATUS(status, b->p);
  b->_sizes = GDL_MALLOC(size_t, b->p);
  status = fread (b->_sizes, sizeof (size_t), b->p, stream);
  GDL_FREAD_STATUS(status, b->p);
  b->_nssize = GDL_MALLOC(size_t *, b->p);
  for ( i = 0; i < b->p; i++) 
  {
  	b->_nssize[i] = GDL_MALLOC(size_t, b->dim[i]);
	status = fread (b->_nssize[i], sizeof (size_t), b->dim[i], stream);
	GDL_FREAD_STATUS(status, b->dim[i]);
  }
  b->nested = GDL_MALLOC(size_t *, b->p);
  for ( i = 0; i < b->p; i++) 
  {
  	b->nested[i] = GDL_MALLOC(size_t, b->dim[i]);
	status = fread (b->nested[i], sizeof (size_t), b->dim[i], stream);
	GDL_FREAD_STATUS(status, b->dim[i]);
  }
  b->data = GDL_MALLOC(BASE, b->size);
  status = fread (b->data, sizeof (BASE), b->size, stream);
  GDL_FREAD_STATUS(status, b->size);
  
  return b;
}

int
FUNCTION(gdl_nblock, fwrite) (FILE * stream, const TYPE (gdl_nblock) * b)
{
  size_t i, status; 	
	
  status = fwrite (&b->size, sizeof (size_t), 1, stream);
  GDL_FWRITE_STATUS(status, 1);
  status = fwrite (&b->p, sizeof (size_t), 1, stream);
  GDL_FWRITE_STATUS(status, 1);
  status = fwrite (b->dim, sizeof (size_t), b->p, stream);
  GDL_FWRITE_STATUS(status, b->p);
  status = fwrite (b->_ssize, sizeof (size_t), b->p, stream);
  GDL_FWRITE_STATUS(status, b->p);
  status = fwrite (b->_sizes, sizeof (size_t), b->p, stream);
  GDL_FWRITE_STATUS(status, b->p);
  for (i = 0; i < b->p; i++)
  {
  	status = fwrite (b->_nssize[i], sizeof (size_t), b->dim[i], stream);
  	GDL_FWRITE_STATUS(status, b->dim[i]);
  }
  for (i = 0; i < b->p; i++)
  {
  	status = fwrite (b->nested[i], sizeof (size_t), b->dim[i], stream);
    GDL_FWRITE_STATUS(status, b->dim[i]);
  }
  status = fwrite (b->data, sizeof (BASE), b->size, stream);
  GDL_FWRITE_STATUS(status, b->size);
  
  return GDL_SUCCESS;  
}
