/*
 *  genex/gene.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_sort_long.h>
#include <gdl/gdl_genex_chromosome.h>

gdl_genex_chromosome *
gdl_genex_chromosome_alloc (gdl_string * name, const size_t npop)
{
	gdl_genex_chromosome * c;
	
	c = GDL_MALLOC (gdl_genex_chromosome, 1);
	
	c->name   = name;
	c->npop   = npop;
	c->pop_sizes = GDL_CALLOC (size_t, npop);
	c->probes = gdl_hashtable_alloc (gdl_hash_default, 0);
	c->genes  = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	return c;
}

void
gdl_genex_chromosome_free (gdl_genex_chromosome * c)
{
	if (c)
	{
		gdl_hashtable_itr * itr;
		gdl_string_free (c->name);
		gdl_hashtable_free (c->probes);
		if (gdl_hashtable_size (c->genes))
		{
			itr = gdl_hashtable_iterator (c->genes);
			do
			{
				gdl_genex_gene_free (gdl_hashtable_iterator_value (itr));
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
		}
		GDL_FREE (c->pop_sizes);
		GDL_FREE (c);
	}	
}

void
gdl_genex_chromosome_add_probe (gdl_genex_chromosome * c, gdl_genex_probe * probe)
{
	gdl_genex_probe * known_probe;
   if ((known_probe=gdl_hashtable_lookup (c->probes, probe->name))!=0)
   {
     probe->idx  = known_probe->idx;
     return;
   }
	probe->idx  = gdl_hashtable_size (c->probes);
	if (!probe->data) probe->data = GDL_CALLOC (double *, c->npop);
	if (!probe->mean) probe->mean = GDL_CALLOC (double, c->npop);
	if (!probe->var)  probe->var = GDL_CALLOC (double, c->npop);
	gdl_hashtable_add (c->probes, probe->name, probe, 0);
}

gdl_genex_probe *
gdl_genex_chromosome_rm_probe (gdl_genex_chromosome * c, gdl_genex_probe * probe)
{
	gdl_genex_probe * known_probe;
   if ((known_probe=gdl_hashtable_lookup (c->probes, probe->name))!=0)
   {
    	gdl_hashtable_remove (c->probes, probe->name);
   }
   return known_probe;
}

void
gdl_genex_chromosome_add_gene (gdl_genex_chromosome * c, gdl_genex_gene * gene)
{
	gdl_hashtable_add (c->genes, gene->name, gene, 0);
}

gdl_genex_gene *
gdl_genex_chromosome_search_gene (const gdl_genex_chromosome * c, const gdl_string * name)
{
	return (gdl_genex_gene *) gdl_hashtable_lookup (c->genes, name);
}

gdl_genex_probe *
gdl_genex_chromosome_search_probe (const gdl_genex_chromosome * c, const gdl_string * name)
{
	return (gdl_genex_probe *) gdl_hashtable_lookup (c->probes, name);
}

void
gdl_genex_chromosome_set_popsize (gdl_genex_chromosome * c, size_t p, size_t n)
{
	c->pop_sizes[p]=n;
	if (gdl_hashtable_size (c->probes))
	{
		gdl_hashtable_itr * itr;
		
		itr = gdl_hashtable_iterator (c->probes);
		do
		{
			gdl_genex_probe * probe = (gdl_genex_probe *) gdl_hashtable_iterator_value (itr);
			probe->data[p] = GDL_CALLOC (double, n);
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}	
}

size_t
gdl_genex_chromosome_gene_size (const gdl_genex_chromosome * c)
{
	return gdl_hashtable_size (c->genes);	
}

size_t
gdl_genex_chromosome_probe_size (const gdl_genex_chromosome * c)
{
	return gdl_hashtable_size (c->probes);
}

gdl_genex_gene **
gdl_genex_chromosome_genes (const gdl_genex_chromosome * c)
{
	size_t i, j, n;
	long * pos;
	size_t * idx, * swp;
	gdl_hashtable_itr * itr;
	gdl_genex_gene ** tmp;
	gdl_genex_gene ** genes;
	
	n = gdl_hashtable_size (c->genes);
	
	genes = GDL_MALLOC (gdl_genex_gene *, n);
	pos   = GDL_MALLOC (long, n);
	idx   = GDL_MALLOC (size_t, n);
	//swp   = GDL_MALLOC (size_t, n);
	itr   = gdl_hashtable_iterator (c->genes);
	i = 0;
	do
	{
		genes[i]=(gdl_genex_gene *)gdl_hashtable_iterator_value(itr);
		pos[i]=genes[i]->start;
		i++;
	}
	while (gdl_hashtable_iterator_next (itr));
	gdl_hashtable_iterator_free (itr);
		
	gdl_sort_long_index (idx, pos, 1, n);
	
	tmp = GDL_MALLOC (gdl_genex_gene *, n);
	
	for (i = 0; i < n; i++)
	{
		tmp[i]=genes[idx[i]];
	}
	
	GDL_FREE (genes);
	GDL_FREE (pos);
	GDL_FREE (idx);
	//GDL_FREE (swp);
	
	return tmp;	
}

gdl_genex_probe **
gdl_genex_chromosome_probes (const gdl_genex_chromosome * c)
{
	size_t i, n;
	gdl_genex_probe ** probes=0;
	gdl_hashtable_itr * itr;
	
	n = gdl_hashtable_size (c->probes);
	if (n)
	{
		itr = gdl_hashtable_iterator (c->probes);
		probes = GDL_MALLOC (gdl_genex_probe *, n);
		do
		{
			gdl_genex_probe * probe = (gdl_genex_probe *) gdl_hashtable_iterator_value (itr);
			probes[probe->idx] = probe;
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	return probes;
}

gdl_genex_chromosome *
gdl_genex_chromosome_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, n;
		gdl_string * name;
		gdl_genex_probe ** probes;
		gdl_genex_chromosome * c;
		
		name = gdl_string_fread(stream);
		GDL_FREAD_STATUS (name!=0, 1);
		status = fread (&n, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		c = gdl_genex_chromosome_alloc (name, n);
		
		status = fread (c->pop_sizes, sizeof (size_t), c->npop, stream);
		GDL_FREAD_STATUS (status, c->npop);
		status = fread (&n, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		probes = GDL_MALLOC (gdl_genex_probe *, n);
		for (i = 0; i < n; i++)
		{
			probes[i] = gdl_genex_probe_fread (stream, c->npop, c->pop_sizes);
			GDL_FREAD_STATUS (probes[i]!=0, 1);
			gdl_genex_chromosome_add_probe (c, probes[i]);
		}
		status = fread (&n, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		for (i = 0; i < n; i++)
		{
		   gdl_genex_gene * gene = gdl_genex_gene_fread (stream, probes);
		   GDL_FREAD_STATUS (gene!=0, 1);
			gdl_genex_chromosome_add_gene (c, gene);
		}
		GDL_FREE (probes);
		
		return c;
	}
	return 0;
}

int
gdl_genex_chromosome_fwrite (FILE * stream, const gdl_genex_chromosome * c)
{
	if (stream && c)
	{
		int status;
		size_t i, n;
		gdl_hashtable_itr * itr;
		gdl_genex_probe ** probes;
		
		status = gdl_string_fwrite (stream, c->name);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = fwrite (&c->npop, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (c->pop_sizes, sizeof (size_t), c->npop, stream);
		GDL_FWRITE_STATUS (status, c->npop);
		n = gdl_hashtable_size (c->probes);
		status = fwrite (&n, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (n)
		{
			itr = gdl_hashtable_iterator (c->probes);
			probes = GDL_MALLOC (gdl_genex_probe *, n); 
			do
			{
				gdl_genex_probe * probe = (gdl_genex_probe *) gdl_hashtable_iterator_value (itr);
				probes[probe->idx] = probe;
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
			for (i = 0; i < n; i++)
			{
				status = gdl_genex_probe_fwrite (stream, probes[i], c->npop, c->pop_sizes);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);				
			}
			GDL_FREE (probes);
		}
		
		n = gdl_hashtable_size (c->genes);
		
		status = fwrite (&n, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (n)
		{
			itr = gdl_hashtable_iterator (c->genes);
			do
			{
			   status = gdl_genex_gene_fwrite (stream, gdl_hashtable_iterator_value (itr));
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}
