/*
 *  genex/gene.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_genex_gene.h>

gdl_genex_gene *
gdl_genex_gene_alloc (gdl_string * name, const long id)
{
	gdl_genex_gene * g;
	
	g = GDL_CALLOC (gdl_genex_gene, 1);
	
	g->name   = name;
	g->id     = id;
	g->ignore = 'n';
	
	return g;
}

void
gdl_genex_gene_free (gdl_genex_gene * g)
{
	if (g)
	{
		if (g->blocks)
		{
			size_t i;
			for (i = 0; i < g->size; i++)
			{
				gdl_genex_block_free (g->blocks[i]);	
			}
			GDL_FREE (g->blocks);
		}
		gdl_string_free (g->name);
		GDL_FREE (g->exonStarts);
		GDL_FREE (g->exonEnds);
		GDL_FREE (g);
	}
}

void
gdl_genex_gene_add_block (gdl_genex_gene * g, gdl_genex_block * b)
{
	if (g->size)
	{
		gdl_genex_block ** tmp = GDL_MALLOC (gdl_genex_block *, g->size+1);
		memcpy (tmp, g->blocks, sizeof(gdl_genex_block *)*g->size);
		GDL_FREE (g->blocks);
		g->blocks = tmp;
		g->size++;
		if (b->start < g->start)
		{
			g->start = b->start;	
		}
		if (b->end > g->end)
		{
			g->end = b->end;	
		}
	}
	else
	{
		g->blocks = GDL_MALLOC (gdl_genex_block *, 1);
		g->size=1;
		g->start = b->start;
		g->end   = b->end;
	}
	g->blocks[g->size-1]=b;
		
}
// use of strand is now deprecated
gdl_genex_block *
gdl_genex_gene_search_block (const gdl_genex_gene * g, const long start, const long end, const unsigned char strand)
{
	if (!g->size) return 0;
	
	size_t i;
	
	for (i = 0; i < g->size; i++)
	{
		if (g->blocks[i]->start==start
		    && g->blocks[i]->end==end)
		    //&& g->blocks[i]->strand==strand)
		{    
			return g->blocks[i];
		}
	}
	
	return 0;
}

void
gdl_genex_gene_remove_block (gdl_genex_gene * g, size_t b)
{
	size_t i;
	
	gdl_genex_block_free (g->blocks[b]);
	
	for (i = b; i < g->size-1; i++)
	{
		g->blocks[i]=g->blocks[i+1];
	}
	
	g->size--;
}

// assume i2 > i1
void
gdl_genex_gene_merge_block (gdl_genex_gene * g, size_t i1, size_t i2)
{
	size_t i;
	gdl_genex_block * b1, * b2;
	gdl_hashtable * probes;
	gdl_hashtable_itr * itr;
	
	b1 = g->blocks[i1];
	b2 = g->blocks[i2];
	
	probes = gdl_hashtable_alloc (gdl_hash_default, b1->size+b2->size);
	
	for (i = 0; i < b1->size; i++) gdl_hashtable_add (probes, b1->probes[i]->name, b1->probes[i], 0);
	for (i = 0; i < b2->size; i++) gdl_hashtable_add (probes, b2->probes[i]->name, b2->probes[i], 0);
	
	GDL_FREE (b1->probes);
	b1->size = gdl_hashtable_size (probes);
	b1->probes = GDL_MALLOC (gdl_genex_probe *, b1->size);
	
	itr = gdl_hashtable_iterator (probes);
	i = 0;
	do
	{
		b1->probes[i++] = gdl_hashtable_iterator_value (itr);
	}
	while (gdl_hashtable_iterator_next (itr));
	gdl_hashtable_iterator_free (itr);
	
	b1->start = GDL_MIN (b1->start, b2->start);
	b1->end   = GDL_MAX (b1->end, b2->end);
	
	gdl_genex_gene_remove_block (g, i2);
}

gdl_genex_gene *
gdl_genex_gene_fread (FILE * stream, gdl_genex_probe ** probes)
{
	if (stream && probes)
	{
		int status;
		size_t i;
		gdl_string * name;
		long id;
		gdl_genex_gene * g;
		
		name = gdl_string_fread (stream);
		GDL_FREAD_STATUS (name!=0, 1);
		status = fread (&id, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g = gdl_genex_gene_alloc (name, id);
		
		status = fread (&g->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g->blocks = GDL_MALLOC (gdl_genex_block *, g->size);
		
		for (i = 0; i < g->size; i++)
		{
			g->blocks[i] = gdl_genex_block_fread (stream, probes);
			GDL_FREAD_STATUS (g->blocks[i]!=0, 1);
		}
		status = fread (&g->ignore, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		status = fread (&g->strand, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		status = fread (&g->txStart, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&g->txEnd, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		status = fread (&g->cdsStart, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&g->cdsEnd, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		status = fread (&g->exonCount, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (g->exonCount)
		{
			g->exonStarts = GDL_MALLOC (long, g->exonCount);	
			g->exonEnds   = GDL_MALLOC (long, g->exonCount);
			
			status = fread (g->exonStarts, sizeof(long), g->exonCount, stream);
			GDL_FREAD_STATUS (status, g->exonCount);
			status = fread (g->exonEnds, sizeof(long), g->exonCount, stream);
			GDL_FREAD_STATUS (status, g->exonCount);
		}
		
		return g;
	}
	return 0;	
}

int
gdl_genex_gene_fwrite (FILE * stream, const gdl_genex_gene * g)
{
	if (stream && g)
	{
		int status;
		size_t i;
		
		status = gdl_string_fwrite (stream, g->name);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = fwrite (&g->id, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < g->size; i++)
		{
			status = gdl_genex_block_fwrite (stream, g->blocks[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		status = fwrite (&g->ignore, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		status = fwrite (&g->strand, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->txStart, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->txEnd, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->cdsStart, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->cdsEnd, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		status = fwrite (&g->exonCount, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (g->exonCount)
		{
			status = fwrite (g->exonStarts, sizeof(long), g->exonCount, stream);
			GDL_FWRITE_STATUS (status, g->exonCount);
			status = fwrite (g->exonEnds, sizeof(long), g->exonCount, stream);
			GDL_FWRITE_STATUS (status, g->exonCount);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

