/*  
 * 	entity/entity.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <string.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_entity.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_allele.h>
#include <gdl/gdl_genotype.h>
#include <gdl/gdl_accession.h>
#include <gdl/gdl_chromosome.h>
#include <gdl/gdl_genome.h>
#include <gdl/gdl_factor.h>
#include <gdl/gdl_factor_level.h>

static const gdl_entity_type *
_gdl_entity_get_type (const char * name)
{
	if (!strcmp (name, GDL_ENTITY->name))
		return GDL_ENTITY;
	if (!strcmp (name, GDL_ACCESSION->name))
		return GDL_ACCESSION;
	if (!strcmp (name, GDL_LOCUS->name))
		return GDL_LOCUS;
	if (!strcmp (name, GDL_ALLELE->name))
		return GDL_ALLELE;
	if (!strcmp (name, GDL_ALLELE_MISSING->name))
		return GDL_ALLELE_MISSING;
	if (!strcmp (name, GDL_ALLELE_RECESSIVE->name))
		return GDL_ALLELE_RECESSIVE;
	if (!strcmp (name, GDL_GENOTYPE->name))
		return GDL_GENOTYPE;
	if (!strcmp (name, GDL_CHROMOSOME->name))
		return GDL_CHROMOSOME;
	if (!strcmp (name, GDL_GENOME->name))
		return GDL_GENOME;
	if (!strcmp (name, GDL_FACTOR->name))
		return GDL_FACTOR;
	if (!strcmp (name, GDL_FACTOR_LEVEL->name))
		return GDL_FACTOR_LEVEL;
	return NULL;
}

gdl_entity *
gdl_entity_alloc (const gdl_entity_type * T)
{
	gdl_entity * va;
	
	va = GDL_CALLOC (gdl_entity, 1);
	
	if (va == 0)
	{
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_entity_alloc",
		               GDL_ENOMEM,
		               0);
	}
	
	va->type  = T;
	
	if (va->type->size)
	{
		va->state = gdl_malloc (T->size);
		if (va->state == 0)
		{
			GDL_FREE (va);
			GDL_ERROR_VAL ("Unable to allocate memory in gdl_entity_alloc",
			               GDL_ENOMEM,
			               0);	
		}
		(va->type->alloc)(va->state);
	}
	
	va->name = gdl_string_alloc (1);
	
	va->extra = NULL;
	
	return va;
}

void
gdl_entity_free (gdl_entity * va)
{
	if (va)
	{
		if (va->type->free != NULL)
			(va->type->free)(va->state);
		GDL_FREE (va->state);
		GDL_FREE (va->name);
		GDL_FREE (va);
	}
}

gdl_entity *
gdl_entity_clone (const gdl_entity * va)
{
	if (va == 0)
		return NULL;
	else
	{
		gdl_entity * clone = gdl_entity_alloc (va->type);
		clone->idx  = va->idx;
		clone->name = gdl_string_clone (va->name);
		if (va->type->copy)
			(va->type->copy)(clone->state, va->state);
		clone->extra = va->extra;
		
		return clone;
	}
}

int
gdl_entity_compare (const gdl_entity * e1, const gdl_entity * e2)
{
	if (e1 == 0 || e2 == 0)
		return (-1);
	else if (e1 == e2)
	{
		return (0);
	}
	else if (e1->type != e2->type)
	{
		return (-1);	
	}
	else
	{
		int diff;
		if (e1->name != NULL && e2->name != NULL)
		{
			 diff = strcoll (e1->name, e2->name);
			 if (diff)
				return (diff);
		}
		diff = e1->idx - e2->idx;
		if (diff)
			return (diff);
		if (e1->type->compare != NULL)
			return (e1->type->compare)(e1, e2);
		else
			return (0);
	}
}

gdl_entity *
gdl_entity_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t status;
		char * type;
		gdl_boolean has;
		gdl_entity * va;
		const gdl_entity_type * T;
		
		type = gdl_string_fread (stream);
		GDL_FREAD_STATUS (type != NULL, 1);
		
		T = _gdl_entity_get_type (type);
		
		GDL_FREE (type);

		va = gdl_entity_alloc (T);
		
		gdl_string_free (va->name);
		
		status = fread (&has, sizeof(gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (has)
		{
			va->name = gdl_string_fread (stream);
			GDL_FREAD_STATUS (va->name != NULL, 1);
		}
		else
		{
			va->name = NULL;	
		}
		
		status = fread (&va->idx, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (va->type->fread)
		{
			status = (va->type->fread)(stream, va->state);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return va;	
	}
}

int
gdl_entity_fwrite (FILE * stream, const gdl_entity * va)
{
	if (stream == 0 || va == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		gdl_boolean has;
		
		status = gdl_string_fwrite (stream, va->type->name);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		has = (va->name) ? gdl_true : gdl_false;
		status = fwrite (&has, sizeof (gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has)
		{
			status = gdl_string_fwrite (stream, va->name);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		status = fwrite (&va->idx, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (va->type->fwrite)
		{
			status = (va->type->fwrite)(stream, va->state);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
}

void
gdl_entity_set_name (gdl_entity * entity, const gdl_string * name)
{
	if (entity->name)
		gdl_string_free (entity->name);
	entity->name = gdl_string_clone (name);
}

void
gdl_entity_set_idx (gdl_entity * entity, const size_t idx)
{
	entity->idx = idx;	
}

const gdl_string *
gdl_entity_get_name (const gdl_entity * entity)
{
	return entity->name;	
}

size_t
gdl_entity_get_idx (const gdl_entity * entity)
{
	return entity->idx;	
}

const gdl_entity_type *
gdl_entity_get_type (const gdl_entity * entity)
{
	return entity->type;	
}

static void
_gdl_entity_free (void * vva)
{
	gdl_entity * va = (gdl_entity *) vva;
	gdl_entity_free (va);
}

static void *
_gdl_entity_clone (const void * vva)
{
	gdl_entity * va = (gdl_entity *) vva;
	return gdl_entity_clone (va);
}

static int
_gdl_entity_compare (const void * vva1, const void * vva2)
{
	gdl_entity * va1 = (gdl_entity *) vva1;
	gdl_entity * va2 = (gdl_entity *) vva2;
	return gdl_entity_compare (va1, va2);
}

static void *
_gdl_entity_fread (FILE * stream)
{
	return gdl_entity_fread (stream);
}

static int
_gdl_entity_fwrite (FILE * stream, const void * vva)
{
	gdl_entity * va = (gdl_entity *) vva;
	return gdl_entity_fwrite (stream, va);
}

static const gdl_entity_type _gdl_entity_type =
{
	0,
	"E",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

const gdl_entity_type * GDL_ENTITY = &_gdl_entity_type;

static const gdl_data_interface _gdl_entity_interface =
{
	&_gdl_entity_free,
	&_gdl_entity_clone,
	&_gdl_entity_compare,
    &_gdl_entity_fread,
    &_gdl_entity_fwrite
};

const gdl_data_interface * gdl_entity_interface = &_gdl_entity_interface;

