/*  
 * 	entity/genome.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_chromosome.h>
#include <gdl/gdl_genome.h>

typedef struct 
{
	size_t size;
	gdl_hashtable  * table;
	gdl_chromosome ** _chroms;
} _gdl_genome;

static int
_gdl_genome_alloc (void * vg)
{
	if (vg)
	{
		_gdl_genome * g = (_gdl_genome *) vg;
		g->table   = NULL;
		g->_chroms = NULL;
		g->size    = 0;
	}
	return GDL_SUCCESS;
}

static void
_gdl_genome_free (void * vg)
{
	if (vg)
	{
		_gdl_genome * g = (_gdl_genome *) vg;
		
		gdl_hashtable_free (g->table);
		g->table = NULL;
		GDL_FREE (g->_chroms);
		g->_chroms = NULL;
		g->size = 0;
	}
}

static int
_gdl_genome_grow (_gdl_genome * g, size_t size)
{
	size_t i;
	
	if (g->size)
	{
		gdl_chromosome ** nc;
		
		nc = GDL_CALLOC (gdl_chromosome *, size);
		memcpy (nc, g->_chroms, sizeof(gdl_chromosome *)*size);
		GDL_FREE (g->_chroms);
		g->_chroms = nc;
	}
	else
	{
		g->table   = gdl_hashtable_alloc (gdl_entity_interface, size);
		g->_chroms = GDL_CALLOC (gdl_chromosome *, size);
	}
	
	g->size = size;
	
	return GDL_SUCCESS;
}

static int
_gdl_genome_copy (void * vdest, const void * vsrc)
{
	if (vdest == 0 || vsrc == 0)
		return (-1);
	else
	{
		size_t i;
		_gdl_genome * dest = (_gdl_genome *) vdest;
		_gdl_genome * src  = (_gdl_genome *) vsrc;
		_gdl_genome_free (dest);
		_gdl_genome_grow (dest, src->size);
		for (i = 0; i < src->size; i++)
		{
			gdl_chromosome * cc = gdl_entity_clone (src->_chroms[i]);
			gdl_hashtable_add (dest->table, cc->name, cc, 1);
			dest->_chroms[i] = cc;
		}
		return GDL_SUCCESS;
	}	
}

static int
_gdl_genome_fread (FILE * stream, void * vg)
{
	if (stream == 0 || vg == 0)
		return GDL_EINVAL;
	else
	{
		size_t size, status;
		_gdl_genome * g = (_gdl_genome *) vg;
		
		_gdl_genome_free (g);
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (size)
		{
			size_t i;
			
			_gdl_genome_grow (g, size);
			
			for (i = 0; i < g->size; i++)
			{
				g->_chroms[i] = gdl_entity_fread (stream);
				GDL_FWRITE_STATUS (g->_chroms[i] == 0, 0);
				gdl_hashtable_add (g->table, g->_chroms[i]->name, g->_chroms[i], 1);
			}
		}
		return GDL_SUCCESS;
	}
}

static int
_gdl_genome_fwrite (FILE * stream, const void * vg)
{
	if (stream == 0 || vg == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_genome * g = (_gdl_genome *) vg;
		
		status = fwrite (&g->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (g->size)
		{
			size_t i;
			
			for (i = 0; i < g->size; i++)
			{
				status = gdl_entity_fwrite (stream, g->_chroms[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}		  
		}
				
		return GDL_SUCCESS;	
	}
}


gdl_genome *
gdl_genome_alloc (void)
{
	gdl_genome * vg = gdl_entity_alloc (GDL_GENOME);
	return vg;
}

gdl_genome *
gdl_genome_new (const char * name)
{
	gdl_genome * ce = gdl_genome_alloc ();
	gdl_entity_set_name (ce, name);
	return ce;
}

void
gdl_genome_free (gdl_genome * g)
{
	gdl_entity_free (g);
}

int
gdl_genome_add (gdl_genome * vg, gdl_chromosome * c)
{
	if (vg == 0 || c == 0)
	{
		return (-1);
	}
	else
	{
		int i;
		_gdl_genome * g = (_gdl_genome *) vg->state;
		i = g->size;
		_gdl_genome_grow (g, g->size+1);
		g->_chroms[i] = c;
		gdl_hashtable_add (g->table, c->name, c, 1);
		return (i);
	}
}

size_t
gdl_genome_size (const gdl_genome * vg)
{
	return ((_gdl_genome *)vg->state)->size;
}

gdl_chromosome *
gdl_genome_search (const gdl_genome * vg, const char * name)
{
	if (vg == 0 || name == 0)
	{
		return NULL;
	}
	else
	{
		_gdl_genome * g = (_gdl_genome *) vg->state;
		return (gdl_chromosome *)
			gdl_hashtable_lookup (g->table, name);
	}
}

gdl_chromosome *
gdl_genome_get (const gdl_genome * vg, size_t i)
{
	if (vg == 0)
	{
		return NULL;
	}
	else
	{
		_gdl_genome * g = (_gdl_genome *) vg->state;
		return g->_chroms[i];
	}
}

int
gdl_genome_fprintf (FILE * stream, const gdl_genome * vg)
{
	if (vg == 0 || stream == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		size_t i;
		
		_gdl_genome * g = (_gdl_genome *) vg->state;
		
		fprintf (stream, ">GENOME %s\n", vg->name);
		
		for (i = 0; i < g->size; i++)
		{
			gdl_chromosome_fprintf (stream, g->_chroms[i]);
		}
		
		return GDL_SUCCESS;
	}	
}

static const gdl_entity_type _gdl_genome_type =
{
	sizeof (_gdl_genome),
	"GE",
	&_gdl_genome_alloc,
	&_gdl_genome_free,
	NULL,
	&_gdl_genome_copy,
    &_gdl_genome_fread,
    &_gdl_genome_fwrite
};

const gdl_entity_type * GDL_GENOME = &_gdl_genome_type;
