/*  
 * 	entity/genotype.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_allele.h>
#include <gdl/gdl_genotype.h>

typedef struct 
{
	gdl_list * alleles;
} _gdl_genotype;

int
_gdl_genotype_alloc (void * vg)
{
	_gdl_genotype * g = (_gdl_genotype *) vg;
	
	g->alleles = gdl_list_alloc (gdl_entity_interface);
	
	return GDL_SUCCESS;
}

void
_gdl_genotype_free (void * vg)
{
	if (vg)
	{
		_gdl_genotype * g = (_gdl_genotype *) vg;
		gdl_list_free (g->alleles);
	}
}

int
_gdl_genotype_copy (void * vdest, const void * vsrc)
{
	if (vdest == 0 || vsrc == 0)
		return (-1);
	else
	{
		gdl_list_itr * itr;
		_gdl_genotype * dest = (_gdl_genotype *) vdest;
		_gdl_genotype * src  = (_gdl_genotype *) vsrc;
		
		gdl_list_free (dest->alleles);
		
		dest->alleles = gdl_list_alloc (gdl_entity_interface);
		
		itr = gdl_list_iterator_front (src->alleles);
		do
		{
			gdl_allele * a1 = (gdl_allele *) gdl_list_iterator_value (itr);
			gdl_allele * a2 = gdl_entity_clone (a1);
			gdl_list_insert (dest->alleles, a2, 1);
		} while (gdl_list_iterator_next (itr));
		
		gdl_list_iterator_free (itr);
		
		return GDL_SUCCESS;
	}	
}

int
_gdl_genotype_fread (FILE * stream, void * vg)
{
	if (stream == 0 || vg == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_genotype * g = (_gdl_genotype *) vg;
		
		status = gdl_list_fread (stream, g->alleles);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
}

int
_gdl_genotype_fwrite (FILE * stream, const void * vg)
{
	if (stream == 0 || vg == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_genotype * g = (_gdl_genotype *) vg;
		
		status = gdl_list_fwrite (stream, g->alleles);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				
		return GDL_SUCCESS;	
	}
}

static void
_gdl_genotype_create_name (gdl_genotype * vg)
{
	size_t len = 0;
	gdl_list_itr * itr;
	char * name;
	_gdl_genotype * g = (_gdl_genotype *) vg->state;
	
	GDL_FREE (vg->name);
	
	itr = gdl_list_iterator_front (g->alleles);
	
	do
	{
		gdl_allele * va = (gdl_allele *)
		                        gdl_list_iterator_value (itr);
			                       
		if (va->type != GDL_ALLELE_MISSING
		    && va->type != GDL_ALLELE_RECESSIVE)
		{
			len += strlen (va->name);
		}
		len++;
	} while	(gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	vg->name = GDL_CALLOC (char, len);
	
	itr = gdl_list_iterator_front (g->alleles);
	
	do
	{
		gdl_allele * va = (gdl_allele *)
		                        gdl_list_iterator_value (itr);
		
		if (va->type != GDL_ALLELE_MISSING
		    && va->type != GDL_ALLELE_RECESSIVE)
		{
			strcat (vg->name, va->name);
		}
		
		if (gdl_list_iterator_next (itr))
		{
			strcat (vg->name, "/");
		}
		else
		{
			break;	
		}		
	} while	(0 == 0);
	
	gdl_list_iterator_free (itr);
}

gdl_genotype *
gdl_genotype_alloc (void)
{
	return gdl_entity_alloc (GDL_GENOTYPE);	
}

void
gdl_genotype_free (gdl_genotype * g)
{
	gdl_entity_free (g);
}

gdl_list_itr *
gdl_genotype_allele_iterator (const gdl_genotype * g)
{
	return gdl_list_iterator_front (((_gdl_genotype *)g->state)->alleles);
}

int
gdl_genotype_add (gdl_genotype * vg, gdl_allele * va, size_t owner)
{
	if (vg == 0 || va == 0)
		return (-1);
	gdl_list_insert (((_gdl_genotype *)vg->state)->alleles, va, owner);
	_gdl_genotype_create_name (vg);
	return (0);
}

int
gdl_genotype_add_missing (gdl_genotype * vg)
{
	if (vg == 0)
		return (-1);
	else
	{	
		gdl_allele * a    = gdl_entity_clone (gdl_allele_missing);
		_gdl_genotype * g = (_gdl_genotype *) vg->state;
		gdl_list_insert (g->alleles, a, 1);
		_gdl_genotype_create_name (vg);
	}
	return (0);
}

int
gdl_genotype_add_recessive (gdl_genotype * vg)
{
	if (vg == 0)
		return (-1);
    else
    {
    	gdl_allele * a = gdl_entity_clone (gdl_allele_recessive);
		gdl_list_insert (((_gdl_genotype *)vg->state)->alleles, a, 1);
		_gdl_genotype_create_name (vg);
    }
	return (0);
}

gdl_allele *
gdl_genotype_get_allele (const gdl_genotype * vg, size_t p)
{
	return (gdl_allele *)
		gdl_list_get (((_gdl_genotype *)vg->state)->alleles, p);
}

size_t
gdl_genotype_size (const gdl_genotype * vg)
{
	return gdl_list_size (((_gdl_genotype *)vg->state)->alleles);
}

gdl_boolean
gdl_genotype_is_homozygous (const gdl_genotype * vg)
{
	_gdl_genotype * g  = (_gdl_genotype *)vg->state;
	int idx = -1;
	gdl_list_itr * itr 
	     = gdl_list_iterator_front (g->alleles);
	do
	{
		gdl_allele * a = (gdl_allele *)
		                        gdl_list_iterator_value (itr);
		if (a->type == GDL_ALLELE_MISSING
		    || a->type == GDL_ALLELE_RECESSIVE)
		{
			gdl_list_iterator_free (itr);
			return gdl_false;
		}
		if (idx >= 0 && a->idx != (size_t)idx)
		{
			gdl_list_iterator_free (itr);
			return gdl_false;
		}
		else
		{
			idx = a->idx;
		}
	} while	(gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	return gdl_true;
}

gdl_boolean
gdl_genotype_has_missing (const gdl_genotype * vg)
{
	_gdl_genotype * g  = (_gdl_genotype *)vg->state;
	int idx = -1;
	gdl_list_itr * itr 
	     = gdl_list_iterator_front (g->alleles);
	do
	{
		gdl_allele * a = (gdl_allele *)
		                        gdl_list_iterator_value (itr);
		if (a->type == GDL_ALLELE_MISSING)
		{
			gdl_list_iterator_free (itr);
			return gdl_true;
		}
	} while	(gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	return gdl_false;	
}

gdl_boolean
gdl_genotype_has_recessive (const gdl_genotype * vg)
{
	_gdl_genotype * g  = (_gdl_genotype *)vg->state;
	int idx = -1;
	gdl_list_itr * itr 
	     = gdl_list_iterator_front (g->alleles);
	do
	{
		gdl_allele * a = (gdl_allele *)
		                        gdl_list_iterator_value (itr);
		if (a->type == GDL_ALLELE_RECESSIVE)
		{
			gdl_list_iterator_free (itr);
			return gdl_true;
		}
	} while	(gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	return gdl_false;		
}


static const gdl_entity_type _gdl_genotype_type =
{
	sizeof (_gdl_genotype),
	"G",
	&_gdl_genotype_alloc,
	&_gdl_genotype_free,
	NULL,
	&_gdl_genotype_copy,
    &_gdl_genotype_fread,
    &_gdl_genotype_fwrite
};

const gdl_entity_type * GDL_GENOTYPE = &_gdl_genotype_type;
