/*  
 * 	entity/locus.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_allele.h>
#include <gdl/gdl_genotype.h>

enum _gdl_locus_mode
{
	codominant,
	dominant
};

typedef struct
{
	size_t _na;
	size_t _ng;
	gdl_hashtable * alleles;
	gdl_hashtable * genotypes;
	gdl_allele   ** _alleles;
	gdl_genotype ** _genotypes;
} _gdl_locus;

static int
_gdl_locus_alleles_grow (_gdl_locus * vl, size_t n)
{
	if (vl->_alleles == NULL)
	{
		vl->_alleles = GDL_MALLOC (gdl_allele *, n);
	}
	else
	{
		gdl_allele ** new = GDL_MALLOC (gdl_allele *, n);
		memcpy (new, vl->_alleles, sizeof(gdl_allele *)*vl->_na);
		GDL_FREE (vl->_alleles);
		vl->_alleles = new;
	}
	vl->_na = n;
}

static int
_gdl_locus_genotypes_grow (_gdl_locus * vl, size_t n)
{
	if (vl->_genotypes == NULL)
	{
		vl->_genotypes = GDL_MALLOC (gdl_genotype *, n);
	}
	else
	{
		gdl_genotype ** new = GDL_MALLOC (gdl_genotype *, n);
		memcpy (new, vl->_genotypes, sizeof(gdl_genotype *)*vl->_ng);
		GDL_FREE (vl->_genotypes);
		vl->_genotypes = new;
	}
	vl->_ng = n;
}

static int
_gdl_locus_alloc (void * vlocus)
{
	_gdl_locus * va = (_gdl_locus *) vlocus;
	va->alleles    = gdl_hashtable_alloc (gdl_entity_interface, 0);
	va->genotypes  = gdl_hashtable_alloc (gdl_entity_interface, 0);
	va->_alleles   = NULL;
	va->_genotypes = NULL;
	va->_na        = 0;
	va->_ng        = 0;	
	return GDL_SUCCESS;
}

static void
_gdl_locus_free (void * vlocus)
{
	if (vlocus)
	{
		_gdl_locus * va = (_gdl_locus *) vlocus;
		gdl_hashtable_free (va->alleles);
		gdl_hashtable_free (va->genotypes);
		GDL_FREE (va->_alleles);
		GDL_FREE (va->_genotypes);
	}
}

static int
_gdl_locus_copy (void * vdest, const void * vsrc)
{
	if (vdest == 0 || vsrc == 0)
		return GDL_EINVAL;
	else
	{
		size_t i;
		
		_gdl_locus * dest = (_gdl_locus *) vdest;
		_gdl_locus * src = (_gdl_locus *) vsrc;
		
		_gdl_locus_free (dest);
		_gdl_locus_alloc (dest);
		
		_gdl_locus_alleles_grow (dest, src->_na);
		_gdl_locus_genotypes_grow (dest, src->_ng);
		
		for (i = 0; i < dest->_na; i++)
		{
			dest->_alleles[i] = gdl_entity_clone (src->_alleles[i]);
			gdl_hashtable_add (dest->alleles, dest->_alleles[i]->name, dest->_alleles[i], 1);			
		}
		
		for (i = 0; i < dest->_ng; i++)
		{
			dest->_genotypes[i] = gdl_entity_clone (src->_genotypes[i]);
			gdl_hashtable_add (dest->genotypes, dest->_genotypes[i]->name, dest->_genotypes[i], 1);	
		}
		
		return GDL_SUCCESS;
	}
}

static int
_gdl_locus_compare (const void * v1, const void * v2)
{
	if (v1 == 0 || v2 == 0)
	{
		return GDL_EINVAL;
	}
	else if (v1 == v2)
	{
		return 0;
	}
	else
	{
		_gdl_locus * l1 = (_gdl_locus *) v1;
		_gdl_locus * l2 = (_gdl_locus *) v2;
		
		// TODO
		
		return 0;
	}
}

static int
_gdl_locus_fread (FILE * stream, void * vlocus)
{
	if (stream == 0 || vlocus == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_locus * vl;
		
		vl = (_gdl_locus *) vlocus;
		
		status = gdl_hashtable_fread (stream, vl->alleles);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		status = gdl_hashtable_fread (stream, vl->genotypes);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		vl->_na = gdl_hashtable_size (vl->alleles);
		
		if (vl->_na)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (vl->alleles);
			
			_gdl_locus_alleles_grow (vl, vl->_na);
			
			do
			{
				gdl_allele * a =
					gdl_hashtable_iterator_value (itr);
				vl->_alleles[a->idx] = a; 
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		vl->_ng = gdl_hashtable_size (vl->genotypes);
		
		if (vl->_ng)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (vl->genotypes);
			
			_gdl_locus_genotypes_grow (vl, vl->_ng);
			
			do
			{
				gdl_genotype * g =
					gdl_hashtable_iterator_value (itr);
				vl->_genotypes[g->idx] = g;
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		return GDL_SUCCESS;
	}
}

static int
_gdl_locus_fwrite (FILE * stream, const void * vlocus)
{
	if (stream == 0 || vlocus == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_locus * vl = (_gdl_locus *) vlocus;
		
		status = gdl_hashtable_fwrite (stream, vl->alleles);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_hashtable_fwrite (stream, vl->genotypes);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				
		return GDL_SUCCESS;	
	}
}

gdl_locus *
gdl_locus_alloc (void)
{
	return gdl_entity_alloc (GDL_LOCUS);
}

gdl_locus *
gdl_locus_new (const char * name)
{
	gdl_locus * l = gdl_entity_alloc (GDL_LOCUS);
	gdl_entity_set_name (l, name);
	return l;
}

void
gdl_locus_free (gdl_locus * l)
{
	gdl_entity_free (l);	
}

size_t
gdl_locus_allele (const gdl_locus * l)
{
	return ((_gdl_locus *)l->state)->_na;
}

size_t
gdl_locus_genotype (const gdl_locus * l)
{
	return ((_gdl_locus *)l->state)->_ng;
}

size_t
gdl_locus_informative_genotype (const gdl_locus * vl)
{
	size_t i, ni = 0;
	_gdl_locus   * l = (_gdl_locus *) vl->state;
	
	for (i = 0; i < l->_ng; i++)
	{
		if (!gdl_genotype_has_missing (l->_genotypes[i]))
		{
			ni++;	
		}
	}
	
	return ni;
}

void
gdl_locus_add_allele (gdl_locus * vl, gdl_allele ** va, size_t owner)
{
	if (vl == 0 || va == 0)
		return;
	else
	{
		_gdl_locus   * l = (_gdl_locus *) vl->state;
		gdl_allele   * a = (gdl_allele *) gdl_hashtable_lookup (l->alleles, (*va)->name);
		
		if (a == NULL)
		{
			gdl_hashtable_add (l->alleles, (*va)->name, *va, owner);
			(*va)->idx = gdl_hashtable_size (l->alleles) - 1;
			_gdl_locus_alleles_grow (l, (*va)->idx+1);
			l->_alleles[(*va)->idx] = (*va);
		}
		else if (owner)
		{
			gdl_entity_free (*va);
			*va = a;
		}
	}
}

gdl_allele *
gdl_locus_add_recessive (gdl_locus * vl)
{
	if (vl == 0)
		return;
	else
	{
		_gdl_locus   * l = (_gdl_locus *) vl->state;
		gdl_allele * a   = (gdl_allele *) gdl_hashtable_lookup (l->alleles, gdl_allele_recessive->name);
		
		if (a == NULL)
		{
			a = gdl_entity_clone (gdl_allele_recessive);
			gdl_hashtable_add (l->alleles, a->name, a, 1);
			a->idx = gdl_hashtable_size (l->alleles) - 1;
			_gdl_locus_alleles_grow (l, a->idx+1);
			l->_alleles[a->idx] = a;
		}
		
		return a;
	}
}

void
gdl_locus_add_genotype (gdl_locus * vl, gdl_genotype ** vg, size_t owner)
{
	if (vl == 0 || vg == 0)
		return;
	else
	{
		_gdl_locus   * l = (_gdl_locus *) vl->state;
		gdl_genotype * g = (gdl_genotype *) gdl_hashtable_lookup (l->genotypes, (*vg)->name);
		
		if (g == NULL)
		{
			gdl_list_itr * itr;
			
			gdl_hashtable_add (l->genotypes, (*vg)->name, *vg, owner);
			
			(*vg)->idx = gdl_hashtable_size (l->genotypes) - 1;
			_gdl_locus_genotypes_grow (l, (*vg)->idx+1);
			l->_genotypes[(*vg)->idx] = *vg;
			
			itr = gdl_genotype_allele_iterator (*vg);
			do
			{
				gdl_allele * va 
				   = (gdl_allele *) gdl_list_iterator_value (itr);
				
				if (va->type != GDL_ALLELE_MISSING
				    && va->type != GDL_ALLELE_RECESSIVE)
				{    
					gdl_allele * cva = gdl_entity_clone (va);
					gdl_locus_add_allele (vl, &cva, 1);
					va->idx = cva->idx;
				}
			}
			while (gdl_list_iterator_next (itr));
			gdl_list_iterator_free (itr);	
		}
		else if (owner)
		{
			gdl_entity_free (*vg);
			*vg = g;
		}
	}
}

gdl_allele *
gdl_locus_get_allele (const gdl_locus * vl, size_t idx)
{
	return ((_gdl_locus *)vl->state)->_alleles[idx];
}

gdl_genotype *
gdl_locus_get_genotype (const gdl_locus * vl, size_t idx)
{
	return ((_gdl_locus *)vl->state)->_genotypes[idx];
}

gdl_allele *
gdl_locus_search_allele (const gdl_locus * vl, const gdl_string * name)
{
	return 
		(gdl_allele * ) gdl_hashtable_lookup (((_gdl_locus *)vl->state)->alleles, name);
}

gdl_genotype *
gdl_locus_search_genotype (const gdl_locus * vl, const gdl_string * name)
{
	return 
		(gdl_genotype *) gdl_hashtable_lookup (((_gdl_locus *)vl->state)->genotypes, name);	
}

int
gdl_locus_fprintf (FILE * stream, const gdl_locus * l)
{
	if (stream == 0 || l == 0)
		return (-1);
	else
	{	
		gdl_hashtable_itr * itr;
		_gdl_locus * vl = (_gdl_locus *) l->state;
		
		fprintf (stream, "LOCUS %s %d\n", l->name, l->idx);
		
		if (gdl_hashtable_size (vl->alleles))
		{
			itr = gdl_hashtable_iterator (vl->alleles);
		
			do
			{
				gdl_allele * a = (gdl_allele *) gdl_hashtable_iterator_value (itr);
				if (a->type != GDL_ALLELE_RECESSIVE)
				{
					fprintf (stream, "  ALLELE %s %d\n", a->name, a->idx);
				}
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
			
		if (gdl_hashtable_size (vl->genotypes))
		{
			itr = gdl_hashtable_iterator (vl->genotypes);
			
			do
			{
				gdl_genotype * g = (gdl_genotype *) gdl_hashtable_iterator_value (itr);
				fprintf (stream, "  GENOTYPE %s %d\n", g->name, g->idx);
			}
			while (gdl_hashtable_iterator_next (itr));
		
			gdl_hashtable_iterator_free (itr);
		}
		
	}
}

#include "resolve.c"

static const gdl_locus_mode  _codominant = codominant;
static const gdl_locus_mode  _dominant   = dominant;
const gdl_locus_mode * gdl_locus_codominant = &_codominant;
const gdl_locus_mode * gdl_locus_dominant   = &_dominant;

static const gdl_entity_type _gdl_locus_type =
{
	sizeof(_gdl_locus),
	"L",
	&_gdl_locus_alloc,
	&_gdl_locus_free,
	&_gdl_locus_compare,
	&_gdl_locus_copy,
	&_gdl_locus_fread,
	&_gdl_locus_fwrite
};

const gdl_entity_type * GDL_LOCUS = &_gdl_locus_type;

