/*  
 * 	entity/resolve.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

static void
_push_genotype (gdl_list * gbuf,
                   gdl_allele_ptr ** alleles,
                   size_t * na,
                   size_t p,
                   size_t * path)
{
	size_t i;
	gdl_genotype * geno
	   = gdl_genotype_alloc ();
	for (i = 0; i < p; i++)
	{
		gdl_genotype_add (geno,
		                  alleles[i][path[i]],
		                  0);
	}
	gdl_list_push_front (gbuf, geno, 0);
}

static void
_path_in_alleles ( gdl_list * gbuf,
                   gdl_allele_ptr ** alleles,
                   size_t * na,
                   size_t p,
                   size_t * path,                   
                   size_t j,
                   size_t jj)
{
	size_t k;
	
	k = (j == jj) ? 1 : 0;
	
	for (;k < na[j]; k++)
	{
		path[j] = k;
		if (j == p - 1)
			_push_genotype (gbuf, alleles, na, p, path);
		if (j < p - 1)
			_path_in_alleles (gbuf, alleles, na, p, path, j+1, jj);
	}
	path[j]=0;
}

 
static gdl_genotype **
_build_genotypes (gdl_allele_ptr ** alleles,
                     size_t * na,
                     size_t p,
                     size_t * ng)
{
	size_t j, jj;
	gdl_list_itr * itr;
	gdl_genotype ** genotypes;
	gdl_list * gbuf = gdl_list_alloc (gdl_entity_interface);
	size_t * path   = GDL_CALLOC (size_t, p);
	
	_push_genotype (gbuf, alleles, na, p, path);
	for (jj = p; jj > 0; jj--)
	{
		j = jj - 1;
		_path_in_alleles (gbuf, alleles, na, p, path, j, j);
	}
	
	GDL_FREE (path);
	
	*ng       = gdl_list_size (gbuf);
	genotypes = GDL_CALLOC (gdl_genotype *, *ng);
	
	itr = gdl_list_iterator_front (gbuf);
	j   = 0;
	do
	{
		genotypes[j] 
		    = (gdl_genotype *) gdl_list_iterator_value (itr);
		j++;   
	}
	while (gdl_list_iterator_next (itr));
	gdl_list_iterator_free (itr);
	
	gdl_list_free (gbuf);
	
	return genotypes;
}

static gdl_genotype **
_resolve_genotype (_gdl_locus * loc, size_t i, gdl_allele ** alleles, size_t * ng)
{
	size_t j, k, p;
	size_t * na;
	gdl_allele_ptr ** al;
	gdl_genotype   ** genotypes;
	gdl_list_itr * itr;
	
	p = gdl_genotype_size (loc->_genotypes[i]);
	
	al      = GDL_CALLOC (gdl_allele_ptr *, p);
	na      = GDL_CALLOC (size_t, p);
		   
	itr = gdl_genotype_allele_iterator (loc->_genotypes[i]);
	j   = 0;
	
	do
	{
		gdl_allele * allele
                 = (gdl_allele *) gdl_list_iterator_value (itr);
        if (allele->type == GDL_ALLELE_MISSING
           || allele->type == GDL_ALLELE_RECESSIVE)
        {
        	na[j] = loc->_na;
        	al[j] = GDL_CALLOC (gdl_allele *, na[j]);
        	for (k = 0; k < na[j]; k++)
        	{
        		al[j][k] = alleles[k];
        	}
        }
        else
        {
        	na[j]    = 1;
        	al[j]    = GDL_CALLOC (gdl_allele *, na[j]);
        	al[j][0] = alleles[allele->idx];
        }
        j++;
	}
	while (gdl_list_iterator_next (itr));
		 
	gdl_list_iterator_free (itr);
	
	genotypes = _build_genotypes (al, na, p, ng);
	
	for (j = 0; j < p; j++)
	{
		GDL_FREE (al[j]);
	}
	GDL_FREE (al);
	GDL_FREE (na);
	
	return genotypes;
}

gdl_locus *
gdl_locus_resolve_genotype (const gdl_locus * vl)
{
	size_t i, j, ng;
	_gdl_locus * oloc;
	_gdl_locus * loc  = (_gdl_locus *)vl->state;
	gdl_locus  * out  = (gdl_locus  *)gdl_entity_clone (vl);
	gdl_genotype ** genotypes;
	
	// Clean out locus state
	_gdl_locus_free  (out->state);
	_gdl_locus_alloc (out->state);
	
	oloc = (_gdl_locus *)out->state;
		
	for (i = 0; i < loc->_na; i++)	
	{
		gdl_allele * a = gdl_entity_clone (loc->_alleles[i]);
		gdl_locus_add_allele (out, &a, 1);
	}
		
	for (i = 0; i < loc->_ng; i++)
	{
		genotypes = _resolve_genotype (loc, i, oloc->_alleles, &ng);
		for (j = 0; j < ng; j++)
		{
			gdl_locus_add_genotype (out, genotypes + j, 1);
		}		
		GDL_FREE (genotypes);
	}
	
	return out;
}

