/* 
 * gglm/result.c
 * 
 * Copyright (C) 2006 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_gmatrix.h>
#include <gdl/gdl_fmatrix.h>
#include <gdl/gdl_labels.h>
#include <gdl/gdl_gglm.h>
#include <gdl/gdl_gglm_result.h>

struct _gdl_gglm_result
{
	const gdl_gmatrix_type * type;
	size_t size;
	gdl_glabels * glabels;
	gdl_flabels * tlabels;
	gdl_flabels * olabels;
	gdl_gglm_static_model ** models;
};

gdl_gglm_result *
gdl_gglm_result_alloc (gdl_gglm_workspace * w)
{
	gdl_list * heap;
	gdl_gglm_result * r;
	
	if (!w)
	{
		GDL_ERROR_VAL ("The workspace is udenfined", GDL_EINVAL, 0);	
	}
	
	r = GDL_CALLOC (gdl_gglm_result, 1);
	
	r->type = gdl_gglm_workspace_gcode (w);
	
	heap = gdl_gglm_workspace_heap (w);
	
	r->size = gdl_list_size (heap);
	
	if (r->size)
	{
		size_t i;
		gdl_list_itr * itr;
		
		r->models = GDL_MALLOC (gdl_gglm_static_model *, r->size);
		
		i   = 0;
		itr = gdl_list_iterator_front (heap);
		do
		{
		   r->models[i] = (gdl_gglm_static_model *) gdl_list_iterator_value (itr);
		   i++;
		}
		while (gdl_list_iterator_next (itr));
		gdl_list_iterator_free (itr);
		
		gdl_list_clear (heap);
	}
	
	r->glabels = gdl_glabels_wrapper_alloc (gdl_gglm_workspace_gdata (w), gdl_false, gdl_false);
 	r->tlabels = gdl_flabels_wrapper_alloc (gdl_gglm_workspace_tdata (w), gdl_false);
 	
 	if (gdl_gglm_workspace_odata (w))
 	{
 		r->olabels = gdl_flabels_wrapper_alloc (gdl_gglm_workspace_odata (w), gdl_false);
 	}
	
	return r;
}

void
gdl_gglm_result_free (gdl_gglm_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
		{
			gdl_gglm_static_model_free (r->models[i]);	
		}
		GDL_FREE (r->models);
		gdl_glabels_free (r->glabels);
		gdl_flabels_free (r->tlabels);
		gdl_flabels_free (r->olabels);
		GDL_FREE (r);
	}
}

size_t
gdl_gglm_result_size (const gdl_gglm_result * r)
{
	return r->size;	
}

gdl_gglm_result *
gdl_gglm_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_boolean has;
		size_t i;
		gdl_gglm_result * r;
		
		r = GDL_CALLOC (gdl_gglm_result, 1);
		
		r->type = gdl_gmatrix_type_fread (stream);
		GDL_FREAD_STATUS (r->type!=0, 1);
		status = fread (&(r->size), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (r->size)
		{
			r->models = GDL_MALLOC (gdl_gglm_static_model *, r->size);
			for (i = 0; i < r->size; i++)
			{
				r->models[i] = gdl_gglm_static_model_fread (stream);
				GDL_FREAD_STATUS (r->models[i]!=0, 1);
			}
			r->glabels = gdl_glabels_fread (stream);
			GDL_FREAD_STATUS (r->glabels!=0, 1);
			r->tlabels = gdl_flabels_fread (stream);
			GDL_FREAD_STATUS (r->tlabels!=0, 1);
			status = fread (&has, sizeof (gdl_boolean), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			if (has)
			{
				r->olabels = gdl_flabels_fread (stream);
				GDL_FREAD_STATUS (r->olabels!=0, 1);
			}
		}
		
		return r;
	}
		
	return NULL;
}

int
gdl_gglm_result_fwrite (FILE * stream, const gdl_gglm_result * r)
{
	if (stream && r)
	{	
		int status;
		gdl_boolean has, not_has;
		size_t i;
		
		has     = gdl_true;
		not_has = gdl_false;
		
		status = gdl_gmatrix_type_fwrite (stream, r->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = fwrite (&(r->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (r->size)
		{
			for (i = 0; i < r->size; i++)
			{
				status = gdl_gglm_static_model_fwrite (stream, r->models[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
			status = gdl_glabels_fwrite (stream, r->glabels);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			status = gdl_flabels_fwrite (stream, r->tlabels);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			if (r->olabels)
			{
				status = fwrite (&has, sizeof (gdl_boolean), 1, stream);
				GDL_FWRITE_STATUS (status, 1);
				status = gdl_flabels_fwrite (stream, r->olabels);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
			else
			{
				status = fwrite (&not_has, sizeof (gdl_boolean), 1, stream);
				GDL_FWRITE_STATUS (status, 1);
			}
		}
		else
		{
			status = fwrite (&not_has, sizeof (gdl_boolean), 1, stream);
			GDL_FWRITE_STATUS (status, 1);	
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

#define PRINT_COEF(a, b)(fprintf (stream, "\t%.3f\t(+/- %.3f)\n", (a), (b)))

int
gdl_gglm_result_fprintf (FILE * stream, const gdl_gglm_result * r)
{
	if (stream && r)
	{	
		size_t i, j, k, l, nv;
		double x;
		
		fprintf (stream, "GGLM Result\n");
		fprintf (stream, "\tsize:  %d\n", r->size);
		fprintf (stream, "\tgcode: %s\n", r->type->acronym);
		fprintf (stream, "--------------------------------------------\n\n");
		
		for (i = 0; i < r->size; i++)
		{
			gdl_gglm_static_model * model = r->models[i];
			const gdl_string * name, * level;
			
			fprintf (stream, "--\n");
			fprintf (stream, "%d", i+1);
			name = gdl_flabels_factor (r->tlabels, model->trait);
		   fprintf (stream,"  [ %s ] ~", name);
			for (j = 0; j < model->no; j++)
			{
				name = gdl_flabels_factor (r->olabels, model->other[j]);
	  			fprintf (stream," [ %s ]", name);
	  			if (j < model->no-1 || model->nl) fprintf (stream," +");
			}
			for (j = 0; j < model->nl; j++)
			{
				name = gdl_glabels_locus (r->glabels, model->locus[j]);
	  			fprintf (stream," [ %s ]", name);
	  			if (j && j < model->no-1) fprintf (stream," +");
			}
			fprintf (stream, "\n\n");
			fprintf (stream, "   Number of individuals      %d\n", model->N);
			fprintf (stream, "   Number of variables        %d\n", model->M);
			fprintf (stream, "   Degrees of freedom (H1)    %d\n", model->df);
			fprintf (stream, "   Degrees of freedom (H1:H0) %d\n", model->df1);
			fprintf (stream, "   R-square (H1)              %1.3f\n", model->rsq);
			fprintf (stream, "   R-square (H1:H0)           %1.3f\n", model->rsq1);
			fprintf (stream, "   F-stat (H1:H0)             %.2f\n", model->fstat);
			fprintf (stream, "   P-value (H1:H0)            %e ", model->pval);
			j = 0;
			x = model->pval;
			while (x <= 0.05)
			{
				if (!j)
				{
					fprintf (stream, "(");
				}
				j++;
				x *= 100;
				fprintf (stream, "*");
				if (j==3 || x > 0.05) 
				{
					fprintf (stream, ")");
					break;
				}
			}
			fprintf (stream, "\n\n");
			fprintf (stream, "   Intercept\t-");
			PRINT_COEF(gdl_vector_get (model->coeff, 0), gdl_vector_get (model->sd, 0));
			k = 1;
			for (j = 0; j < model->no; j++)
			{
				name = gdl_flabels_factor (r->olabels, model->other[j]);
	  			nv   = gdl_flabels_factor_level_size (r->olabels, model->other[j]); 
				if (nv)
				{
					for (l = 0; l < nv; l++, k++)
					{
						level = gdl_flabels_level (r->olabels, model->other[j], l);
						fprintf (stream, "   %s\t%s", name, level);
						PRINT_COEF (gdl_vector_get (model->coeff, k), gdl_vector_get (model->sd, k));
					}
				}
				else
				{
					fprintf (stream, "   %s\t-", name);
					PRINT_COEF (gdl_vector_get (model->coeff, k), gdl_vector_get (model->sd, k));
					k++;
				}
			}
			for (j = 0; j < model->nl; j++)
			{
				name = gdl_glabels_locus (r->glabels, model->locus[j]);
				if (r->type == gdl_gmatrix_allele)
				{
		  			nv   = gdl_glabels_locus_allele_size (r->glabels, model->locus[j]); 
					for (l = 0; l < nv; l++, k++)
					{
						level = gdl_glabels_allele (r->glabels, model->locus[j], l);
						fprintf (stream, "   %s\t%s", name, level);
						PRINT_COEF (gdl_vector_get (model->coeff, k), gdl_vector_get (model->sd, k));
					}
				}
				else if (r->type == gdl_gmatrix_genotype)
				{
					nv   = gdl_glabels_locus_genotype_size (r->glabels, model->locus[j]); 
					for (l = 0; l < nv; l++, k++)
					{
						level = gdl_glabels_genotype (r->glabels, model->locus[j], l);
						fprintf (stream, "   %s\t%s", name, level);
						PRINT_COEF (gdl_vector_get (model->coeff, k), gdl_vector_get (model->sd, k));
					}
				}
			}
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

#undef PRINT_COEF


