/*  
 *  gmap/gmap.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gmap.h>

struct _gdl_gmap
{
	size_t size;
	gdl_hashtable  * table;
	gdl_genome     ** _genomes;
};

static int
_gdl_gmap_grow (gdl_gmap * g, size_t size)
{
	size_t i;
	
	if (g->size)
	{
		gdl_genome ** ng;
		
		ng = GDL_CALLOC (gdl_genome *, size);
		memcpy (ng, g->_genomes, sizeof(gdl_genome *)*size);
		GDL_FREE (g->_genomes);
		g->_genomes = ng;
	}
	else
	{
		g->table    = gdl_hashtable_alloc (gdl_entity_interface, size);
		g->_genomes = GDL_CALLOC (gdl_genome *, size);
	}
	
	g->size = size;
	
	return GDL_SUCCESS;
}


gdl_gmap *
gdl_gmap_alloc (void)
{
	gdl_gmap * g = GDL_CALLOC (gdl_gmap, 1);
	return g;	
}

void
gdl_gmap_free (gdl_gmap * g)
{
	if (g && g->size)
	{
		gdl_hashtable_free (g->table);
		GDL_FREE (g->_genomes);
		GDL_FREE (g);
	}	
}

gdl_gmap *
gdl_gmap_clone (const gdl_gmap * g)
{
	size_t i;
	gdl_gmap * c;
	
	if (g==0)
	{
		return NULL;	
	}
	
	c = gdl_gmap_alloc();
	
	_gdl_gmap_grow (c, g->size);
	
	for (i = 0; i < c->size; i++)
	{
		c->_genomes[i]	= gdl_entity_clone (g->_genomes[i]);
		gdl_hashtable_add (c->table, gdl_entity_get_name (c->_genomes[i]), c->_genomes[i], 1);
	}
	
	return c;
}

int
gdl_gmap_add (gdl_gmap * gmap, gdl_genome * genome)
{
	if (gmap == 0 || genome == 0)
	{
		return (-1);
	}
	else
	{
		int i;
		i = gmap->size;
		_gdl_gmap_grow (gmap, gmap->size+1);
		gmap->_genomes[i] = genome;
		gdl_hashtable_add (gmap->table, genome->name, genome, 1);
		return (i);
	}
}

size_t
gdl_gmap_size (const gdl_gmap * gmap)
{
	return gmap->size;
}

size_t
gdl_gmap_genome_size (const gdl_gmap * gmap, size_t i)
{
	return gdl_genome_size (gmap->_genomes[i]);
}

size_t
gdl_gmap_chromosome_size (const gdl_gmap * gmap, size_t i, size_t j)
{
	return gdl_chromosome_size (gdl_genome_get (gmap->_genomes[i], j));
}

gdl_genome *
gdl_gmap_search (const gdl_gmap * gmap, const gdl_string * name)
{
	if (gmap == 0 || name == 0)
	{
		return NULL;
	}
	else
	{
		return (gdl_genome *) gdl_hashtable_lookup (gmap->table, name);
	}
}

gdl_genome *
gdl_gmap_get (const gdl_gmap * gmap, size_t i)
{
	return gmap->_genomes[i];
}

gdl_chromosome *
gdl_gmap_get_chromosome (const gdl_gmap * g, size_t i, size_t j)
{
	return gdl_genome_get (g->_genomes[i], j);
}

gdl_locus *
gdl_gmap_get_locus (const gdl_gmap * g, size_t i, size_t j, size_t l)
{
	gdl_chromosome * c = gdl_gmap_get_chromosome (g, i, j);
	return gdl_chromosome_get (c, l);
}

gdl_gdistance *
gdl_gmap_get_distance (const gdl_gmap * g, size_t i, size_t j, size_t from, size_t to)
{
	return gdl_chromosome_get_distance (gdl_gmap_get_chromosome (g, i, j), from, to);
}

gdl_gmap *
gdl_gmap_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t size, status;
		gdl_gmap * g;
		
		g = gdl_gmap_alloc ();
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (size)
		{
			size_t i;
			
			_gdl_gmap_grow (g, size);
			
			for (i = 0; i < g->size; i++)
			{
				g->_genomes[i] = gdl_entity_fread (stream);
				GDL_FWRITE_STATUS (g->_genomes[i] == 0, 0);
				gdl_hashtable_add (g->table, g->_genomes[i]->name, g->_genomes[i], 1);
			}
		}
		
		return g;
	}
}

int
gdl_gmap_fwrite (FILE * stream, const gdl_gmap * g)
{
	if (stream == 0 || g == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		
		status = fwrite (&g->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (g->size)
		{
			size_t i;
			
			for (i = 0; i < g->size; i++)
			{
				status = gdl_entity_fwrite (stream, g->_genomes[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}		  
		}
				
		return GDL_SUCCESS;	
	}
}

gdl_locus **
gdl_gmap_loci (const gdl_gmap * g, size_t * nl)
{
	if (g == 0)
		return NULL;
	else
	{
		gdl_locus ** loci;
		size_t i, j, k, l, nc, m;
		
		*nl = 0;
		
		for (i = 0; i < g->size; i++)
		{
			nc = gdl_genome_size (g->_genomes[i]);
			for (j = 0; j < nc; j++)
			{
				gdl_chromosome * c = gdl_genome_get (g->_genomes[i], j);
				(*nl) += gdl_chromosome_size (c);
			}
		}
		
		if (*nl)
		{
			loci = GDL_CALLOC (gdl_locus *, *nl);
			
			for (l = i = 0; i < g->size; i++)
			{
				nc = gdl_genome_size (g->_genomes[i]);
				for (j = 0; j < nc; j++)
				{
					gdl_chromosome * c = gdl_genome_get (g->_genomes[i], j);
					m = gdl_chromosome_size (c);
					for (k = 0; k < m; k++)
					{
						loci[l++] = gdl_chromosome_get (c, k);	
					}
				}
			}			
			
			return loci;
		}
		else
		{
			return NULL;	
		}
	}
}

size_t
gdl_gmap_chrom_number (const gdl_gmap * g)
{
	if (g)
	{
		size_t i, tnc = 0;
		
		for (i = 0; i < g->size; i++)
		{
			tnc += gdl_genome_size (g->_genomes[i]);
		}
		
		return tnc;	
	}
	else
	{
		return 0;
	}
}

size_t * 
gdl_gmap_genome_sizes (const gdl_gmap * g)
{
	size_t i;
	size_t * n;
	
	n = GDL_CALLOC (size_t, g->size);
	
	for (i = 0; i < g->size; i++)
	{
		n[i] = gdl_genome_size (g->_genomes[i]);
	}
	
	return n;
}

