/*  
 *  gmatrix/genotype.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmatrix.h>

static int
gdl_gmatrix_genotype_alloc (gdl_gmatrix * gm, const gdl_gview * gview, const gdl_mask * mask, const gdl_clustering * clustering, gdl_boolean last_allele)
{
	size_t i, j, k, l, nr, nl, np, nn;
	gdl_locus * locus;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	np = gdl_gview_ploidy (gview);
	nr = nn = GDL_GVIEW_ACCESSION_SIZE (gview, mask);
	if (clustering)
	{
		nr = gdl_clustering_nclust (clustering);
	}
	nl = GDL_GVIEW_LOCUS_SIZE (gview, mask);
	
	gm->size1 = nr;
	gm->nl  = nl;
	gm->rw  = gdl_vector_alloc (nr);
	gm->nc  = GDL_MALLOC (size_t, nl);
	gm->tnc = GDL_MALLOC (size_t, nl);
	gm->tnc[0] = 0;
	
	for (i = 0; i < nl; i++)
	{
		locus     = GDL_GVIEW_GET_LOCUS (gview, mask, i);
		gm->nc[i] = gdl_locus_genotype (locus);
		if (gm->nc[i] <= 1)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (!last_allele)
		{
			(gm->nc[i])--;
		}
		if (i)
		{
			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
		}
	}
	
	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
	gm->data  = gdl_matrix_calloc (gm->size1, gm->size2);
	
	gbuf = GDL_GVIEW_GET_NEW (gview, mask);
	
	for (i = 0; i < nr; i++)
	{
		t = 1.0;
		if (clustering)
		{
			t = (double) gdl_clustering_clust_size (clustering, i);
		}
		t /= (double) nn;
		
		gdl_vector_set (gm->rw, i, t);
		
		for (j = 0; j < nl; j++)
		{
			if (clustering)
			{
				GDL_GVIEW_GET_GENOTYPE (gview, mask, gdl_clustering_clust_idx (clustering, i), j, gbuf);
			}
			else
			{
				GDL_GVIEW_GET_GENOTYPE (gview, mask, i, j, gbuf);
			}
			gv = gdl_gvalues_get_gvalues (gbuf);
			gdl_gmatrix_locus_set (gm, i, j, gv);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static int
gdl_gmatrix_genotype_wrapper_alloc (gdl_gmatrix * gm, const gdl_gview_wrapper * gwrap, gdl_boolean accession_cluster, gdl_boolean last_allele)
{
	size_t i, j, k, l, nr, nl, np, nn;
	gdl_locus * locus;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	np = gdl_gview_wrapper_ploidy (gwrap);
	nr = nn = gdl_gview_wrapper_accession_size (gwrap);
	if (accession_cluster)
	{
		nr = gdl_gview_wrapper_accession_size_c (gwrap);
	}
	nl = gdl_gview_wrapper_locus_size (gwrap);
	
	gm->size1 = nr;
	gm->nl  = nl;
	gm->rw  = gdl_vector_alloc (nr);
	gm->nc  = GDL_MALLOC (size_t, nl);
	gm->tnc = GDL_MALLOC (size_t, nl);
	gm->tnc[0] = 0;
	
	for (i = 0; i < nl; i++)
	{
		locus     = gdl_gview_wrapper_get_locus (gwrap, i);
		gm->nc[i] = gdl_locus_genotype (locus);
		if (gm->nc[i] <= 1)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (!last_allele)
		{
			(gm->nc[i])--;
		}
		if (i)
		{
			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
		}
	}
	
	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
	gm->data  = gdl_matrix_calloc (gm->size1, gm->size2);
	
	gbuf = gdl_gview_wrapper_get_new (gwrap);
	
	for (i = 0; i < nr; i++)
	{
		t = 1.0;
		if (accession_cluster)
		{
			t = (double) gdl_gview_wrapper_accession_mult_c (gwrap, i);
		}
		t /= (double) nn;
		
		gdl_vector_set (gm->rw, i, t);
		
		for (j = 0; j < nl; j++)
		{
			if (accession_cluster)
			{
				gdl_gview_wrapper_get_genotype_c (gwrap, i, j, gbuf);
			}
			else
			{
				gdl_gview_wrapper_get_genotype (gwrap, i, j, gbuf);
			}
			gv = gdl_gvalues_get_gvalues (gbuf);
			gdl_gmatrix_locus_set (gm, i, j, gv);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static int
gdl_gmatrix_genotype_wrapper_mask_alloc (gdl_gmatrix * gm, const gdl_gview_wrapper * gwrap, const gdl_mask * mask, gdl_boolean last_allele)
{
	size_t i, j, k, l, nr, nl, np, nn;
	gdl_locus * locus;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	np = gdl_gview_wrapper_ploidy (gwrap);
	
	nr = nn = gdl_mask_size (mask, GDL_ACCESSION);
	
	nl = gdl_mask_size (mask, GDL_LOCUS);
	
	gm->size1 = nr;
	gm->nl  = nl;
	gm->rw  = gdl_vector_alloc (nr);
	gm->nc  = GDL_MALLOC (size_t, nl);
	gm->tnc = GDL_MALLOC (size_t, nl);
	gm->tnc[0] = 0;
	
	for (i = 0; i < nl; i++)
	{
		locus     = gdl_gview_wrapper_get_locus (gwrap, gdl_mask_get_idx (mask, GDL_LOCUS, i));
		gm->nc[i] = gdl_locus_genotype (locus);
		if (gm->nc[i] <= 1)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (!last_allele)
		{
			(gm->nc[i])--;
		}
		if (i)
		{
			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
		}
	}
	
	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
	gm->data  = gdl_matrix_calloc (gm->size1, gm->size2);
	
	gbuf = gdl_gview_wrapper_get_new (gwrap);
	
	for (i = 0; i < nr; i++)
	{
		gdl_vector_set (gm->rw, i, 1.0/(double)nr);
		
		for (j = 0; j < nl; j++)
		{
			gdl_gview_wrapper_get_genotype (gwrap,
				                              gdl_mask_get_idx (mask, GDL_ACCESSION, i),
				                              gdl_mask_get_idx (mask, GDL_LOCUS, j),
				                              gbuf);
			gv = gdl_gvalues_get_gvalues (gbuf);
			gdl_gmatrix_locus_set (gm, i, j, gv);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static const gdl_gmatrix_type _gdl_gmatrix_genotype =
{
	"gdl_gmatrix_genotype",
	"genotype",
	&gdl_gmatrix_genotype_alloc,
	&gdl_gmatrix_genotype_wrapper_alloc,
	&gdl_gmatrix_genotype_wrapper_mask_alloc
};

const gdl_gmatrix_type * gdl_gmatrix_genotype = &_gdl_gmatrix_genotype;
