/*  
 *  pstruct/plot.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_fmatrix.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_graphics.h>
#include <gdl/gdl_plot.h>
#include <gdl/gdl_gpca.h>
#include <gdl/gdl_gpca_result.h>

static const double GDL_GPCA_PLOT_BIPLOT_WIDTH    = 800;
static const double GDL_GPCA_PLOT_BIPLOT_HEIGHT   = 800;
static const double GDL_GPCA_PLOT_BIPLOT_MARGIN   = 75;
static const double GDL_GPCA_PLOT_EXTRA_STROKE    = 0.5;
static const int    GDL_GPCA_PLOT_EXTRA_SEGMENT   = 1;

typedef struct
{
	double height;
	double width;
	const gdl_gpca_result * result;
} gdl_gpca_plot_t;

static int
gdl_gpca_plot_extra_factor (gdl_plot_parameters * par, const gdl_string * factor, const gdl_string * file)
{
	size_t i;
	FILE * stream;
	const gdl_view * v;
	const gdl_fview * f;
	gdl_factor * fc;
	gdl_fview_wrapper * fw;
	gdl_mask          * m;
	gdl_plot_parameter * p;
	gdl_rgb_color_vector * palette;
		
	stream = gdl_fileopen (file, "r");
	
	if (!stream)
	{
		return GDL_EINVAL; 	
	}
	
	v = gdl_view_fread (stream);
	
	gdl_fileclose (file, stream);
	
	if (!v)
	{
		return GDL_EINVAL;
	}
	
	f = gdl_view_get_oview (v);
	
	if (!f)
	{
		return GDL_EINVAL;
	}
	
	fc = gdl_fview_search_factor (f, factor);
	
	if (!fc || gdl_factor_get_type (fc) == gdl_factor_continuous)
	{
		return GDL_EINVAL;
	}
	
	m = gdl_mask_alloc ();
	gdl_mask_add (m, fc);
	
	fw = gdl_fview_wrapper_alloc (gdl_fview_wrapper_global, f, m);
	
	p = gdl_plot_parameter_alloc (gdl_plot_parameter_extra, 1);
	gdl_plot_parameter_set (p, 0, fw);
	gdl_plot_parameters_set (par, "extra-clustering", p);
	
	// add the colors
	p = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, gdl_factor_size (fc));
	palette = gdl_rgb_color_palette (gdl_factor_size (fc));
	for (i = 0; i < gdl_factor_size (fc); i++)
	{
		gdl_plot_parameter_set_copy (p, i, palette->colors[i]);
	}
	gdl_plot_parameters_set (par, "extra-colors", p);
	gdl_rgb_color_vector_free (palette);
	
	return GDL_SUCCESS;
}

static int
gdl_gpca_plot_extra (const gdl_string * option, gdl_plot_parameters * params)
{
	gdl_hashtable_itr * itr;
	gdl_hashtable * args = gdl_plot_parse_extra (option);
	
	if (!args)
	{
		GDL_ERROR_VAL ("Invalid extra argument", GDL_EINVAL, GDL_EINVAL);
	}
	else if (gdl_hashtable_size (args))
	{
		const gdl_string * factor, * level;
		gdl_plot_parameter * par;
		
		factor = gdl_hashtable_lookup (args, "factor");
		if (factor)
		{
			const gdl_string * file;
			
			file = gdl_hashtable_lookup (args, "data");
			if (!file)
			{
				gdl_hashtable_free (args);
				GDL_ERROR_VAL ("Factor defined but no data file defined", GDL_EINVAL, GDL_EINVAL);
			}
			if (gdl_gpca_plot_extra_factor (params, factor, file) != GDL_SUCCESS)
			{
				GDL_ERROR_VAL ("Unable to load factor from data file", GDL_EINVAL, GDL_EINVAL);
			}
		}
		level = gdl_hashtable_lookup (args, "level");
		if (level)
		{
			gdl_plot_parameter * p;
			size_t * l = GDL_MALLOC (size_t, 1);
			
			(*l) = atoi(level);
			
			p = gdl_plot_parameter_alloc (gdl_plot_parameter_extra, 1);
			gdl_plot_parameter_set (p, 0, l);
			gdl_plot_parameters_set (params, "extra-level", p);
		}
		par = gdl_plot_parameters_get (params, "extra-stroke");
		if (!par)
		{
			par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
			gdl_plot_parameter_set_copy (par, 0, &GDL_GPCA_PLOT_EXTRA_STROKE);
			gdl_plot_parameters_set (params, "extra-stroke", par);
		}
		par = gdl_plot_parameters_get (params, "extra-segment");
		if (!par)
		{
			par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
			gdl_plot_parameter_set_copy (par, 0, &GDL_GPCA_PLOT_EXTRA_SEGMENT);
			gdl_plot_parameters_set (params, "extra-segment", par);
		}
	}
	
	gdl_hashtable_free (args);
	
	return GDL_SUCCESS;
}

static void
gdl_gpca_plot_default_parameters (gdl_gpca_plot_t * plot, gdl_plot_parameters * params)
{
	gdl_plot_parameter * par;

	par = gdl_plot_parameters_get (params, "biplot-margin");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_GPCA_PLOT_BIPLOT_MARGIN);
		gdl_plot_parameters_set (params, "biplot-margin", par);
	}
	par = gdl_plot_parameters_get (params, "biplot-height");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_GPCA_PLOT_BIPLOT_HEIGHT);
		gdl_plot_parameters_set (params, "biplot-height", par);
	}
	par = gdl_plot_parameters_get (params, "biplot-width");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_GPCA_PLOT_BIPLOT_WIDTH);
		gdl_plot_parameters_set (params, "biplot-width", par);
	}
	par = gdl_plot_parameters_get (params, "biplot-pca");
	if (!par)
	{
		int pca;
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 2);
		pca = 0;
		gdl_plot_parameter_set_copy (par, 0, &pca);
		pca = 1;
		gdl_plot_parameter_set_copy (par, 1, &pca);
		gdl_plot_parameters_set (params, "biplot-pca", par);
	}
	par = gdl_plot_parameters_get (params, "label-color");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, 1);
		gdl_plot_parameter_set (par, 0, gdl_rgb_color_get_black());
		gdl_plot_parameters_set (params, "label-color", par);
	}
//	par = gdl_plot_parameters_get (params, "pop-color");
//	if (!par || gdl_plot_parameter_size (par) < gdl_gpca_result_population_size (plot->result))
//	{
//		size_t i, nk;
//		gdl_rgb_color_vector * palette;
//		
//		nk      = gdl_gpca_result_population_size (plot->result);
//		palette = gdl_rgb_color_palette (nk);
//		par     = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, nk);
//		
//		for (i  = 0; i < nk; i++)
//		{
//			gdl_plot_parameter_set_copy (par, i, palette->colors[i]);
//		}
//		
//		gdl_plot_parameters_set (params, "pop-color", par);
//		
//		gdl_rgb_color_vector_free (palette);
//	}
//	par = gdl_plot_parameters_get (params, "label-color");
//	if (!par)
//	{
//		par = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, 1);
//		gdl_plot_parameter_set (par, 0, gdl_rgb_color_get_black ());
//		gdl_plot_parameters_set (params, "label-color", par);
//	}
	par = gdl_plot_parameters_get (params, "label-font");
	if (!par)
	{
		gdl_font * font = gdl_font_default ();
		font->size = 5;
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_font, 1);
		gdl_plot_parameter_set (par, 0, font);
		gdl_plot_parameters_set (params, "label-font", par);
	}
}

static int
gdl_gpca_plot_create (gdl_gpca_plot_t * p, gdl_plot_parameters * params)
{
	const gdl_gpca_result * r = p->result;
	gdl_plot_parameter * par;
	double * v, * m;
	
	par = gdl_plot_parameters_get (params, "biplot-margin");
	m   = (double *) gdl_plot_parameter_get (par, 0);
	par = gdl_plot_parameters_get (params, "biplot-height");
	v   = (double *) gdl_plot_parameter_get (par, 0);
	p->height = (*v) + 2*(*m);
	par = gdl_plot_parameters_get (params, "biplot-width");
	v   = (double *) gdl_plot_parameter_get (par, 0);
	p->width  = (*v) + 2*(*m);
}

static int
gdl_gpca_plot_alloc (void * plot, gdl_plot_parameters * par, const void * data, double * width, double * height)
{
	gdl_gpca_plot_t * p;
	
	p = (gdl_gpca_plot_t *) plot;
	
	p->result = (gdl_gpca_result *) data;
	
	gdl_gpca_plot_default_parameters (p, par);
	
	gdl_gpca_plot_create (p, par);
	
	*width  = p->width;
	*height = p->height;
	
	return GDL_SUCCESS;
}

static int
gdl_gpca_plot_free (void * plot)
{
	if (plot)
	{
		gdl_gpca_plot_t * p = (gdl_gpca_plot_t *) plot;
	}
}

static double
gdl_gpca_plot_extra_barycenter (const gdl_vector * x, const gdl_mask * mask, const gdl_matrix * m, size_t j)
{
	size_t i, ii;
	double g = 0, t = 0;
	
	for (i = 0; i < gdl_mask_size (mask, GDL_ACCESSION); i++)
	{
		ii = gdl_mask_get_idx (mask, GDL_ACCESSION, i);
		g += gdl_vector_get (x, ii) * gdl_matrix_get (m, i, j);
		t += gdl_matrix_get (m, i, j);
	}
	
	return g/t;
}

static gdl_matrix *
gdl_gpca_plot_extra_segment (const gdl_vector * x, const gdl_vector * y, const gdl_mask * mask, double xg, double yg, const gdl_matrix * m, size_t j)
{
	size_t i, ii;
	double xs, ys, w;
	gdl_matrix * seg;
	
	seg = gdl_matrix_alloc (x->size, 2);
	
	for (i = 0; i < gdl_mask_size (mask, GDL_ACCESSION); i++)
	{
		ii = gdl_mask_get_idx (mask, GDL_ACCESSION, i);
		xs = (xg - gdl_vector_get (x, ii));
		ys = (yg - gdl_vector_get (y, ii));
		w  = gdl_matrix_get (m, i, j);
		xs *= w;
		ys *= w;
		gdl_matrix_set (seg, i, 0, gdl_vector_get (x, ii) + xs);
   	gdl_matrix_set (seg, i, 1, gdl_vector_get (y, ii) + ys);
  }
	
	return seg;
}

static int
gdl_gpca_plot_draw_accession_extra (gdl_gpca_plot_t * plot, const gdl_plot_parameters * par, gdl_graphics2D * graph, const gdl_vector * pca1, const gdl_vector * pca2, const gdl_fview_wrapper * extra)
{
	int idx, * segment;
	size_t i, j, jj, fl, ll, na;
	double xg, yg, * stroke;
	gdl_mask * fmask, * pmask;
	gdl_factor  * fc;
	gdl_fmatrix * fm;
	gdl_matrix  * seg;
	gdl_rgb_color * clabel, * color;
	const gdl_plot_parameter * colors;
	const gdl_matrix  * m;
	const gdl_plot_parameter * level;
	const gdl_gpca_result * pca = plot->result;
	const gdl_glabels * labels  = gdl_gpca_result_get_labels (pca);
	
	na = gdl_fview_wrapper_accession_size (extra);
	// first, get the common accession between pca and extra
	fmask = gdl_mask_alloc ();
	pmask = gdl_mask_alloc ();
	
	fc = gdl_fview_wrapper_get_factor (extra, 0);
	
	gdl_mask_add (fmask, fc);
	
	for (i = 0; i < na; i++)
	{
		gdl_accession * a = gdl_fview_wrapper_get_accession (extra, i);
		idx = gdl_glabels_search_accession (labels, gdl_entity_get_name (a));
		if (idx >= 0)
		{
			gdl_mask_add_idx (fmask, GDL_ACCESSION, i);
			gdl_mask_add_idx (pmask, GDL_ACCESSION, idx);
		}
	}
	
	fm    = gdl_fmatrix_wrapper_mask_alloc (extra, fmask, gdl_true);
	m     = gdl_fmatrix_get_matrix (fm);
	
	level   = gdl_plot_parameters_get (par, "extra-level");
	colors  = gdl_plot_parameters_get (par, "extra-colors");
	clabel  = (gdl_rgb_color *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "label-color"), 0);
	segment = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "extra-segment"), 0);
	stroke  = (double *)  gdl_plot_parameter_get (gdl_plot_parameters_get (par, "extra-stroke"), 0);
	
	idx = -1;
	if (level)
	{
		int * l = (int *) gdl_plot_parameter_get (level, 0);
		idx = (*l);	
	}
	for (i = 0; i < gdl_factor_size (fc); i++)
	{
		xg  = gdl_gpca_plot_extra_barycenter (pca1, pmask, m, i);
		yg  = gdl_gpca_plot_extra_barycenter (pca2, pmask, m, i);
		
		if (*segment)
		{
			seg = gdl_gpca_plot_extra_segment (pca1, pca2, pmask, xg, yg, m, i);
		}
		
		gdl_graphics2D_set_color (graph, (gdl_rgb_color *) gdl_plot_parameter_get (colors, i));
		
		gdl_graphics2D_draw_text (graph, xg, yg, "O");	
		
		for (j = 0; j < gdl_mask_size (pmask, GDL_ACCESSION); j++)
		{
			gdl_vector_const_view view = gdl_matrix_const_row (m, j);
			
			jj = gdl_mask_get_idx (pmask, GDL_ACCESSION, j);
			
			if (gdl_vector_max_index (&(view.vector)) == i)
			{
				const gdl_string * name = gdl_glabels_accession (labels, jj);
				gdl_graphics2D_draw_text (graph, gdl_vector_get (pca1, jj), gdl_vector_get (pca2, jj), name);
			}
//			else if (idx >= 0)
//			{
//				const gdl_string * name = gdl_glabels_accession (labels, j);
//				color = gdl_graphics2D_get_color (graph);
//				gdl_graphics2D_set_color (graph, clabel);
//				gdl_graphics2D_draw_text (graph, gdl_vector_get (pca1, j), gdl_vector_get (pca2, j), name);
//				gdl_graphics2D_set_color (graph, color);
//			}
			if (*segment)
			{
				if ((idx>=0 && i==idx) || idx==-1)
				{
					gdl_graphics2D_set_stroke (graph, *stroke, 0, 0);
					gdl_graphics2D_draw_line (graph, gdl_vector_get (pca1, jj), gdl_vector_get (pca2, jj), gdl_matrix_get (seg, j, 0), gdl_matrix_get (seg, j, 1));
				}
			}
		}
		if (*segment)
		{
			gdl_matrix_free (seg);
		}
	}
	
	gdl_fmatrix_free (fm);
	gdl_mask_free (fmask);
	gdl_mask_free (pmask);
	
	return GDL_SUCCESS;
}

static int
gdl_gpca_plot_draw_extra (gdl_gpca_plot_t * plot, const gdl_plot_parameters * par, gdl_graphics2D * graph, const gdl_vector * pca1, const gdl_vector * pca2, const gdl_fview_wrapper * extra)
{
	const gdl_gpca_workspace_type * T;
	const gdl_gpca_result * r = plot->result;
	
	T = gdl_gpca_result_get_type (r);
	
	if (T == gdl_gpca_locus_allele_covariance
	    || T == gdl_gpca_locus_allele_correlation)
	{
		return gdl_gpca_plot_draw_accession_extra	(plot, par, graph, pca1, pca2, extra);
	}
	else if (T == gdl_gpca_accession_allele_covariance
	         || T == gdl_gpca_accession_allele_correlation
	         || T == gdl_gpca_accession_allele_euclidean_distance)
	{
	  return 0;
	}
}

static int
gdl_gpca_plot_draw (void * plot, const gdl_plot_parameters * par, gdl_graphics2D * graph)
{
	int a, * p1, * p2;
	size_t i, j, na, np;
	double x0, y0, x, y, max, min, * height, * width, * margin;
	gdl_string * name;
	gdl_rgb_color * c0, * c1, * black;
	gdl_font      * f0;
	gdl_plot_parameter * extra;
	gdl_plot_parameter * colors;
	
	gdl_gpca_plot_t * p       = (gdl_gpca_plot_t *) plot;
	const gdl_gpca_result * r = p->result;
	const gdl_matrix * pca;
	gdl_vector * pca1, * pca2;
	
	black = gdl_rgb_color_get_black ();
	
	c0 = gdl_graphics2D_get_color (graph);
	f0 = gdl_graphics2D_get_font (graph);
	
	height = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "biplot-height"), 0);
	width  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "biplot-width"), 0);
	margin = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "biplot-margin"), 0);
	p1     = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "biplot-pca"), 0);
	p2     = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "biplot-pca"), 1);
	c1     = (gdl_rgb_color *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "label-color"), 0);
	
	np = gdl_gpca_result_naxe (r);
	
	if (*p1 >= na || *p2 >= na)
	{
		GDL_ERROR_VAL ("PCA index is out of range\n", GDL_EINVAL, GDL_EINVAL);	
	}
	
	x0 = *margin;
	y0 = *margin;
	
	gdl_graphics2D_set_color (graph, black);
	gdl_graphics2D_set_stroke (graph, 2, 0, 0);
	gdl_graphics2D_draw_rect (graph, x0, y0, *width, *height);
	
	name = gdl_string_sprintf ("PCA%d", *p1 + 1);
	gdl_graphics2D_draw_text (graph, x0 + *width/2, y0 + *height + *margin/2, name);
	gdl_string_free (name);
	
	name = gdl_string_sprintf ("PCA%d", *p2 + 1);
	gdl_graphics2D_draw_text (graph, *margin/4, y0 + *height/2, name);
	gdl_string_free (name);
	
	pca = gdl_gpca_result_projection (r);
	gdl_vector_const_view view1 = gdl_matrix_const_column (pca, *p1);
	pca1 = gdl_vector_alloc (view1.vector.size);
	gdl_vector_memcpy (pca1, &(view1.vector));
	gdl_vector_const_view view2 = gdl_matrix_const_column (pca, *p2);
	pca2 = gdl_vector_alloc (view2.vector.size);
	gdl_vector_memcpy (pca2, &(view2.vector));
	
	max = gdl_vector_max (pca1);
	min = gdl_vector_min (pca1);
	
	gdl_vector_add_constant (pca1, -min);
	gdl_vector_scale (pca1, 0.9*(*width)/(max-min));
	gdl_vector_add_constant (pca1, x0 + 0.05*(*width));
	
	max = gdl_vector_max (pca2);
	min = gdl_vector_min (pca2);
	
	gdl_vector_add_constant (pca2, -max);
	gdl_vector_scale (pca2, -0.9*(*height)/(max-min));
	gdl_vector_add_constant (pca2, y0 + 0.05*(*height));
	
	extra = gdl_plot_parameters_get (par, "extra-clustering");
	if (extra)
	{
		const gdl_fview_wrapper * fdata = (gdl_fview_wrapper *) gdl_plot_parameter_get (extra, 0);
		gdl_gpca_plot_draw_extra (plot, par, graph, pca1, pca2, fdata);
	}
	else
	{
		gdl_graphics2D_set_color (graph, c1);
		for (i = 0; i < pca1->size; i++)
		{
			x = gdl_vector_get (pca1, i);
			y = gdl_vector_get (pca2, i);
			name = gdl_gpca_result_get_row_label (r, i);
			gdl_graphics2D_draw_text (graph, x, y, name);
			gdl_string_free (name);
		}	
	}
	
	gdl_vector_free (pca1);
	gdl_vector_free (pca2);
	gdl_rgb_color_free (black);
	
	gdl_graphics2D_set_color (graph, c0);
	gdl_graphics2D_set_font (graph, f0);
	
	return GDL_SUCCESS;
}

static const gdl_plot_type _gdl_gpca_result_plot =
{
	"gdl_gpca_result",
	sizeof (gdl_gpca_plot_t),
	&gdl_gpca_plot_alloc,
	&gdl_gpca_plot_free,
	&gdl_gpca_plot_draw,
	&gdl_gpca_plot_extra
};

const gdl_plot_type * gdl_gpca_result_plot = &_gdl_gpca_result_plot;

