/*  
 *  gpca/tag_result.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_odb.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_pca.h>
#include <gdl/gdl_gpca_tag.h>
#include <gdl/gdl_gpca_tag_result.h>

struct _gdl_gpca_tag_result
{
	const gdl_gpca_workspace_type * type;
	gdl_string  ** names;
	gdl_mask     * mask;
	size_t       size;
};

gdl_gpca_tag_result *
gdl_gpca_tag_result_alloc (const gdl_gpca_tag_workspace * w)
{
	size_t i;
	const gdl_entity_type * E;
	const gdl_gview       * G;
	gdl_gpca_tag_result   * r;
	
	r = GDL_MALLOC (gdl_gpca_tag_result, 1);
	
	r->mask = gdl_mask_clone (gdl_gpca_tag_workspace_get (w));
	r->type = gdl_gpca_tag_workspace_type (w);
	
	G = gdl_gpca_tag_workspace_data (w);
	E = gdl_gpca_tag_workspace_mode (r->type);
	
	r->size  = gdl_mask_size (r->mask, E);
	r->names = GDL_CALLOC (gdl_string *, r->size);
	
	for (i = 0; i < r->size; i++)
	{
		const gdl_string * name;
		const gdl_entity * entity;
		if (E == GDL_LOCUS)
		{
			entity = gdl_gview_get_locus (G, gdl_mask_get_idx (r->mask, E, i));
			name   = gdl_entity_get_name (entity);
			r->names[i] = gdl_string_clone (name);
		}
		else if (E == GDL_ACCESSION)
		{
			entity = gdl_gview_get_accession (G, gdl_mask_get_idx (r->mask, E, i));
			name   = gdl_entity_get_name (entity);
			r->names[i] = gdl_string_clone (name);
		}
	}
	
	return r;	
}

void
gdl_gpca_tag_result_free (gdl_gpca_tag_result * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
		{
			gdl_string_free (r->names[i]);	
		}
		GDL_FREE (r->names);
		gdl_mask_free (r->mask);
		GDL_FREE (r);
	}	
}

size_t
gdl_gpca_tag_result_size (const gdl_gpca_tag_result * r)
{
	return r->size;
}

const gdl_mask *
gdl_gpca_tag_result_mask (const gdl_gpca_tag_result * r)
{
	return r->mask;	
}

gdl_gpca_tag_result *
gdl_gpca_tag_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_gpca_tag_result * r;
		
		r = GDL_MALLOC (gdl_gpca_tag_result, 1);
		
		r->type = gdl_gpca_workspace_type_fread (stream);
		GDL_FREAD_STATUS (r->type!=0, 1);
		r->mask = gdl_mask_fread (stream);
		GDL_FREAD_STATUS (r->mask!=0, 1);
		status  = fread (&(r->size), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		r->names = GDL_MALLOC (gdl_string *, r->size);
		for (i = 0; i < r->size; i++)
		{
			r->names[i] = gdl_string_fread (stream);
			GDL_FREAD_STATUS (r->names[i] != 0, 1);
		} 
		
		return r;
	}
	
	return NULL;
}

int
gdl_gpca_tag_result_fwrite (FILE * stream, const gdl_gpca_tag_result * r)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status =  gdl_gpca_workspace_type_fwrite (stream, r->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_mask_fwrite (stream, r->mask);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status  = fwrite (&(r->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < r->size; i++)
		{
			status = gdl_string_fwrite (stream, r->names[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		} 
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

int
gdl_gpca_tag_result_fprintf (FILE * stream, const gdl_gpca_tag_result * r)
{
	if (stream && r)
	{
		size_t i;
		const gdl_entity_type * E;
		
		E = gdl_gpca_tag_workspace_mode (r->type);
		
		fprintf (stream, "PCA-Tag Result\n");
		fprintf (stream, "\tType [ %s ]\n", r->type->acronym);
		fprintf (stream, "\tNumber of selected entities [ %d ]\n", r->size);
		fprintf (stream, "\n--------------------------------------------\n");
		fprintf (stream, "\nTag-ID\tData-ID\tName\n");
		for (i = 0; i < gdl_mask_size (r->mask, E); i++)
		{
			fprintf (stream, "%d\t%d\t%s\n", i+1, gdl_mask_get_idx (r->mask, E, i)+1, r->names[i]);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}
