/*  
 * 	gpoint/gvreg.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gpoint.h>

struct _gdl_gvalues_registry
{
	size_t size;
	gdl_gvalues ** data;
};

static int
gdl_gvalues_registry_grow (gdl_gvalues_registry * b, size_t size)
{
	if (b->size)
	{
		gdl_gvalues ** new = GDL_MALLOC (gdl_gvalues *, size);
		memcpy (new, b->data, sizeof (gdl_gvalues *)*b->size);
		GDL_FREE (b->data);
		b->data = new;
		b->size = size;
	}
	else
	{
		b->data = GDL_MALLOC (gdl_gvalues *, size);
		b->size = size;
	}
	return 0;
}

gdl_gvalues_registry *
gdl_gvalues_registry_alloc ()
{
	gdl_gvalues_registry * b;
	
	b = GDL_MALLOC (gdl_gvalues_registry, 1);
	
	if (b)
	{
		b->size    = 0;
		b->data    = NULL;
	}
	
	return b;
}

void
gdl_gvalues_registry_free (gdl_gvalues_registry * b)
{
	if (b)
	{
		if (b->data)
		{
			size_t i;
			for (i = 0; i < b->size; i++)
			{
				gdl_gvalues_free (b->data[i]);
			}
			GDL_FREE (b->data);
		}
		GDL_FREE (b);
	}
}

int
gdl_gvalues_registry_add (gdl_gvalues_registry * b, const gdl_gvalues * values)
{
	if (b == NULL || values == NULL)
		return (-1);
	else
	{
		size_t i;
		
		for (i = 0; i < b->size; i++)
		{
			if (!gdl_gvalues_compare (b->data[i], values))
				return i;
		}
		
		gdl_gvalues_registry_grow (b, b->size+1);
		
		b->data[i] = gdl_gvalues_clone (values);
		
		return i;
	}
}

gdl_gvalues *
gdl_gvalues_registry_get (gdl_gvalues_registry * b, size_t i)
{
	if (b->size)
	{
		return b->data[i];
	}
}

gdl_gvalues_registry *
gdl_gvalues_registry_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t i, size, status;
		gdl_gvalues_registry * g;
		
		status = fread (&size, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g = gdl_gvalues_registry_alloc ();
		
		if (size)
		{
			gdl_gvalues_registry_grow (g, size);
			for (i = 0; i < size; i++)
			{
				g->data[i] = gdl_gvalues_fread (stream);
				GDL_FREAD_STATUS (g->data[i]!=NULL, 1);
			}
		}
		
		return g;		
	}
}

int
gdl_gvalues_registry_fwrite (FILE * stream, const gdl_gvalues_registry * g)
{
	if (stream == 0 || g == 0)
		return GDL_EINVAL;
	else
	{
		size_t i, status;
		
		status = fwrite (&g->size, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < g->size; i++)
		{
			status = gdl_gvalues_fwrite (stream, g->data[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}	
	}
	return GDL_SUCCESS;	
}

