/*  
 *  graphics/colorRGB
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <stdlib.h>
#include <stdio.h>  
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_io.h> 
#include <gdl/gdl_graphics.h>


static const int PALETTE_SIZE        = 156;
static const gdl_rgb_color PALETTE[] =
{
{0x000000, "\0\0\0\0\0\0\0", 189, 14, 63, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 148, 184, 96, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 7, 25, 154, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 134, 118, 166, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 6, 123, 133, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 173, 181, 241, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 243, 225, 69, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 237, 107, 82, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 120, 171, 210, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 186, 146, 188, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 111, 160, 129, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 148, 19, 101, 1.0},	
{0x000000, "\0\0\0\0\0\0\0", 127, 255, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 210, 105, 30, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 100, 149, 237, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 248, 220, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 240, 248, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 75, 0, 130, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 245, 222, 179, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 95, 158, 160, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 250, 235, 215, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 255, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 127, 80, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 127, 255, 212, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 240, 255, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 245, 245, 220, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 228, 196, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 235, 205, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 0, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 165, 42, 42, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 222, 184, 135, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 220, 20, 60, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 255, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 0, 139, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 139, 139, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 184, 132, 11, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 169, 169, 169, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 100, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 169, 169, 169, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 189, 183, 107, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 139, 0, 139, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 85, 107, 47, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 140, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 153, 50, 204, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 139, 0, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 233, 150, 122, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 138, 43, 226, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 143, 188, 143, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 72, 61, 139, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 47, 79, 79, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 47, 79, 79, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 206, 209, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 148, 0, 211, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 20, 147, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 191, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 105, 105, 105, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 105, 105, 105, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 30, 144, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 178, 34, 34, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 34, 139, 34, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 0, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 220, 220, 220, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 248, 248, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 215, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 218, 165, 32, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 128, 128, 128, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 128, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 173, 255, 47, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 128, 128, 128, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 240, 255, 240, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 105, 180, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 205, 92, 92, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 255, 240, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 240, 230, 140, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 230, 230, 250, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 240, 245, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 124, 252, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 250, 205, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 173, 216, 230, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 240, 128, 128, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 224, 255, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 250, 250, 210, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 211, 211, 211, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 144, 238, 144, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 211, 211, 211, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 182, 193, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 160, 122, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 32, 178, 170, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 135, 206, 250, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 119, 136, 153, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 119, 136, 153, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 176, 196, 222, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 255, 224, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 255, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 50, 205, 50, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 250, 240, 230, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 0, 255, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 128, 0, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 102, 205, 170, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 0, 205, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 186, 85, 211, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 147, 112, 219, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 60, 179, 113, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 123, 104, 238, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 250, 154, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 72, 209, 204, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 199, 21, 133, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 25, 25, 112, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 245, 255, 250, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 228, 225, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 228, 181, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 222, 173, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 0, 128, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 253, 245, 230, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 128, 128, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 107, 142, 35, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 165, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 69, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 218, 112, 214, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 238, 232, 170, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 152, 251, 152, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 175, 238, 238, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 219, 112, 147, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 239, 213, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 218, 185, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 205, 133, 63, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 192, 203, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 221, 160, 203, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 176, 224, 230, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 128, 0, 128, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 0, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 188, 143, 143, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 65, 105, 225, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 139, 69, 19, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 250, 128, 114, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 244, 164, 96, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 46, 139, 87, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 245, 238, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 160, 82, 45, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 192, 192, 192, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 135, 206, 235, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 106, 90, 205, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 119, 128, 144, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 119, 128, 144, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 255, 250, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 255, 127, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 70, 130, 180, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 210, 180, 140, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 0, 128, 128, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 216, 191, 216, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 99, 71, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 64, 224, 208, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 238, 130, 238, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 245, 222, 179, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 245, 245, 245, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 255, 255, 0, 1.0},
{0x000000, "\0\0\0\0\0\0\0", 154, 205, 50, 1.0}
};


gdl_rgb_color *
gdl_rgb_color_alloc()
{
   gdl_rgb_color *  gptr = NULL;

   gptr = GDL_MALLOC (gdl_rgb_color, 1);

	gptr->hex    = 0xffffff;
	gptr->strhex = gdl_string_alloc (7);
	gptr->red   = 0;
	gptr->green = 0;
	gptr->blue  = 0;
	gptr->alpha = 1.0;

   return gptr;
}
/**
 * 
 */
void gdl_rgb_color_free (gdl_rgb_color * color)
{
	if (color) {
		GDL_FREE (color->strhex);
		GDL_FREE (color);
	}
}

gdl_rgb_color_vector *
gdl_rgb_color_vector_alloc (size_t size)
{
	size_t i;
	gdl_rgb_color_vector * v;
	
	v = GDL_MALLOC (gdl_rgb_color_vector, 1);
	
	v->size = size;
	
	v->colors = GDL_CALLOC (gdl_rgb_color *, size);
	
	for (i = 0; i < size; i++)
	{
		v->colors[i] = gdl_rgb_color_alloc ();
	}
	
	return v;
}

void
gdl_rgb_color_vector_free (gdl_rgb_color_vector * v)
{
	if (v)
	{
		size_t i;
		for (i = 0; i < v->size; i++)
		{
			gdl_rgb_color_free (v->colors[i]);
		}
		GDL_FREE (v->colors);
		GDL_FREE (v);
	}	
}
/**
 * 
 */
gdl_rgb_color *
gdl_rgb_color_create (int r, int g, int b)
{
	gdl_rgb_color * c = gdl_rgb_color_alloc();
	c->red   = r;
	c->green = g;
	c->blue  = b;
	gdl_rgb_color_rgb2l(&c->hex, r, g, b);
	gdl_rgb_color_rgb2str(c->strhex, r, g, b);
	return c;
}
/**
 * 
 */
void 
gdl_rgb_color_rgb2str (char *str, int r, int g, int b)
{
	int i;
	// Clean str
	for(i=0;i<7;i++) str[i]='\0';
	gdl_rgb_color_comp2str(&str[0], r);
	gdl_rgb_color_comp2str(&str[2], g);
	gdl_rgb_color_comp2str(&str[4], b);
}

void 
gdl_rgb_color_comp2str (char *str, int val)
{
	gdl_string * buff;
	
	buff = gdl_string_sprintf("%x", val);
	
	switch (strlen(buff)) {
		case 1 :
			str[0]='0';
			str[1]=buff[0];
			break;
		case 2 :
			str[0]=buff[0];
			str[1]=buff[1];
			break;
	}
	
	gdl_string_free (buff);
}
/**
 * 
 */
gdl_rgb_color *
gdl_rgb_color_strto (char * str)
{
	int i,len,*val;
	char *tmp;
	gdl_rgb_color * c = gdl_rgb_color_alloc();
	
	c->hex = strtol(str, NULL, 16);
	strncpy(c->strhex, str, GDL_MIN(6, strlen(str)));
	gdl_rgb_color_l2rgb(c->hex, &c->red, &c->green, &c->blue);
	
	return c;
}
/**
 * 
 * 
 */
void
gdl_rgb_color_l2rgb(long color, int *r, int *g, int *b)
{
	long q;
	
	*r = color%16;
	 q = color/16;
	*g = q%16;
	*b = q/16;

}
/**
 * 
 * 
 */
void
gdl_rgb_color_rgb2l(long *color, int r, int g, int b)
{
	*color = r+16*g+16*16*b;
}
/**
 * 
 */
void gdl_rgb_color_copy(gdl_rgb_color * dest, gdl_rgb_color * src)
{
	if (dest == NULL) return;
	if (src == NULL) return;
	
	dest->hex = src->hex;
	dest->red = src->red;
	dest->green = src->green;
	dest->blue  = src->blue;
	dest->alpha = src->alpha;
	strcpy(dest->strhex, src->strhex);
}

gdl_rgb_color * gdl_rgb_color_get_white()
{
	return gdl_rgb_color_create(255,255,255);
}
/**
 * 
 */
gdl_rgb_color * gdl_rgb_color_get_black()
{
	return gdl_rgb_color_create(0,0,0);
}
/**
 * 
 */
gdl_rgb_color * gdl_rgb_color_get_red()
{
	return gdl_rgb_color_create(255,0,0);	
}
/**
 * 
 */
gdl_rgb_color * gdl_rgb_color_get_green()
{
	return gdl_rgb_color_create(0,255,0);
}
/**
 * 
 */
gdl_rgb_color * gdl_rgb_color_get_blue()
{
	return gdl_rgb_color_create(0,0,255);
}
/**
 * 
 */
gdl_rgb_color * gdl_rgb_color_get_yellow()
{
	return gdl_rgb_color_create(255,255,0);
}
/**
 * 
 * 
 */
gdl_rgb_color_vector * 
gdl_rgb_color_gradient(gdl_rgb_color * from, gdl_rgb_color * to, size_t nclass)
{
	int i,r,g,b;
	gdl_rgb_color_vector * palette = gdl_rgb_color_vector_alloc (nclass);
	
	gdl_rgb_color_copy(palette->colors[0], from);
	gdl_rgb_color_copy(palette->colors[nclass-1], to);
	
	r = to->red-from->red;
	g = to->green-from->green;
	b = to->blue-from->blue;
	
	for(i=1;i<nclass-1;i++)
	{
		palette->colors[i] = gdl_rgb_color_create(from->red + i*r/(nclass-1), from->green + i*g/(nclass-1), from->blue + i*b/(nclass-1));
	}
	
	return palette;
}
/**
 * 
 */
gdl_rgb_color_vector *
gdl_rgb_color_palette (size_t ncolor)
{
	int i;
	gdl_rgb_color_vector * palette;
	
	if (ncolor > PALETTE_SIZE)
	{
		GDL_ERROR_VAL ("Too many number of colors !", GDL_EINVAL, 0); 	
	}
	
	palette = gdl_rgb_color_vector_alloc (ncolor);
	
	for(i=0;i<ncolor;i++)
	{
		gdl_rgb_color_copy(palette->colors[i], (gdl_rgb_color *)&PALETTE[i]);
		gdl_rgb_color_rgb2l(&palette->colors[i]->hex, palette->colors[i]->red, palette->colors[i]->green, palette->colors[i]->blue);
		gdl_rgb_color_rgb2str(palette->colors[i]->strhex, palette->colors[i]->red, palette->colors[i]->green, palette->colors[i]->blue);
	}
	
	return palette;
}
/**
 * 
 * 
 */
int
gdl_rgb_color_vector_fprintf (FILE *stream, const gdl_rgb_color_vector * v)
{
	int i;
	
	fprintf(stream, "-ncolor=%d\n", v->size);
	
	for(i=0;i<v->size;i++)
	{
		gdl_rgb_color_fprintf (stream, v->colors[i]);		
		fprintf(stream, "\n");
	}
}
/**
 * 
 * 
 */
int
gdl_rgb_color_fprintf (FILE *stream, const gdl_rgb_color * color)
{
	return fprintf (stream, "#%s", color->strhex);
}

gdl_rgb_color *
gdl_rgb_color_fscanf (FILE *stream)
{
	if (stream)
	{
		int status;
		gdl_string * hexa = gdl_string_alloc (7);
		gdl_rgb_color * color;
		
		status = fscanf (stream, "#%s", hexa);
		
		if (status!=1)
		{
			GDL_ERROR_VAL ("Unable to scan rgb color", GDL_FAILURE, 0);	
		}
		
		color = gdl_rgb_color_strto (hexa);
		
		gdl_string_free (hexa);
		
		return color;
	}
	
	return NULL;	
}
