/*  
 *  graphics/graphics2D
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_io.h> 
#include <gdl/gdl_graphics.h>

/**
 * 
 */
gdl_graphics2D_ptr
gdl_graphics2D_alloc(void)
{
  gdl_graphics2D_ptr  gptr;
  
  gptr = GDL_MALLOC (gdl_graphics2D, 1);
  
 if (gptr != NULL) {
 	gptr->device         = GDL_GRAPHICS_DEVICE_SVG;
 	gptr->stroke_ndash	= 0;
 	gptr->stroke_dashes  = NULL;
 	gptr->stroke_width   = 1;
	gptr->color  	  = gdl_rgb_color_get_white();
	gptr->paint  	  = gdl_rgb_color_get_white();
	gptr->background = gdl_rgb_color_get_white();
	gptr->font       = gdl_font_default();
	gptr->transform  = NULL;
	gptr->_last_transform  = NULL;
 	gptr->stream     = NULL;
 }
 return gptr;
}
/**
 * 
 */
void
gdl_graphics2D_free(gdl_graphics2D_ptr gptr)
{
	if (gptr != NULL) {
	 	if (gptr->color!=NULL) gdl_rgb_color_free(gptr->color);
		if (gptr->paint!=NULL) gdl_rgb_color_free(gptr->paint);
		if (gptr->background!=NULL) gdl_rgb_color_free(gptr->background);
		gdl_affine_transform_clear(gptr->transform);
	 	GDL_FREE (gptr);
    }
}
/**
 * 
 */
void
gdl_graphics2D_set_stroke(gdl_graphics2D_ptr ptr, double width, int ndash, double *dashes)
{
	ptr->stroke_width = width;
	
	if (ptr->stroke_dashes != NULL) 
	{
		GDL_FREE (ptr->stroke_dashes);
	}
	if (ndash) {
		ptr->stroke_ndash = ndash;
		ptr->stroke_dashes = GDL_MALLOC (double, ptr->stroke_ndash);
		memcpy (ptr->stroke_dashes, dashes, sizeof (double)*ndash);
	}
}

void
gdl_graphics2D_set_color(gdl_graphics2D_ptr ptr, struct gdl_rgb_color *color)
{
	gdl_rgb_color_copy(ptr->color, color);
}

void
gdl_graphics2D_set_paint(gdl_graphics2D_ptr ptr, struct gdl_rgb_color *color)
{
	gdl_rgb_color_copy(ptr->paint, color);
}

void
gdl_graphics2D_set_background(gdl_graphics2D_ptr ptr, struct gdl_rgb_color *color)
{
	gdl_rgb_color_copy(ptr->background, color);
}

void
gdl_graphics2D_set_transform(gdl_graphics2D_ptr ptr, struct gdl_affine_transform *transform)
{
	gdl_affine_transform_clear(ptr->transform);
	ptr->transform = gdl_affine_transform_clone(transform);
}

struct
gdl_affine_transform *gdl_graphics2D_get_transform(gdl_graphics2D_ptr ptr)
{
	return gdl_affine_transform_clone(ptr->transform);
}

void
gdl_graphics2D_add_transform(gdl_graphics2D_ptr ptr, struct gdl_affine_transform *transform)
{
	gdl_affine_transform_ptr tptr,gptr;
	
	if (ptr->transform == NULL)
		ptr->_last_transform = ptr->transform = gdl_affine_transform_clone(transform);
	else {
		tptr = gdl_affine_transform_clone(transform);
		gptr = gdl_affine_transform_last(ptr->transform);
		gptr->next = tptr;
		tptr->prev = gptr;
		ptr->_last_transform = tptr;		
	}
}
/**
 * 
 */
void
gdl_graphics2D_transform(gdl_graphics2D_ptr ptr,
                         double x,
                         double y,
                         double *tx,
                         double *ty
                         )
{
	gdl_affine_transform_apply(ptr->_last_transform, x, y, tx, ty);
}                         
/**
 * 
 */
gdl_graphics2D_ptr
gdl_graphics2D_open(const char *file, double width, double height, int device)
{
	gdl_graphics2D_ptr g = gdl_graphics2D_alloc();
	g->stream            = gdl_fileopen(file, "w");	
	g->device            = device;
	switch(g->device) {
		case GDL_GRAPHICS_DEVICE_PS :
			gdl_graphics2D_start_ps(g, width, height);
			break;
		case GDL_GRAPHICS_DEVICE_SVG :
			gdl_graphics2D_start_svg(g, width, height);
			break;
		default :
			gdl_graphics2D_start_svg(g, width, height);
			break;
	}
	return g;
}

void
gdl_graphics2D_close(gdl_graphics2D_ptr ptr, const char *file)
{
	switch(ptr->device) {
		case GDL_GRAPHICS_DEVICE_PS :
			gdl_graphics2D_end_ps(ptr);
			break;
		case GDL_GRAPHICS_DEVICE_SVG :
			gdl_graphics2D_end_svg(ptr);
			break;
		default :
			gdl_graphics2D_end_svg(ptr);
			break;
	}
	gdl_fileclose(file, ptr->stream);
}
/**
 * 
 */
char
*gdl_graphics2D_get_strpath2D(double *x, double *y, int np, int *len)
{
	int i,j;
	gdl_string * pts, * buf;
	
	// First compute the length of the string path
	for(*len=0,i=0;i<np;i++) {
		if (i<np-1) {
			buf = gdl_string_sprintf("%g,%g,",x[i],y[i]);
		}
		else 
		{
			buf = gdl_string_sprintf("%g,%g",x[i],y[i]);
		}
		(*len) += strlen(buf);
		gdl_string_free (buf);
	}
	(*len)++;
	
	pts = gdl_string_alloc (*len);

	for(i=0;i<np;i++) {
		
		if (i<np-1) {
			buf = gdl_string_sprintf("%g,%g,",x[i],y[i]);
		}
		else 
		{
			buf = gdl_string_sprintf("%g,%g",x[i],y[i]);
		}
		
		strcat(pts, buf);
		
		gdl_string_free (buf);
	}
	
	return pts;
}
/**
 * 
 * 
 */
gdl_rgb_color_ptr
gdl_graphics2D_get_color(gdl_graphics2D_ptr ptr)
{
	gdl_rgb_color_ptr color = gdl_rgb_color_alloc();
	
	gdl_rgb_color_copy(color, ptr->color);
	
	return color; 
}
/**
 * 
 * 
 */
void
gdl_graphics2D_set_font(gdl_graphics2D_ptr ptr, struct gdl_font *font)
{
	gdl_font_copy(ptr->font, font);
}
/**
 * 
 * 
 */
struct
gdl_font *gdl_graphics2D_get_font(gdl_graphics2D_ptr ptr)
{
	gdl_font_ptr font = gdl_font_alloc();
	gdl_font_copy(font, ptr->font);
	return font;
}
