/*  
 * 	gview/mask.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>

size_t
gdl_gview_gmask_accession_size (const gdl_gview * gview, const gdl_mask * m)
{
	const gdl_entity_mask * em = gdl_mask_get (m, GDL_ACCESSION);
	
	if (em)
	{
		return gdl_entity_mask_size (em);
	}
	else
	{
		return gdl_gview_accession_size (gview);
	}
}

size_t
gdl_gview_gmask_locus_size (const gdl_gview * gview, const gdl_mask * m)
{
	const gdl_entity_mask * em = gdl_mask_get (m, GDL_LOCUS);
	
	if (em)
	{
		return gdl_entity_mask_size (em);
	}
	else
	{
		return gdl_gview_locus_size (gview);
	}	
}

gdl_accession *
gdl_gview_gmask_get_accession (const gdl_gview * gview, const gdl_mask * m, size_t i)
{
	return gdl_gview_get_accession (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i));	
}

gdl_locus *
gdl_gview_gmask_get_locus (const gdl_gview * gview, const gdl_mask * m, size_t i)
{
	return gdl_gview_get_locus (gview, gdl_mask_get_idx (m, GDL_LOCUS, i));
}

gdl_gvalues_get *
gdl_gview_gmask_get_new (const gdl_gview * g, const gdl_mask * m)
{
	gdl_gvalues_get * get;
	size_t i, j, amax=0, gmax=0, size, nl;
	
	nl = GDL_GVIEW_LOCUS_SIZE (g, m);
	
	for (i = 0; i < nl; i++)
	{
		gdl_locus * vl 
		   = GDL_GVIEW_GET_LOCUS (g, m, i);
		j = gdl_locus_allele (vl);		
		if (j > amax) amax = j;
		j = gdl_locus_genotype (vl);
		if (j > gmax) gmax = j;
	}
	size = GDL_MAX (gmax, amax);
	return gdl_gvalues_get_alloc (size);
}

int
gdl_gview_gmask_genotype (const gdl_gview * gview, const gdl_mask * m, size_t i, size_t j, gdl_gvalues_get * g)
{
	return gdl_gview_genotype_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_LOCUS, j), g);
}

int
gdl_gview_gmask_allele (const gdl_gview * gview, const gdl_mask * m, size_t i, size_t j, size_t k, gdl_gvalues_get * g)
{
	return gdl_gview_allele_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_LOCUS, j), k, g);
}

int
gdl_gview_gmask_genotype_template (const gdl_gview * gview, const gdl_mask * m, size_t i, gdl_locus * locus, gdl_gvalues_get * g)
{
	return gdl_gview_genotype_template_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), locus, g);
}

gdl_boolean
gdl_gview_gmask_has_missing (const gdl_gview * gview, const gdl_mask * m, size_t i, size_t j)
{
	return gdl_gview_has_missing_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_LOCUS, j));
}

gdl_boolean
gdl_gview_gmask_is_missing (const gdl_gview * gview, const gdl_mask * m, size_t i, size_t j, size_t p)
{
	return gdl_gview_is_missing_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_LOCUS, j), p);
}

static size_t
gdl_gview_gmask_accession_missing_size_f (const gdl_gview * gview, size_t i, const gdl_mask * m)
{
	size_t j, nl, nm=0;
	
	nl = gdl_mask_get_size (m, GDL_LOCUS);
	
	for (j = 0; j < nl; j++)
	{
		nm += gdl_gview_gpoint_missing_size_f (gview, i, gdl_mask_get_idx (m, GDL_LOCUS, j));
	}
	
	return nm;
}

static size_t
gdl_gview_gmask_locus_missing_size_f (const gdl_gview * gview, size_t i, const gdl_mask * m)
{
	size_t j, na, nm=0;
	
	na = gdl_mask_get_size (m, GDL_ACCESSION);
	
	for (j = 0; j < na; j++)
	{
		nm += gdl_gview_gpoint_missing_size_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, j), i);
	}
	
	return nm;
}

size_t
gdl_gview_gmask_accession_missing_size (const gdl_gview * gview, const gdl_mask * m, size_t i)
{
	const gdl_entity_mask * locus = gdl_mask_get (m, GDL_LOCUS);
	
	if (locus)
	{
		return gdl_gview_gmask_accession_missing_size_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), m);
	}
	else
	{
		return gdl_gview_accession_missing_size_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i));
	}	
}

size_t
gdl_gview_gmask_locus_missing_size (const gdl_gview * gview, const gdl_mask * m, size_t i)
{
	const gdl_entity_mask * accession = gdl_mask_get (m, GDL_ACCESSION);
	
	if (accession)
	{
		return gdl_gview_gmask_locus_missing_size_f (gview, gdl_mask_get_idx (m, GDL_LOCUS, i), m);
	}
	else
	{
		return gdl_gview_locus_missing_size_f (gview, gdl_mask_get_idx (m, GDL_LOCUS, i));
	}	
}

size_t
gdl_gview_gmask_gpoint_missing_size (const gdl_gview * gview, const gdl_mask * m, size_t i, size_t j)
{
	return gdl_gview_gpoint_missing_size_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_LOCUS, j));
}


gdl_boolean
gdl_gview_gmask_is_homozygous (const gdl_gview * gview, const gdl_mask * m, size_t i, size_t j)
{
	return gdl_gview_is_homozygous_f (gview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_LOCUS, j));
}

int
gdl_gview_gmask_accession_compare (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	return gdl_gview_accession_compare_f (g, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_ACCESSION, j), gdl_mask_get (m, GDL_LOCUS));
}

int
gdl_gview_gmask_locus_compare (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	return gdl_gview_locus_compare_f (g, gdl_mask_get_idx (m, GDL_LOCUS, i), gdl_mask_get_idx (m, GDL_LOCUS, j), gdl_mask_get (m, GDL_ACCESSION));
}

gdl_vector *
gdl_gview_gmask_genotype_freq (const gdl_gview * g, const gdl_mask * m, size_t l, gdl_gvalues_get * gb)
{
	size_t i, j, k, n, obs=0;
	const gdl_gvalues * x;
	gdl_vector * f;
	
	n = GDL_GVIEW_ACCESSION_SIZE (g, m);
	
	f = gdl_vector_calloc (gdl_locus_genotype (GDL_GVIEW_GET_LOCUS (g, m, l)));
	
	for (i = 0; i < n; i++)
	{
		GDL_GVIEW_GET_GENOTYPE (g, m, i, l, gb);
		x = gdl_gvalues_get_gvalues (gb);
		if (x)
		{
			for (k = 0; k < x->size; k++)
			{
				size_t a = x->values[k]->idx;
				gdl_vector_set (f, a, gdl_vector_get (f, a) + x->values[k]->value);
			}
			obs++;
		}	
	}
	
	gdl_vector_scale (f, 1.0/obs);
	
	return f;	
}

gdl_vector *
gdl_gview_gmask_allele_freq (const gdl_gview * g, const gdl_mask * m, size_t l, gdl_gvalues_get * gb)
{
	size_t i, j, k, n, p, obs=0;
	const gdl_gvalues * x;
	gdl_vector * f;
	
	p = gdl_gview_ploidy (g);
	n = GDL_GVIEW_ACCESSION_SIZE (g, m);
	
	f = gdl_vector_calloc (gdl_locus_allele (GDL_GVIEW_GET_LOCUS (g, m, l)));
	
	for (i = 0; i < n; i++)
	{
		for (j = 0; j < p; j++)
		{
			GDL_GVIEW_GET_ALLELE (g, m, i, l, j, gb);
			x = gdl_gvalues_get_gvalues (gb);
			if (x)
			{
				for (k = 0; k < x->size; k++)
				{
					size_t a = x->values[k]->idx;
					gdl_vector_set (f, a, gdl_vector_get (f, a) + x->values[k]->value);
				}
				obs++;
			}
		}	
	}
	
	gdl_vector_scale (f, 1.0/obs);
	
	return f;	
}

size_t
GDL_GVIEW_ACCESSION_SIZE (const gdl_gview * g, const gdl_mask * m)
{
	if (m) return gdl_gview_gmask_accession_size (g, m);
    else return gdl_gview_accession_size (g);
}

size_t
GDL_GVIEW_LOCUS_SIZE (const gdl_gview * g, const gdl_mask * m)
{
	if (m) return gdl_gview_gmask_locus_size (g, m);
    else return gdl_gview_locus_size (g);
}

gdl_accession * 
GDL_GVIEW_GET_ACCESSION (const gdl_gview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_gview_gmask_get_accession (g, m, i);
    else return gdl_gview_get_accession (g, i);
}

gdl_locus *
GDL_GVIEW_GET_LOCUS (const gdl_gview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_gview_gmask_get_locus (g, m, i);
	else return gdl_gview_get_locus (g, i);	
}

gdl_boolean
GDL_GVIEW_HAS_MISSING (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	if (m) return gdl_gview_gmask_has_missing (g, m, i, j);
    else return gdl_gview_has_missing_f (g, i, j);	
}

gdl_boolean
GDL_GVIEW_IS_MISSING (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j, size_t p)
{
	if (m) return gdl_gview_gmask_is_missing (g, m, i, j, p);
    else return gdl_gview_is_missing_f (g, i, j, p);
}

size_t
GDL_GVIEW_ACCESSION_MISSING_SIZE (const gdl_gview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_gview_gmask_accession_missing_size (g, m, i);
    else return gdl_gview_accession_missing_size_f (g, i);
}

size_t
GDL_GVIEW_LOCUS_MISSING_SIZE (const gdl_gview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_gview_gmask_locus_missing_size (g, m, i);
    else return gdl_gview_locus_missing_size_f (g, i);
}

size_t
GDL_GVIEW_GPOINT_MISSING_SIZE (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	if (m) return gdl_gview_gmask_gpoint_missing_size (g, m, i, j);
    else return gdl_gview_gpoint_missing_size_f (g, i, j);
}

gdl_boolean
GDL_GVIEW_IS_HOMOZYGOUS (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	if (m) return gdl_gview_gmask_is_homozygous (g, m, i, j);
    else return gdl_gview_is_homozygous_f (g, i, j);
}

gdl_gvalues_get *
GDL_GVIEW_GET_NEW (const gdl_gview * g, const gdl_mask * m)
{
	if (m) return gdl_gview_gmask_get_new (g, m);
    else return gdl_gview_get_new (g);	
}

int
GDL_GVIEW_GET_GENOTYPE (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j, gdl_gvalues_get * x)
{
    if (m) return gdl_gview_gmask_genotype (g, m, i, j, x);
    else return gdl_gview_genotype_f (g, i, j, x);
}

int
GDL_GVIEW_GET_ALLELE (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j, size_t k, gdl_gvalues_get * x)
{
	if (m) return gdl_gview_gmask_allele (g, m, i, j, k, x);
    else return gdl_gview_allele_f (g, i, j, k, x);
}

int
GDL_GVIEW_GET_GENOTYPE_TEMPLATE (const gdl_gview * g, const gdl_mask * m, size_t i, gdl_locus * locus, gdl_gvalues_get * x)
{
	if (m) return gdl_gview_gmask_genotype_template (g, m, i, locus, x);
    else return gdl_gview_genotype_template_f (g, i, locus, x);	
}

int
GDL_GVIEW_ACCESSION_COMPARE (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	if (m) return gdl_gview_gmask_accession_compare (g, m, i, j);
    else return gdl_gview_accession_compare_f (g, i, j, NULL);
}

int
GDL_GVIEW_LOCUS_COMPARE (const gdl_gview * g, const gdl_mask * m, size_t i, size_t j)
{
	if (m) return gdl_gview_gmask_locus_compare (g, m, i, j);
    else return gdl_gview_locus_compare_f (g, i, j, NULL);
}

gdl_vector *
GDL_GVIEW_GENOTYPE_FREQ (const gdl_gview * g, const gdl_mask * m, size_t l, gdl_gvalues_get * gb)
{
	if (m) return gdl_gview_gmask_genotype_freq (g, m, l, gb);
	else return gdl_gview_genotype_freq_f (g, l, gb);
}

gdl_vector *
GDL_GVIEW_ALLELE_FREQ (const gdl_gview * g, const gdl_mask * m, size_t l, gdl_gvalues_get * gb)
{
	if (m) return gdl_gview_gmask_allele_freq (g, m, l, gb);
	else return gdl_gview_allele_freq_f (g, l, gb);
}

gdl_mask *
gdl_gview_gmask_reverse (const gdl_gview * g, const gdl_mask * m)
{
	gdl_mask * rm;
	const gdl_entity_mask * am, * lm;
	gdl_entity_mask * ram, * rlm;
	
	rm = gdl_mask_alloc ();
	
	am  = gdl_mask_get (m, GDL_ACCESSION);
	
	if (am)
	{
	   ram = gdl_entity_mask_reverse (am, gdl_gview_accession_size (g));
	   gdl_mask_set (rm, GDL_ACCESSION, ram, 1);
	}
	
	lm  = gdl_mask_get (m, GDL_LOCUS);
	
	if (lm)
	{
	   rlm = gdl_entity_mask_reverse (lm, gdl_gview_locus_size (g));
		gdl_mask_set (rm, GDL_LOCUS, rlm, 1);
	}
	
	return rm;	
}
