/*  
 * 	gview/uninfo.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>

static gdl_mask * gdl_gview_gmask_uninformative_locus (const gdl_gview * g, gdl_mask * m);
static gdl_mask * gdl_gview_gmask_uninformative_accession (const gdl_gview * g, gdl_mask * m);

gdl_mask *
gdl_gview_gmask_uninformative (const gdl_gview * g,
                                  gdl_mask * m,
                              const gdl_entity_type * type)
{
	if (type == GDL_LOCUS)
	{
		return gdl_gview_gmask_uninformative_locus (g, m);
	}
	else if (type == GDL_ACCESSION)
	{
		return gdl_gview_gmask_uninformative_accession (g, m);
	}
	else
	{
		return NULL;	
	}
}

static gdl_mask *
gdl_gview_gmask_uninformative_locus (const gdl_gview * g,
                                          gdl_mask * m)
{
	size_t i;
	size_t nl;
	gdl_mask * u = NULL;
	
	nl = GDL_GVIEW_LOCUS_SIZE (g, m);
	
	for (i = 0; i < nl; i++)
	{
		gdl_locus * locus 
		   = GDL_GVIEW_GET_LOCUS (g, m, i);
		size_t ng = gdl_locus_informative_genotype (locus);
		if (ng >= 2)
		{
			// keep it
			if (u == 0)
			{
				u = gdl_mask_alloc ();
				if (m != 0)
				{
					gdl_mask_set (u, GDL_ACCESSION, gdl_mask_get_clone (m, GDL_ACCESSION), 1);
				}
			}
			gdl_mask_add (u, locus);
		}
	}
	
	return u;
}

static gdl_mask *
gdl_gview_gmask_uninformative_accession (const gdl_gview * g,
                                          gdl_mask * m)
{
	size_t i, j;
	size_t na, nl, ni;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues * x;
	gdl_mask * u = NULL;
	
	na = GDL_GVIEW_ACCESSION_SIZE (g, m);
	nl = GDL_GVIEW_LOCUS_SIZE (g, m);
	
	gbuf = GDL_GVIEW_GET_NEW (g, m);
	
	for (i = 0; i < na; i++)
	{
		for (ni = j = 0; j < nl; j++)
		{
			GDL_GVIEW_GET_GENOTYPE (g, m, i, j, gbuf);
			x = gdl_gvalues_get_gvalues (gbuf);
			if (x==0)
			{
				ni++;	
			}
		}
		if (ni < nl)
		{
			// keep it
			gdl_accession * a = GDL_GVIEW_GET_ACCESSION (g, m, i);
			if (u == 0)
			{
				u = gdl_mask_alloc ();
				if (m)
				{
					gdl_mask_set (u, GDL_LOCUS, gdl_mask_get_clone (m, GDL_LOCUS), 1);
				}
			}
			gdl_mask_add (u, a);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return u;
}

gdl_mask *
gdl_gview_gmask_not_informative_recursive (const gdl_gview * g, const gdl_mask * mask)
{
	size_t nl1, na1, nl2, na2;
	gdl_mask * m1 = NULL, * m2;
	
	nl1 = gdl_gview_locus_size (g);
	
	if (mask)
	{
	   m1 = gdl_mask_clone (mask);
	}
	
	do
	{
		m2  = gdl_gview_gmask_uninformative_locus (g, m1);
		gdl_mask_free (m1);
		nl2 = gdl_mask_get_size (m2, GDL_LOCUS);
		if (nl2 == nl1)
		{
			m1=m2;
			break;
		}
		na2 = gdl_mask_get_size (m2, GDL_ACCESSION);
		m1  = gdl_gview_gmask_uninformative_accession (g, m2);
		gdl_mask_free (m2);
		na1 = gdl_mask_get_size (m1, GDL_ACCESSION);
		if (na1 == na2)
		{
			break;
		}
		nl1 = gdl_mask_get_size (m1, GDL_LOCUS);
	} while (0==0);
	
	return m1;
}


