/*  
 * 	gview/virtual.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_accession.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>
 
#include "ventry.c"

typedef struct
{
	size_t size;
	size_t na;
	size_t nl;
	size_t pl;
	gdl_hashtable * locus;
	gdl_hashtable * accession;
	gdl_hashtable * inputs;
	gdl_list      * inames;
	gdl_locus    ** _locus;
	gdl_accession ** _accession;
	gdl_string    * _gn;
	gdl_gview     * _g;
	gdl_locus     * _l;
	gdl_accession * _a;
	gdl_gvalues_get * _gb;
} gdl_gview_virtual_t;

static int
_gdl_gview_virtual_grow_locus (gdl_gview_virtual_t * v, size_t n)
{
	if (v->_locus == NULL)
	{
		v->_locus = GDL_MALLOC (gdl_locus *, n);
	}
	else
	{
		gdl_locus ** new = GDL_MALLOC (gdl_locus *, n);
		memcpy (new, v->_locus, sizeof(gdl_locus *)*v->nl);
		GDL_FREE (v->_locus);
		v->_locus = new;
	}
	v->nl = n;
	return (0);
}

static int
_gdl_gview_virtual_grow_accession (gdl_gview_virtual_t * v, size_t n)
{
	if (v->_accession == NULL)
	{
		v->_accession = GDL_MALLOC (gdl_accession *, n);
	}
	else
	{
		gdl_accession ** new = GDL_MALLOC (gdl_accession *, n);
		memcpy (new, v->_accession, sizeof(gdl_accession *)*v->na);
		GDL_FREE (v->_accession);
		v->_accession = new;
	}
	v->na = n;
	return (0);
}

typedef struct
{
	size_t * aidx;
	size_t * gidx;
} _virtual_gcode;

static void
_virtual_gcode_free (void * v)
{
	if (v)
	{
		_virtual_gcode * vc = (_virtual_gcode *)v;	
		GDL_FREE (vc->aidx);
		GDL_FREE (vc->gidx);
		GDL_FREE (vc);
	}
}

static gdl_data_interface _virtual_gcode_interface = 
{
	&_virtual_gcode_free,
	NULL,
	NULL,
	NULL,
	NULL
};

static void
_apply_virtual_gcode_genotype (const gdl_locus * loc, gdl_string * name, gdl_gvalues_get * gb)
{
	gdl_hashtable * hash = (gdl_hashtable *) loc->extra;
	_virtual_gcode * vgc = (_virtual_gcode *) gdl_hashtable_lookup (hash, name);
	if (vgc)
	{
		size_t i;
		const gdl_gvalues * x = gdl_gvalues_get_gvalues (gb);
		if (x)
		{
			for (i = 0; i < x->size; i++)
			{
				x->values[i]->idx = vgc->gidx[x->values[i]->idx];
			}
		}
	}
}

static void
_apply_virtual_gcode_allele (const gdl_locus * loc, gdl_string * name, gdl_gvalues_get * gb)
{
	gdl_hashtable * hash = (gdl_hashtable *) loc->extra;
	_virtual_gcode * vgc = (_virtual_gcode *) gdl_hashtable_lookup (hash, name);
	if (vgc)
	{
		size_t i;
		const gdl_gvalues * x = gdl_gvalues_get_gvalues (gb);
		for (i = 0; i < x->size; i++)
		{
			x->values[i]->idx = vgc->aidx[x->values[i]->idx];
		}
	}
}

static _virtual_gcode *
gdl_gview_virtual_locus_update (gdl_locus * vloc, const gdl_locus * loc, gdl_boolean * diff)
{
	size_t i, j, na;
	gdl_allele * a, * va;
	gdl_genotype * g, * vg;
	_virtual_gcode * vgc = GDL_CALLOC (_virtual_gcode, 1);
	
	na = gdl_locus_allele (loc);
	
	if (na)
	{
		vgc->aidx = GDL_MALLOC (size_t, na);
		for (i = 0; i < na; i++)
		{
			a  = gdl_locus_get_allele (loc, i);
			va = gdl_locus_search_allele (vloc, a->name);
			if ((va == 0) || (va && va->idx != a->idx))
			{
					*diff = gdl_true;
			}
			if (va == 0)
			{
				va = gdl_entity_clone (a);
				gdl_locus_add_allele (vloc, &va, 1);
			}
			vgc->aidx[a->idx] = va->idx;
		}
	}
	
	na = gdl_locus_genotype (loc);
	
	if (na)
	{
		vgc->gidx = GDL_MALLOC (size_t, na);
		
		for (i = 0; i < na; i++)
		{
			g  = gdl_locus_get_genotype (loc, i);

            vg = gdl_locus_search_genotype (vloc, g->name);
			
			if ((vg == 0) || (vg && vg->idx != g->idx))
			{
				*diff = gdl_true;
			}
			if (vg == 0)
			{
				vg = gdl_entity_clone (g);
				gdl_locus_add_genotype (vloc, &vg, 1);
			}
			vgc->gidx[g->idx] = vg->idx;
		}
	}
	
	return vgc;
}

static int
_gdl_gview_virtual_add (gdl_gview_virtual_t * v, const gdl_string * name, gdl_gview * g)
{
	size_t n, i, j, gb_size;
	size_t * aidx;
	gdl_boolean diff;
	gdl_string * iname;
	gdl_hashtable * hash;
	gdl_locus * locus;
	gdl_accession * accession;
	gdl_gview_ventry * vloc, * vac;
	_virtual_gcode * vgc;
	
	iname = gdl_string_clone (name);
	
	gdl_list_push_front (v->inames, iname, 1);

	n = gdl_gview_locus_size (g);
	
	gb_size = (v->_gb) ? v->_gb->size : 0;
	
	for (i = 0; i < n; i++)
	{
		locus = gdl_gview_get_locus (g, i);
		vloc  = (gdl_gview_ventry *) gdl_hashtable_lookup (v->locus, locus->name);
		if (vloc == 0)
		{
			vloc              = gdl_gview_ventry_alloc ();
			vloc->entity      = gdl_entity_clone (locus);
			vloc->entity->idx = gdl_hashtable_size (v->locus);
			_gdl_gview_virtual_grow_locus (v, vloc->entity->idx+1);
			v->_locus[vloc->entity->idx] = vloc->entity;
			gdl_hashtable_add (v->locus, locus->name, vloc, 1);
		}
		else
		{
			// check if the allele/genotype integer code
			// is different
			diff = gdl_false;
			vgc  = gdl_gview_virtual_locus_update (vloc->entity, locus, &diff);
			if (diff)
			{
				if (vloc->entity->extra)
				{
					hash = (gdl_hashtable *) vloc->entity->extra;
				}
				else
				{
					hash = gdl_hashtable_alloc (&_virtual_gcode_interface, 0);
					vloc->entity->extra = hash;
				}
				gdl_hashtable_add (hash, iname, vgc, 1);
			}
			else
			{
				_virtual_gcode_free (vgc);
			}		
		}
		gdl_gview_ventry_grow (vloc, vloc->size+1);
		gdl_gview_ventry_set (vloc, vloc->size-1, iname);
		// Check locus size
		j = GDL_MAX (gdl_locus_genotype (vloc->entity),
		             gdl_locus_allele (vloc->entity));
		if (j > gb_size)
		{
			gb_size = j;
		}
	}
	
	n = gdl_gview_accession_size (g);
	
	for (i = 0; i < n; i++)
	{
		accession = gdl_gview_get_accession (g, i);
		vac       = (gdl_gview_ventry *) gdl_hashtable_lookup (v->accession, accession->name);
		if (vac == 0)
		{
			vac = gdl_gview_ventry_alloc ();
			vac->entity      = gdl_entity_clone (accession);
			vac->entity->idx = gdl_hashtable_size (v->accession);
			_gdl_gview_virtual_grow_accession (v, vac->entity->idx+1);
			v->_accession[vac->entity->idx] = vac->entity;
			gdl_hashtable_add (v->accession, accession->name, vac, 1);
		}
		gdl_gview_ventry_grow (vac, vac->size+1);
		gdl_gview_ventry_set (vac, vac->size-1, iname);		
	}
	
	gdl_gvalues_get_free (v->_gb);
	v->_gb = gdl_gvalues_get_alloc (gb_size);
	
	v->size++;
	
	return GDL_SUCCESS;
}

static int 
gdl_gview_virtual_get_gview (gdl_gview_virtual_t * g, gdl_gview_ventry * va, gdl_gview_ventry * vl)
{
	size_t i, ii, j, jj;
	
	g->_g = NULL;
	
	for (i = vl->size; i > 0; i--)
	{
		ii = i - 1;
		for (j = va->size; j > 0; j--)
		{
			jj = j -1;
			if (vl->gviews[ii] == va->gviews[jj])
			{
				g->_gn = vl->gviews[ii];
				g->_g  = gdl_hashtable_lookup (g->inputs, g->_gn);
				return (0); 
			}
		}	
	}	
	return (1);
}

static int
gdl_gview_virtual_attach_gview (gdl_gview_virtual_t * g, const gdl_accession * a, const gdl_locus * l)
{
	gdl_gview_ventry * va = gdl_hashtable_lookup (g->accession, a->name);
	gdl_gview_ventry * vl = gdl_hashtable_lookup (g->locus, l->name);
	
	gdl_gview_virtual_get_gview (g, va, vl);
	
	if (g->_g)
	{
		g->_a = gdl_gview_search_accession (g->_g, a->name);
		g->_l = gdl_gview_search_locus (g->_g, l->name);
	}
	
	return GDL_SUCCESS;
}

static int
gdl_gview_virtual_attach_gview_f (gdl_gview_virtual_t * g, size_t a, size_t l)
{
	return gdl_gview_virtual_attach_gview (g, g->_accession[a], g->_locus[l]);
}

int
gdl_gview_virtual_alloc (void * vg)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	
	g->pl = g->na = g->nl = g->size = 0;
	g->locus      = NULL;
	g->accession  = NULL;
	g->_locus      = NULL;
	g->_accession  = NULL;
	g->inputs     = NULL;
	g->inames    = NULL;
	g->_gb       = NULL;
	g->_g       = NULL;
	g->_gn      = NULL;
	g->_a      = NULL;
	g->_l      = NULL;

	return GDL_SUCCESS;
}

void
gdl_gview_virtual_free (void * vg)
{
	if (vg)
	{
		gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
		
		gdl_gvalues_get_free (g->_gb);
		gdl_hashtable_free (g->locus);
		gdl_hashtable_free (g->accession);
		gdl_hashtable_free (g->inputs);
		gdl_list_free (g->inames);
		GDL_FREE (g->_locus);
		GDL_FREE (g->_accession);
	}	
}

int
gdl_gview_virtual_init (void * vg, size_t na, size_t nl, size_t p)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *)vg;
	
	g->pl        = p;
	g->locus     = gdl_hashtable_alloc (gdl_gview_ventry_interface, 0);
	g->accession = gdl_hashtable_alloc (gdl_gview_ventry_interface, 0);
	g->inputs    = gdl_hashtable_alloc (gdl_gview_interface, 0);
	g->inames    = gdl_list_alloc (gdl_list_default);
	
	return GDL_SUCCESS;
}

size_t
gdl_gview_virtual_locus_size (const void * vg)
{
	return ((gdl_gview_virtual_t *)vg)->nl;
}

size_t
gdl_gview_virtual_accession_size (const void * vg)
{
	return ((gdl_gview_virtual_t *)vg)->na;
}

size_t
gdl_gview_virtual_ploidy (const void * vg)
{
	return ((gdl_gview_virtual_t *)vg)->pl;
}

gdl_locus *
gdl_gview_virtual_add_locus (void * vg, const gdl_string * name)
{
	GDL_ERROR_VAL ("Cannot add a locus to a virtual gview",
	               GDL_EINVAL,
	               0);
}

gdl_accession *
gdl_gview_virtual_add_accession (void * vg, const gdl_string * name)
{
	GDL_ERROR_VAL ("Cannot add a locus to a virtual gview",
	               GDL_EINVAL,
	               0);
}

gdl_locus *
gdl_gview_virtual_get_locus (const void * vg, size_t idx)
{
	return ((gdl_gview_virtual_t *)vg)->_locus[idx];
}

gdl_accession *
gdl_gview_virtual_get_accession (const void * vg, size_t idx)
{
	return ((gdl_gview_virtual_t *)vg)->_accession[idx];
}

gdl_locus *
gdl_gview_virtual_search_locus (const void * vg, const char * name)
{
	gdl_gview_ventry * ve = (gdl_gview_ventry *) gdl_hashtable_lookup (((gdl_gview_virtual_t *)vg)->locus, name);
	if (ve)
	{
		return ve->entity;
	}
	return NULL;
}

gdl_accession *
gdl_gview_virtual_search_accession (const void * vg, const char * name)
{
	gdl_gview_ventry * ve = (gdl_gview_ventry *) gdl_hashtable_lookup (((gdl_gview_virtual_t *)vg)->accession, name);
	if (ve)
	{
		return ve->entity;
	}
	return NULL;
}

int
gdl_gview_virtual_set_gdatapoint (void * vg, const gdl_accession * a, const gdl_locus * l, gdl_gdatapoint * gd)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview_virtual_attach_gview (g, a, l);
	return gdl_gview_set_gdatapoint (g->_g, g->_a, g->_l, gd);
}

gdl_gvalues_get *
gdl_gview_virtual_get_new (const void * gv)
{
	gdl_gvalues_get * get;
	size_t i, j, max=0;
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) gv;
	
	for (i = 0; i < g->nl; i++)
	{
		gdl_locus * vl = gdl_gview_virtual_get_locus(g, i);
		j = GDL_MAX (gdl_locus_allele (vl), gdl_locus_genotype (vl));
		if (j > max)
		{
			max = j;
		}
	}
	return gdl_gvalues_get_alloc (max);
}

int
gdl_gview_virtual_genotype (const void * vg, const gdl_accession * a, const gdl_locus * l, gdl_gvalues_get * gd)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview_virtual_attach_gview (g, a, l);
	if (g->_g)
	{
		gdl_gview_genotype (g->_g, g->_a, g->_l, gd);
		if (!gd->na)
		{
			_apply_virtual_gcode_genotype (l, g->_gn, gd);
		}
	}
	else
	{
		gd->na = gdl_true;
	}
	return GDL_SUCCESS;	
}

int
gdl_gview_virtual_allele (const void * vg, const gdl_accession * a, const gdl_locus * l, size_t p, gdl_gvalues_get * gd)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview_virtual_attach_gview (g, a, l);
	if (g->_g)
	{
		gdl_gview_allele (g->_g, g->_a, g->_l, p, gd);
		if (!gd->na)
		{
			_apply_virtual_gcode_allele (l, g->_gn, gd);
		}
	}
	else
	{
		gd->na = gdl_true;
	}
	return GDL_SUCCESS;	
}

int
gdl_gview_virtual_gtemplate (const void * vg, const gdl_accession * a, const gdl_locus * l, gdl_gvalues_get * gd)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	
	gdl_gview_virtual_attach_gview (g, a, l);
	
	if (g->_g)
	{
		return gdl_gview_genotype_template (g->_g, g->_a, g->_l, gd);
	}
	else 
	{
		size_t j;
		double x;
		
		gd->na = gdl_false;
		gd->x->size = gdl_locus_genotype (l);
		x = 1./(double)gd->x->size;
		
		for (j = 0; j < gd->x->size; j++)
		{
			gd->x->values[j]->idx   = j;
			gd->x->values[j]->value = x;
		}
		
		return GDL_SUCCESS;
	}
}

gdl_boolean
gdl_gview_virtual_has_missing (const void * vg, const gdl_accession * a, const gdl_locus * l)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview_virtual_attach_gview (g, a, l);
	if (g->_g)
	{
		return gdl_gview_has_missing (g->_g, g->_a, g->_l);
	}
	else
	{
		return gdl_true;	
	}
}

gdl_boolean
gdl_gview_virtual_is_missing (const void * vg, const gdl_accession * a, const gdl_locus * l, size_t p)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview_virtual_attach_gview (g, a, l);
	if (g->_g)
	{
		return gdl_gview_is_missing (g->_g, g->_a, g->_l, p);
	}
	else
	{
		return gdl_true;	
	}
}

gdl_boolean
gdl_gview_virtual_is_homozygous (const void * vg, const gdl_accession * a, const gdl_locus * l)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview_virtual_attach_gview (g, a, l);
	if (g->_g)
	{
		return gdl_gview_is_homozygous (g->_g, g->_a, g->_l);
	}
	else
	{
		return gdl_false;	
	}
}

int
gdl_gview_virtual_accession_compare (const void * vg, const gdl_accession * a1, const gdl_accession * a2, const gdl_locus_mask * m)
{
	size_t l, ll, nl;
	const gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	
	nl = (m == 0) ? g->nl : gdl_entity_mask_size (m);
	
	for (l = 0; l < nl; l++)
	{
		ll = (m == 0) ? l : gdl_entity_mask_idx (m, l);
		gdl_locus * loc = gdl_gview_virtual_get_locus (g, ll);
		if (gdl_gview_virtual_genotype_compare (g, a1, loc, a2, loc))
			return (-1);
	}
	
	return (0);
}

int
gdl_gview_virtual_locus_compare (const void * vg, const gdl_locus * l1, const gdl_locus * l2, const gdl_accession_mask * m)
{
	size_t l, ll, na;
	const gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	
	na = (m == 0) ? g->na : gdl_entity_mask_size (m);
	
	for (l = 0; l < na; l++)
	{
		ll = (m == 0) ? l : gdl_entity_mask_idx (m, l);
		gdl_accession * a = gdl_gview_virtual_get_accession (g, ll);
		if (gdl_gview_virtual_genotype_compare (g, a, l1, a, l2))
			return (-1);
	}
	
	return (0);
}

int
gdl_gview_virtual_genotype_compare (const void * vg, const gdl_accession * a1, const gdl_locus * l1, const gdl_accession * a2, const gdl_locus * l2)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gvalues * x1;
	const gdl_gvalues * x2;
	int comp;
	
	gdl_gview_virtual_genotype (g, a1, l1, g->_gb);
	x1 = gdl_gvalues_clone (gdl_gvalues_get_gvalues (g->_gb));
	gdl_gview_virtual_genotype (g, a2, l2, g->_gb);
	x2 = gdl_gvalues_get_gvalues (g->_gb);
	comp = gdl_gvalues_compare (x1, x2);
	gdl_gvalues_free (x1);
	
	return comp;
}

int
gdl_gview_virtual_set_gdatapoint_f (void * vg, size_t i, size_t j, gdl_gdatapoint * gd)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_set_gdatapoint (g, g->_accession[i], g->_locus[j], gd);
}

int
gdl_gview_virtual_genotype_f (const void * vg, size_t i, size_t j, gdl_gvalues_get * gb)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_genotype (g, g->_accession[i], g->_locus[j], gb);
}

	
int
gdl_gview_virtual_allele_f (const void * vg, size_t i, size_t j, size_t k, gdl_gvalues_get * gb)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_allele (g, g->_accession[i], g->_locus[j], k, gb);
}

int
gdl_gview_virtual_gtemplate_f (const void * vg, size_t i, const gdl_locus * locus, gdl_gvalues_get * gb)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_gtemplate (g, g->_accession[i], locus, gb);
}

gdl_boolean
gdl_gview_virtual_is_homozygous_f (const void * vg, size_t i, size_t j)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_is_homozygous (g, g->_accession[i], g->_locus[j]);
}

gdl_boolean
gdl_gview_virtual_has_missing_f (const void * vg, size_t i, size_t j)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_has_missing (g, g->_accession[i], g->_locus[j]);
}

gdl_boolean
gdl_gview_virtual_is_missing_f (const void * vg, size_t i, size_t j, size_t p)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_is_missing (g, g->_accession[i], g->_locus[j], p);
}

int 
gdl_gview_virtual_accession_compare_f (const void * vg, size_t a1, size_t a2, const gdl_locus_mask * m)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_accession_compare (g, g->_accession[a1], g->_accession[a2], m);
}

int
gdl_gview_virtual_locus_compare_f (const void * vg, size_t l1, size_t l2, const gdl_accession_mask * m)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_locus_compare (g, g->_locus[l1], g->_locus[l2], m);
}

int
gdl_gview_virtual_genotype_compare_f (const void * vg, size_t a1, size_t l1, size_t a2, size_t l2)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	return gdl_gview_virtual_genotype_compare (g, g->_accession[a1], g->_locus[l1], g->_accession[a2], g->_locus[l2]);
}

void *
gdl_gview_virtual_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t p, status;
		gdl_gview_virtual_t * v;
		
		status = fread (&p, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		v = GDL_CALLOC (gdl_gview_virtual_t, 1);
		
		gdl_gview_virtual_init (v, 0, 0, p);
		
		status = gdl_hashtable_fread (stream, v->inputs);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		if (gdl_hashtable_size (v->inputs))
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (v->inputs);
			do
			{
				gdl_string * gn = gdl_hashtable_iterator_key (itr);
				gdl_gview  * g  = (gdl_gview *) gdl_hashtable_iterator_value (itr);
				_gdl_gview_virtual_add (v, gn, g);
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
				
		return v;
	}	
}

int
gdl_gview_virtual_fwrite (FILE * stream, void * gv)
{
	if (stream == 0 || gv == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		gdl_gview_virtual_t * v = (gdl_gview_virtual_t *) gv;
		
		status = fwrite (&v->pl, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_hashtable_fwrite (stream, v->inputs);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}	
}

int
gdl_gview_virtual_add (void * vg, const gdl_string * name, void * s, size_t owner)
{
	gdl_gview_virtual_t * g = (gdl_gview_virtual_t *) vg;
	gdl_gview * gv = (gdl_gview *) s;
	_gdl_gview_virtual_add (g, name, gv);
	gdl_hashtable_add (g->inputs, name, gv, owner);
	
	return GDL_SUCCESS;
}

const gdl_gview *
gdl_gview_virtual_get (const gdl_gview * g, const gdl_string * name)
{
	if (g->type != gdl_gview_virtual)
	{
		return NULL;	
	}
	else
	{
		return gdl_hashtable_lookup (((gdl_gview_virtual_t *)g->data)->inputs, name);
	}
}

static const gdl_gview_type _virtual = 
{
	"gdl_gview_virtual",
	sizeof (gdl_gview_virtual_t),
	&gdl_gview_virtual_alloc,
	&gdl_gview_virtual_free,
	&gdl_gview_virtual_init,
	&gdl_gview_virtual_add_locus,
	&gdl_gview_virtual_add_accession,
	&gdl_gview_virtual_ploidy,
	&gdl_gview_virtual_locus_size,
	&gdl_gview_virtual_accession_size,
	&gdl_gview_virtual_get_locus,
	&gdl_gview_virtual_get_accession,
	&gdl_gview_virtual_search_locus,
	&gdl_gview_virtual_search_accession,
	&gdl_gview_virtual_get_new,
	&gdl_gview_virtual_set_gdatapoint,
	&gdl_gview_virtual_genotype,
	&gdl_gview_virtual_allele,
	&gdl_gview_virtual_has_missing,
	&gdl_gview_virtual_is_missing,
	&gdl_gview_virtual_is_homozygous,
	&gdl_gview_virtual_gtemplate,
	&gdl_gview_virtual_accession_compare,
	&gdl_gview_virtual_locus_compare,
	&gdl_gview_virtual_genotype_compare,
	&gdl_gview_virtual_set_gdatapoint_f,
	&gdl_gview_virtual_genotype_f,
	&gdl_gview_virtual_allele_f,
	&gdl_gview_virtual_has_missing_f,
	&gdl_gview_virtual_is_missing_f,
	&gdl_gview_virtual_is_homozygous_f,
	&gdl_gview_virtual_gtemplate_f,
	&gdl_gview_virtual_accession_compare_f,
	&gdl_gview_virtual_locus_compare_f,
	&gdl_gview_virtual_genotype_compare_f,
	&gdl_gview_virtual_fread,
	&gdl_gview_virtual_fwrite,
	&gdl_gview_virtual_add
};

const gdl_gview_type * gdl_gview_virtual = &_virtual;
