/*  
 * 	hash/factory.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_hash.h>

static int
gdl_hashentry_fread (FILE * stream, gdl_hashentry * entry, gdl_data_fread vfread)
{
	size_t status, i, len;
	gdl_hashentry * iter = entry;
	
	status = fread (&len, sizeof(size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	
	if (!len)
	{
		iter->inuse = 0;
	}
	else
	{
		for (i = 0; i < len; i++)
		{
			iter->owner = 1;
			iter->inuse = 1;
			iter->name  = gdl_string_fread (stream);
			GDL_FREAD_STATUS (iter->name != 0, 1);
			iter->value = (vfread)(stream);
			GDL_FREAD_STATUS (iter->value != 0, 1);
			if ( i < len - 1)
			{
				iter->next = GDL_CALLOC (gdl_hashentry, 1);
				iter       = iter->next;
			}
		}
	}
	
    return GDL_SUCCESS;
}

static int
gdl_hashentry_fwrite (FILE * stream, gdl_hashentry * entry, gdl_data_fwrite vwrite)
{
	size_t status, i, len;
	gdl_hashentry * iter = entry;
	
	for (len = 0; iter != NULL && iter->inuse != 0; iter = iter->next, len++);
	
	status = fwrite (&len, sizeof(size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	
	if (len)
	{
		iter = entry;
		for (i = 0; i < len; i++)
		{
			status = gdl_string_fwrite (stream, iter->name);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			status = (vwrite)(stream, iter->value);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			iter = iter->next;			
		}
	}
	
    return GDL_SUCCESS;
}

int
gdl_hashtable_fread (FILE * stream, gdl_hashtable * table)
{
	size_t i, size, n, status;
	gdl_hashentry * etable;
	
	if (stream == 0 || table == 0)
	{
		return GDL_EINVAL;
	}
	if (table->type->fread == 0)
	{
		return GDL_EINVAL;	
	}
	
	// Clean the table previously allocated.
	gdl_hash_internal_table_free (table, table->type->free);
		
	status = fread (&size, sizeof(size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&n, sizeof(size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	etable   = GDL_CALLOC (gdl_hashentry, size);
	if (etable == 0)
	{
		return GDL_ENOMEM;	
	}
	for (i = 0; i < size; i++)
	{
		status = gdl_hashentry_fread (stream, &etable[i], table->type->fread);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
	}
	table->size  = size;
	table->n     = n;
	table->table = etable;
	    
    return GDL_SUCCESS;
}

int
gdl_hashtable_fwrite (FILE * stream, gdl_hashtable * table)
{
	size_t i, status;
	
	if (stream == 0 || table == 0)
	{
		return GDL_EINVAL;	
	}
	if (table->type->fwrite == 0)
	{
		return GDL_EINVAL;	
	}
	
	status = fwrite (&table->size, sizeof(size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&table->n, sizeof(size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	
	for (i = 0; i < table->size; i++)
	{
		status = gdl_hashentry_fwrite (stream, &table->table[i], table->type->fwrite);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
	}
	
	return GDL_SUCCESS;
}
