/*  
 * 	hash/hash.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <string.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>

#include "private.c"

gdl_hashtable *
gdl_hashtable_alloc (const gdl_data_interface * type, size_t size)
{
    gdl_hashtable * table;
  
    if (size <= 0)
        size = 256;
  
    table = GDL_MALLOC (gdl_hashtable, 1);
    
    if (table) {
    	
    	table->type    = type;
        table->size    = size;
	    table->n       = 0;
        table->table   = GDL_CALLOC (gdl_hashentry, size);
        
        if (table->table == 0)
        {
        	GDL_FREE (table);
        }
        else
        {
        	return table;	
        }
    }
    
    return (NULL);
}

void
gdl_hashtable_free (gdl_hashtable * table)
{
    if (table == NULL)
		return;
		
    gdl_hash_internal_table_free (table, table->type->free);
	
	GDL_FREE (table);
}

size_t
gdl_hashtable_add (gdl_hashtable * table,
                         const char * name,
                         void * value,
                         size_t owner)
{
    unsigned long key, len = 0;
    gdl_hashentry * entry;
    gdl_hashentry * insert;

    if ((table == NULL) || (name == NULL))
		return (1);

    key = gdl_hash_key (table, name);
    
    if (table->table[key].inuse == 0)
    {
		insert = NULL;
    }
    else
    {
    	insert = &(table->table[key]);
    	
        for (; insert->next != NULL; insert = insert->next)
        {
			if (!strcmp (insert->name, name))
			{
			    return(1);
			}
			len++;
	    }
	    if (!strcmp (insert->name, name))
	    {
	    	return(1);
	    }
	}

    if (insert == NULL)
    {
		entry = &(table->table[key]);
	}
	else
	{
		entry = GDL_MALLOC (gdl_hashentry, 1);
		if (entry == NULL)
		{
	       return(1);
		}
    }

    entry->name  = gdl_string_clone (name);
    entry->value = value;
    entry->next  = NULL;
    entry->inuse = 1;
    entry->owner = owner;

    if (insert != NULL) 
		insert->next = entry;

    table->n++;
    
    if (len > MAX_HASH_LEN)
		gdl_hash_grow (table, MAX_HASH_LEN * table->size);

    return(0);
}

size_t
gdl_hashtable_add_const (gdl_hashtable * table,
                         const char * name,
                         const void * value)
{
    unsigned long key, len = 0;
    gdl_hashentry * entry;
    gdl_hashentry * insert;

    if ((table == NULL) || (name == NULL))
		return (1);

    key = gdl_hash_key (table, name);
    
    if (table->table[key].inuse == 0)
    {
		insert = NULL;
    }
    else
    {
    	insert = &(table->table[key]);
    	
        for (; insert->next != NULL; insert = insert->next)
        {
			if (!strcmp (insert->name, name))
			{
			    return(1);
			}
			len++;
	    }
	    if (!strcmp (insert->name, name))
	    {
	    	return(1);
	    }
	}

    if (insert == NULL)
    {
		entry = &(table->table[key]);
	}
	else
	{
		entry = GDL_MALLOC (gdl_hashentry, 1);
		if (entry == NULL)
		{
	       return(1);
		}
    }

    entry->name        = gdl_string_clone (name);
    entry->const_value = value;
    entry->next  = NULL;
    entry->inuse = 1;

    if (insert != NULL) 
		insert->next = entry;

    table->n++;
    
    if (len > MAX_HASH_LEN)
		gdl_hash_grow (table, MAX_HASH_LEN * table->size);

    return(0);
}

size_t
gdl_hashtable_update (gdl_hashtable * table,
                               const char * name,
                               void * value,
                               size_t owner)
{
	unsigned long key;
    gdl_hashentry * entry;
    gdl_hashentry * insert;

#define UPDATE_ENTRY { \
			if (!strcmp (insert->name, name)) \
			{ \
		    	if (insert->owner && table->type->free != NULL) \
					(table->type->free)(insert->value); \
		    	insert->value = value; \
		    	return(0); \
			}};

    if ((table == NULL) || name == NULL)
		return(1);

    key = gdl_hash_key (table, name);
    
    if (table->table[key].inuse == 0)
    {
		insert = NULL;
    }
    else
    {
    	insert = &(table->table[key]);
    	
        for (; insert->next != NULL; insert = insert->next)
        {
        	UPDATE_ENTRY
	    }
	    UPDATE_ENTRY
    }

    if (insert == NULL)
    {
		entry =  &(table->table[key]);
    }
    else
    {
		entry = GDL_MALLOC (gdl_hashentry, 1);
		if (entry == NULL)
		     return(1);
    }
	
	entry->name  = gdl_string_clone (name);
    entry->value = value;
    entry->next  = NULL;
    entry->inuse = 1;
    entry->owner = owner;
    table->n++;

    if (insert != NULL)
    {
		insert->next = entry;
    }

#undef UPDATE_ENTRY    
    
    return(0);	
}

void *
gdl_hashtable_lookup (gdl_hashtable * table, const char * name)
{
    unsigned long key;
    gdl_hashentry * entry;

    if (table == NULL)
		return (NULL);
    if (name == NULL)
		return (NULL);
    
    key = gdl_hash_key (table, name);
    
    if (table->table[key].inuse == 0)
		return (NULL);
    
    entry = &(table->table[key]);
    
    for (; entry != NULL; entry = entry->next)
    {
		if (!strcmp (entry->name, name))
		    return (entry->value);
    }
    
    return (NULL);
}

void *
gdl_hashtable_lookup2 (gdl_hashtable * table, const char * name, size_t * owner)
{
    unsigned long key;
    gdl_hashentry * entry;

    if (table == NULL)
		return (NULL);
    if (name == NULL)
		return (NULL);
    
    key = gdl_hash_key (table, name);
    
    if (table->table[key].inuse == 0)
		return (NULL);
    
    entry = &(table->table[key]);
    
    for (; entry != NULL; entry = entry->next)
    {
		if (!strcmp (entry->name, name))
		{
			 *owner = entry->owner;
		    return (entry->value);
		}
    }
    
    return (NULL);
}

size_t
gdl_hashtable_remove (gdl_hashtable * table, const char * name)
{
    unsigned long key;
    gdl_hashentry * entry;
    gdl_hashentry * prev = NULL;

    if (table == NULL || name == NULL)
        return (1);
    
    if (table->n == 0)
    	return (1);

    key = gdl_hash_key (table, name);
        
    if (table->table[key].inuse == 0)
    {
        return (1);
    }
    else
    {
    	entry = &(table->table[key]);
    	
        for (;entry != NULL; entry=entry->next)
        {
            if (!strcmp(entry->name, name))
            {
            	if (table->type->free != NULL && entry->owner)
               {
                    (table->type->free)(entry->value);
               }
                
               entry->value = NULL;
                
               if(prev)
               {
                   prev->next = entry->next;
                   GDL_FREE (entry->name);
			          GDL_FREE (entry);
				   }
				else
				{
		    		if (entry->next == NULL)
		    		{
						entry->inuse=0;
		    		}
		    		else
		    		{
						memcpy(&(table->table[key]), entry->next, sizeof(gdl_hashentry));						
		    		}
				}
                
            table->n--;
                
            return(0);
         }
            
         prev = entry;
            
        }        
        return(1);
    }
}

gdl_hashtable *
gdl_hashtable_clone (gdl_hashtable * table)
{
    size_t i;
    gdl_hashentry * iter;
    gdl_hashtable * clone;

    if (table == NULL)
		return (NULL);
    if (table->type->clone == NULL)
		return (NULL);

    clone = gdl_hashtable_alloc (table->type, table->size);
    
    if (table->table)
    {
		for(i = 0; i < table->size; i++)
		{
	    	if (table->table[i].inuse == 0)
				continue;
				
	    	iter = &(table->table[i]);
	    	
	    	while (iter)
	    	{
				gdl_hashtable_add (clone,
				                   iter->name,
				                   (table->type->clone)(iter->value),
				                   1);
				iter = iter->next;
	    	}
		}
    }
    
    clone->n = table->n;
    
    return(clone);
}

size_t
gdl_hashtable_empty (gdl_hashtable * table)
{
	if (table == NULL)
		return (0);
	return (table->n == 0);	
}


size_t
gdl_hashtable_size (gdl_hashtable * table)
{
    if (table == NULL)
		return (0);
    return (table->n);
}

static gdl_string *
_key (const void * v)
{
	return gdl_string_sprintf ("%p", v);	
}

size_t
gdl_hashtable_vadd (gdl_hashtable * table, const void * key, void * value, size_t owner)
{
	size_t s;
	gdl_string * name = _key (key);
	s = gdl_hashtable_add (table, name, value, owner);
	gdl_string_free (name);
	return s;
}

size_t
gdl_hashtable_vupdate (gdl_hashtable * table, const void * key, void * value, size_t owner)
{
	size_t s;
	gdl_string * name = _key (key);
	s = gdl_hashtable_update (table, name, value, owner);
	gdl_string_free (name);
	return s;
}

void *
gdl_hashtable_vlookup (gdl_hashtable * table, const void * key)
{
	void * s;
	gdl_string * name = _key (key);
	s = gdl_hashtable_lookup (table, name);
	gdl_string_free (name);
	return s;
}

void *
gdl_hashtable_vlookup2 (gdl_hashtable * table, const void * key, size_t * owner)
{
	void * s;
	gdl_string * name = _key (key);
	s = gdl_hashtable_lookup2 (table, name, owner);
	gdl_string_free (name);
	return s;
}

size_t
gdl_hashtable_vremove (gdl_hashtable * table, const void * key)
{
	size_t s;
	gdl_string * name = _key (key);
	s = gdl_hashtable_remove (table, name);
	gdl_string_free (name);
	return s;
}


#include "iterator.c"
#include "factory.c"

const gdl_data_interface * gdl_hash_default = &_gdl_hash_type_default;
