/*  
 * 	hmap/alloc.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:45 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_hmap.h>

struct _gdl_hmap
{
	size_t ng;
	size_t * nc;
	gdl_hview_ptr ** hdata;
	gdl_mask_ptr ** hmask;
	const gdl_gview      * data;
	const gdl_mask      * mask;
	const gdl_gmap       * map;
};

gdl_hmap *
gdl_hmap_alloc (const gdl_gview * g,
                  const gdl_mask * m,
                  const gdl_gmap * map)
{
	size_t i;
	gdl_hmap * h = GDL_CALLOC (gdl_hmap, 1);
	
	h->data = g;
	h->mask = m;
	h->map  = map;
	
	h->ng = gdl_gmap_size (map);
	
	h->nc = GDL_CALLOC (size_t, h->ng);
	h->hdata = GDL_CALLOC (gdl_hview_ptr *, h->ng);
	h->hmask = GDL_CALLOC (gdl_mask_ptr *, h->ng);
	
	for (i = 0; i < h->ng; i++)
	{
		h->nc[i]    = gdl_genome_size ( gdl_gmap_get (map, i) );
		h->hdata[i] = GDL_CALLOC (gdl_hview_ptr, h->nc[i]);
		h->hmask[i] = GDL_CALLOC (gdl_mask_ptr, h->nc[i]);
	}
	
	return h;	
}

int
gdl_hmap_create (gdl_hmap * hmap)
{
	int status;
	size_t i, j, ng, nc;
	
	for (i = 0; i < hmap->ng; i++)
	{
		gdl_genome * genome = gdl_gmap_get (hmap->map, i);
		
		for (j = 0; j < hmap->nc[i]; j++)
		{
			gdl_chromosome * chrom = gdl_genome_get (genome, j);
			gdl_locus_mask * lm    = gdl_chromosome_get_mask (chrom, gdl_mask_get (hmap->mask, GDL_LOCUS));
			gdl_mask * gm         = gdl_mask_alloc ();
			
			gdl_mask_set (gm, GDL_ACCESSION, gdl_mask_get_clone (hmap->mask, GDL_ACCESSION), 1);
			gdl_mask_set (gm, GDL_LOCUS, lm, 1);
			
			hmap->hmask[i][j] = gm;
			hmap->hdata[i][j] = gdl_hview_alloc (hmap->data, gm);
			
			status = gdl_hview_create (hmap->hdata[i][j]);
			
			if (status != GDL_SUCCESS)
			{
				return GDL_FAILURE;	
			}
		}
	}
	
	return GDL_SUCCESS;
}

void
gdl_hmap_free (gdl_hmap * h)
{
	if (h)
	{
		size_t i, j;
		for (i = 0; i < h->ng; i++)
		{
			for (j = 0; j < h->nc[i]; j++)
			{
				gdl_hview_free (h->hdata[i][j]);
				gdl_mask_free (h->hmask[i][j]);
			}
			GDL_FREE (h->hdata[i]);
			GDL_FREE (h->hmask[i]);
		}
		GDL_FREE (h->hdata);
		GDL_FREE (h->hmask);
		GDL_FREE (h->nc);
		GDL_FREE (h);
	}	
}

size_t
gdl_hmap_size (const gdl_hmap * h)
{
	return h->ng;	
}

size_t
gdl_hmap_accession_size (const gdl_hmap * h)
{
	return GDL_GVIEW_ACCESSION_SIZE (h->data, h->mask);	
}

size_t
gdl_hmap_locus_size (const gdl_hmap * h)
{
	return GDL_GVIEW_LOCUS_SIZE (h->data, h->mask);	
}

size_t
gdl_hmap_genome_size (const gdl_hmap * h, size_t genome)
{
	return h->nc[genome];
}

size_t
gdl_hmap_chromosome_size (const gdl_hmap * h, size_t genome, size_t chrom)
{
	gdl_chromosome * c = gdl_genome_get ( gdl_gmap_get (h->map, genome), chrom);
	return gdl_chromosome_size (c);
}

gdl_accession *
gdl_hmap_get_accession (const gdl_hmap * h, size_t a)
{
	return GDL_GVIEW_GET_ACCESSION (h->data, h->mask, a);
}

gdl_locus *
gdl_hmap_get_locus (const gdl_hmap * h, size_t l)
{
	return GDL_GVIEW_GET_LOCUS (h->data, h->mask, l);	
}

const gdl_mask *
gdl_hmap_get_gmask (const gdl_hmap * h)
{
	return h->mask;	
}

gdl_hview *
gdl_hmap_get_hview (const gdl_hmap * h, size_t genome, size_t chrom)
{
	return h->hdata[genome][chrom];
}

const gdl_gview *
gdl_hmap_get_gview (const gdl_hmap * h)
{
	return h->data;
}

const gdl_mask *
gdl_hmap_get_mask (const gdl_hmap * h)
{
	return h->mask;	
}

const gdl_gmap *
gdl_hmap_get_gmap (const gdl_hmap * h)
{
	return h->map;	
}

const size_t * 
gdl_hmap_genome_sizes (const gdl_hmap * h)
{
	return h->nc;	
}

#include "fread.c"

