/*  
 *  hsmap/position.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_gmatrix.h>
#include <gdl/gdl_hsmap_position.h>

gdl_hsmap_position *
gdl_hsmap_position_alloc (const gdl_hsmap_position_type * T)
{
	gdl_hsmap_position * p;
	
	p = GDL_CALLOC (gdl_hsmap_position, 1);
	
	p->type = T;
	
	return p;
}

void
gdl_hsmap_position_free (gdl_hsmap_position * p)
{
	if (p)
	{
		gdl_matrix_free (p->data);
		GDL_FREE (p);	
	}
}

gdl_boolean
gdl_hsmap_position_is_missing (const gdl_hsmap_position * p, size_t i)
{
	if (p->data)
	{
		if (gdl_isnan (gdl_matrix_get (p->data, i, 0)))
		{
			return gdl_true;	
		}
		else
		{
			return gdl_false;
		}	
	}	
	return gdl_true;	
}

struct _gdl_hsmap_position_model
{
	size_t  size;
	size_t _size;
	gdl_hsmap_position ** positions;
};

static void
gdl_hsmap_position_model_grow (gdl_hsmap_position_model * m, size_t size)
{
	if (size && m->_size < size)
	{
		if (m->positions)
		{
			gdl_hsmap_position ** new;
			new = GDL_CALLOC (gdl_hsmap_position *, size);
			memcpy (new, m->positions, sizeof (gdl_hsmap_position *)*m->size);
			GDL_FREE (m->positions);
			m->positions = new;
		}
		else
		{
			m->positions = GDL_CALLOC (gdl_hsmap_position *, size);
		}
		m->_size = size;
	}
}

gdl_hsmap_position_model *
gdl_hsmap_position_model_alloc (void)
{
	gdl_hsmap_position_model * m;
	
	m = GDL_CALLOC (gdl_hsmap_position_model, 1);
	
	return m;	
}

void
gdl_hsmap_position_model_free (gdl_hsmap_position_model * m)
{
	if (m)
	{
		if (m->_size)
		{
			size_t i;
			for (i = 0; i < m->_size; i++)
			{
				gdl_hsmap_position_free (m->positions[i]);	
			}
			GDL_FREE (m->positions);
		}
		GDL_FREE (m);
	}	
}

size_t
gdl_hsmap_position_model_size (const gdl_hsmap_position_model * m)
{
	return m->size;	
}

size_t
gdl_hsmap_position_model_column_size (const gdl_hsmap_position_model * m)
{
	size_t i, n = 0;
	
	for (i = 0; i < m->size; i++)
	{
		n += m->positions[i]->data->size2;
	}
	
	return n;
}

size_t
gdl_hsmap_position_model_add (gdl_hsmap_position_model * m, gdl_hsmap_position * p)
{
	size_t i = m->size;
	
	gdl_hsmap_position_model_grow (m, i+1);
	
	m->positions[i] = p;
	
	m->size = i+1;
	
	return i;
}	

gdl_hsmap_position *
gdl_hsmap_position_model_remove (gdl_hsmap_position_model * m, size_t i)
{
	size_t j;
	gdl_hsmap_position * p;
	
	p = m->positions[i];
	
	for (j = i; j < m->size - 1; j++)
	{
		m->positions[j] = m->positions[j+1];
	}
	(m->size)--;
	
	return p;
}

gdl_hsmap_position *
gdl_hsmap_position_model_get (const gdl_hsmap_position_model * m, size_t i)
{
	return m->positions[i];	
}

static int
gdl_hsmap_position_ancestral_init (gdl_hsmap_position * p, const gdl_hsmap_partition * m, const gdl_mask * mask)
{
	size_t i, j, k, n1, n2, pl;
	double pr, * t;
	gdl_vector * x;
	gdl_vector_view view;
	
	pl = gdl_gview_wrapper_ploidy (m->data);
	n1 = gdl_mask_size (mask, GDL_ACCESSION);
	n2 = gdl_hstruct_partition_result_ancestral_locus_size (m->result, p->locus_idx)-1;
	
	gdl_matrix_free (p->data);
	p->data = NULL;
	
	if (n2)
	{
		p->data = gdl_matrix_calloc (n1, n2);
		if (p->position)
		{
			gdl_hstruct_hmm * hmm;
			hmm = gdl_hstruct_model_partition_get_hmm (m->model);
			for (i = 0; i < n1; i++)
			{
				for (j = 0; j < pl; j++)
				{
					gdl_vector * x = gdl_hstruct_hmm_get_interval_ancestral_proba (hmm, gdl_mask_get_idx (mask, GDL_ACCESSION, i), j, p->locus_idx, p->position);
					for (k = 0; k < n2; k++)
					{
						t = gdl_matrix_ptr (p->data, i, k);
						(*t) += gdl_vector_get (x, k);
					}
					gdl_vector_free (x);
				}
			}
		}
		else
		{
			for (i = 0; i < n1; i++)
			{
				for (k = 0; k < n2; k++)
				{
					for (j = 0; j < pl; j++)
					{
						t = gdl_matrix_ptr (p->data, i, k);
						(*t) += gdl_hstruct_partition_result_ancestral_accession_proba (m->result, k, gdl_mask_get_idx (mask, GDL_ACCESSION, i), j, p->locus_idx);
					}
				}
			}
		}
	}
}

static int
gdl_hsmap_position_locus_init (gdl_hsmap_position * p, const gdl_hsmap_partition * m, const gdl_mask * mask)
{
	size_t i, j, n1, n2;
	
	n1 = gdl_gmatrix_accession_size (m->matrix);
	n2 = gdl_gmatrix_locus_column_size (m->matrix, p->locus_idx);
	i  = gdl_gmatrix_locus_first_column (m->matrix, p->locus_idx);
	
	gdl_matrix_const_view view = gdl_matrix_const_submatrix (gdl_gmatrix_get_matrix (m->matrix), 0, i, n1, n2);
	
	n1 = gdl_mask_size (mask, GDL_ACCESSION);
	
	p->data = gdl_matrix_alloc (n1, n2);
	
	for (i = 0; i < n1; i++)
	{
		for (j = 0; j < n2; j++)
		{
			gdl_matrix_set (p->data, i, j, gdl_matrix_get (&(view.matrix), gdl_mask_get_idx (mask, GDL_ACCESSION, i), j));	
		}
	}
	
	return GDL_SUCCESS;
}

static int
gdl_hsmap_position_mutation_init (gdl_hsmap_position * p, const gdl_hsmap_partition * m, const gdl_mask * mask, size_t k)
{
	
}

int
gdl_hsmap_position_init (gdl_hsmap_position * p, size_t locus_idx, double pos, const gdl_hsmap_partition * m, const gdl_mask * mask, void * extra)
{
	p->locus_idx = locus_idx;
	p->locus = gdl_gview_wrapper_get_locus (m->data, p->locus_idx);
	
	if (p->type == gdl_hsmap_position_ancestral)
	{
		p->position = pos;
		return gdl_hsmap_position_ancestral_init (p, m, mask);	
	}
	else if (p->type == gdl_hsmap_position_locus)
	{
		return gdl_hsmap_position_locus_init (p, m, mask);
	}
	else if (p->type == gdl_hsmap_position_mutation)
	{
		size_t * k = (size_t *) extra;
		p->position = pos;
		return gdl_hsmap_position_mutation_init (p, m, mask, *k);
	}
	
	return -1;
}

static const gdl_hsmap_position_type _gdl_hsmap_position_ancestral = GDL_HSMAP_POSITION_ANCESTRAL;
static const gdl_hsmap_position_type _gdl_hsmap_position_locus = GDL_HSMAP_POSITION_LOCUS;
static const gdl_hsmap_position_type _gdl_hsmap_position_mutation = GDL_HSMAP_POSITION_MUTATION;

const gdl_hsmap_position_type * gdl_hsmap_position_ancestral = &_gdl_hsmap_position_ancestral;
const gdl_hsmap_position_type * gdl_hsmap_position_locus = &_gdl_hsmap_position_locus;
const gdl_hsmap_position_type * gdl_hsmap_position_mutation = &_gdl_hsmap_position_mutation;
