/*  
 *  hsmap/work.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_hmap.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_hstruct_model.h>
#include <gdl/gdl_hstruct_result.h>
#include <gdl/gdl_hsmap.h>

struct _gdl_hsmap_workspace
{
	size_t np;
	const gdl_hstruct_result * result;
	gdl_hstruct_model        * model;
	gdl_hsmap_partition      ** partitions;
	FILE * logger;
};

gdl_hsmap_workspace *
gdl_hsmap_workspace_alloc (const gdl_hstruct_result * result, const gdl_view * view, const gdl_rng * rng)
{
	gdl_hsmap_workspace * w;
	
	w = GDL_CALLOC (gdl_hsmap_workspace, 1);
	
	w->result = result;
	
	w->model  = gdl_hstruct_model_setup (result, view, rng);
	
	if (!w->model)
	{
		GDL_FREE (w);	
		GDL_ERROR_VAL ("Unable to restore hstruct model from result", GDL_FAILURE, 0);
	}
	
	return w;
}
void
gdl_hsmap_workspace_free (gdl_hsmap_workspace * w)
{
	if (w)
	{
		size_t i;
		gdl_hstruct_model_free (w->model);
		if (w->partitions)
		{
			for (i = 0; i < w->np; i++)
			{
				gdl_hsmap_partition_free ((w->partitions)[i]);
			}
			GDL_FREE (w->partitions);
		}
		GDL_FREE (w);	
	}
}

FILE *
gdl_hsmap_workspace_set_logger (gdl_hsmap_workspace * w, FILE * stream)
{
	FILE * old = w->logger;
	
	w->logger = stream;
	
	return old;
}

static int
gdl_hsmap_workspace_trait_perform (const gdl_hsmap_workspace * w, const gdl_fview_wrapper * t, const gdl_fview_wrapper * o, size_t it)
{
	size_t i, j, nc, np;
	const gdl_hstruct_model_chromosome * c;
	const gdl_hstruct_model_partition  * p;
	
	nc = gdl_hstruct_model_size (w->model);
	
	for (i = 0; i < nc; i++)
	{
		 c = gdl_hstruct_model_get_chromosome (w->model, i);
		 
		 np = gdl_hstruct_model_chromosome_size (c);
		 
		 for (j = 0; j < np; j++)
		 {
		 	  if (w->logger)
		 	  {
		 	  		fprintf (w->logger, "--\n");
		 	  		fprintf (w->logger, "Start Haplotype Structure Gene Mapping\n");
		 	  		fprintf (w->logger, "   Chromosome [ %d ] Partition [ %d ]\n", i+1, j+1);
		 	  		fprintf (w->logger, "--\n");
		 	  }
		 	  
		 	  p = gdl_hstruct_model_chromosome_get_partition (c, j);
		 }
	}	
}

int
gdl_hsmap_workspace_init_partition (gdl_hsmap_workspace * w, gdl_boolean walk)
{
	size_t i, ii, j, jj, k, ng, nc, np, na;
	const gdl_hstruct_model_chromosome * c;
	const gdl_hstruct_model_partition  * p;
	const gdl_hstruct_partition_result * r;
	const gdl_hstruct_genome_result * gr;
	const gdl_hstruct_chromosome_result * cr;
	
	w->np = 0;
	
	nc = gdl_hstruct_model_size (w->model);
	
	for (i = 0; i < nc; i++)
	{
		 c      = gdl_hstruct_model_get_chromosome (w->model, i);
		 w->np += gdl_hstruct_model_chromosome_size (c);
	}
	
	w->partitions = GDL_MALLOC (gdl_hsmap_partition *, w->np);
	
	ng = gdl_hstruct_result_size (w->result);
	
	for (k = ii = i = 0; i < ng; i++)
	{
		 gr = gdl_hstruct_result_genome (w->result, i);
		 nc = gdl_hstruct_genome_result_size (gr);
		 
		 for (j = 0; j < nc; j++, ii++)
		 {
		 	c  = gdl_hstruct_model_get_chromosome (w->model, ii);
		 	cr = gdl_hstruct_genome_result_chromosome (gr, j);
		 	
		 	np = gdl_hstruct_model_chromosome_size (c);
		 
			 for (jj = 0; jj < np; jj++, k++)
			 {
			 	  if (w->logger)
			 	  {
			 	  		fprintf (w->logger, "   Check Chromosome [ %d ] Partition [ %d ]\n", i+1, j+1);
			 	  }
			 	  
			 	  p = gdl_hstruct_model_chromosome_get_partition (c, jj);
			 	  r = gdl_hstruct_chromosome_result_partition (cr, jj);
			 	  
			 	  if (walk)
			 	  {
			 	  		gdl_hstruct_hmm * hmm = gdl_hstruct_model_partition_get_hmm (p);
			 	  		if (w->logger)
					 	{
					 		fprintf (w->logger, "   Backup Forward-Backward Variables: wait...");
					 	}
			 	  		gdl_hstruct_hmm_store_fb_probs (hmm);
			 	  		if (w->logger)
					 	{
					 		fprintf (w->logger, "\b\b\b\b\b\b\b[ OK ]\n");
					 	}		 	  	
			 	  }
			 	  
			 	  w->partitions[k] = gdl_hsmap_partition_alloc (r, p);
			 	  
			 	  if (k)
			 	  {
			 	  		gdl_hsmap_partition * p1 = w->partitions[k-1];
			 	  		gdl_hsmap_partition * p2 = w->partitions[k];
			 	  		
			 	  		na = gdl_view_generic_wrapper_shared_accession (p1->data, GDL_LOCUS, &(p1->mask), p2->data, GDL_LOCUS, &(p2->mask));
			 	  		
			 	  		if (!na)
			 	  		{
			 	  			GDL_ERROR_VAL ("No accession in common between two consecutive partitions", GDL_EINVAL, GDL_EINVAL);	
			 	  		}
			 	  }
			 }
		 }
	}
	
	for (i = w->np - 1; i > 0; i--)
	{
		gdl_hsmap_partition * p1 = w->partitions[i];
  		gdl_hsmap_partition * p2 = w->partitions[i-1];
  		gdl_view_generic_wrapper_shared_accession (p1->data, GDL_LOCUS, &(p1->mask), p2->data, GDL_LOCUS, &(p2->mask));
  	}
}

int
gdl_hsmap_workspace_init_shared (gdl_hsmap_workspace * w, const gdl_fview_wrapper * trait)
{
	size_t i, old, na;
	gdl_mask * tmask;
	
	for (i = 0; i < w->np; i++)
	{
		tmask = gdl_mask_alloc ();
		na    = gdl_view_wrapper_shared_accession (w->partitions[i]->data, &(w->partitions[i]->mask), trait, &tmask);
		if (!na)
		{
			GDL_ERROR_VAL ("No accession in common between partition and trait", GDL_EINVAL, GDL_EINVAL);
		}
		else if (w->logger)
		{
			fprintf (w->logger, "   Number of common accession in Partition [ %d ]: %d\n", i+1, na);
		}
		w->partitions[i]->tmask = tmask;
	}
}

static int
gdl_hsmap_workspace_perform_test (gdl_hsmap_workspace * w, const gdl_fview_wrapper * trait, size_t t, const gdl_fview_wrapper * other, const gdl_hsmap_params * params)
{
	size_t i, j, idx, nl;
	gdl_hsmap_partition * p;
	gdl_hsmap_model * m;
	
	for (i = 0; i < w->np; i++)
	{
		p = w->partitions[i];
		
		if (w->logger)
		{
			fprintf (w->logger, "--\n");
			fprintf (w->logger, " Test Partition [ %d ]\n", i+1);
			fprintf (w->logger, "--\n");
		}
		
		m = gdl_hsmap_model_alloc (&p, 1, trait, t, other);
		
		nl = gdl_hstruct_model_partition_size (p->model);
		
		for (j = 0; j < nl; j++)
		{
			gdl_hsmap_position * ancestral;
			gdl_hsmap_position * locus;
			
			ancestral = gdl_hsmap_model_new_ancestral_position (m, 0, j, 0.);
			
			if (ancestral)
			{
			
				idx = gdl_hsmap_model_add_eligible (m, ancestral);
				
				gdl_hsmap_model_eval (m);
				
				if (w->logger)
				{
					gdl_hsmap_model_fprintf (w->logger, m);
				}
			
			}
			
			locus = gdl_hsmap_model_new_locus_position (m, 0, j);
			
			idx = gdl_hsmap_model_add_eligible (m, locus);
			
			gdl_hsmap_model_eval (m);
			
			if (w->logger)
			{
				gdl_hsmap_model_fprintf (w->logger, m);
			}
			
			gdl_hsmap_model_rmv_eligible (m, idx);
			gdl_hsmap_model_rmv_eligible (m, idx-1);
			
			gdl_hsmap_position_free (locus);
			gdl_hsmap_position_free (ancestral);
			
			if (ancestral && (params->walk && j < nl - 1))
			{
				gdl_gdistance * interval;
				
				interval = gdl_hstruct_model_partition_get_distance (p->model, j);
				
				if (interval)
				{
					gdl_gmap_interval_itr * itr;
					double cumul = 0;
					
					itr = gdl_gmap_interval_iterator (interval, params->physic);
					
					if (itr)
					{
						do
						{
							double pos = gdl_gmap_interval_iterator_position (itr);
							
							cumul += pos;
							
							ancestral = gdl_hsmap_model_new_ancestral_position (m, 0, j, pos);
				
							idx = gdl_hsmap_model_add_eligible (m, ancestral);
							
							gdl_hsmap_model_eval (m);
							
							if (w->logger)
							{
								gdl_hsmap_model_fprintf (w->logger, m);
							}
							
							gdl_hsmap_model_rmv_eligible (m, idx);
							
							gdl_hsmap_position_free (ancestral);
						}
						while (gdl_gmap_interval_iterator_next (itr));
						
						gdl_gmap_interval_iterator_free (itr);
						
					}
										
					gdl_gdistance_free (interval);
					
				}
			}
		}
		
		gdl_hsmap_model_free (m);
	}
}

int
gdl_hsmap_workspace_perform (gdl_hsmap_workspace * w, const gdl_fview_wrapper * trait, const gdl_fview_wrapper * other, const gdl_hsmap_params * params)
{
	size_t t, nt;
	gdl_mask * tmask, * mask;
	
	if (!trait)
	{
		GDL_ERROR_VAL ("No trait data defined", GDL_EINVAL, GDL_EINVAL);
	}
	
	if (w->logger)
	{
		fprintf (w->logger, "--\n");
		fprintf (w->logger, "Check partitions and create the partition manager: wait...\n");
		fprintf (w->logger, "--\n");
	}
	
	gdl_hsmap_workspace_init_partition (w, params->walk);
	
	if (w->logger)
	{
		fprintf (w->logger, "--\n");
		fprintf (w->logger, "Build the common accession masks over partition: wait...\n");
		fprintf (w->logger, "--\n");
	}
	
	gdl_hsmap_workspace_init_shared (w, trait);
	
	nt = gdl_fview_wrapper_factor_size (trait);
	
	for (t = 0; t < nt; t++)
	{
		if (w->logger)
		{
			gdl_factor * factor = gdl_fview_wrapper_get_factor (trait, t);
			fprintf (w->logger, "--\n");
			fprintf (w->logger, "Trait [ %d ] [ %s ]\n", t+1, gdl_entity_get_name (factor));
			fprintf (w->logger, "--\n");
		}
		
		if (params->mode == gdl_hsmap_test)
		{
			gdl_hsmap_workspace_perform_test (w, trait, t, other, params);
		}
		else if (params->mode == gdl_hsmap_forward_backward)
		{
			
		}
	}
	
	return GDL_SUCCESS;
}

enum _gdl_hsmap_mode 
{
	GDL_HSMAP_TEST,
	GDL_HSMAP_FORWARD_BACKWARD
};

static const gdl_hsmap_mode _gdl_hsmap_test = GDL_HSMAP_TEST;
static const gdl_hsmap_mode _gdl_hsmap_forward_backward = GDL_HSMAP_FORWARD_BACKWARD;

const gdl_hsmap_mode * gdl_hsmap_test = &_gdl_hsmap_test;
const gdl_hsmap_mode * gdl_hsmap_forward_backward = &_gdl_hsmap_forward_backward;

