/*  
 * 	hstruct/chrom.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gpartition.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_fuzzy.h>
#include <gdl/gdl_hstruct_hmm.h>
#include <gdl/gdl_hstruct_parameters.h>
#include <gdl/gdl_hstruct_point.h>
#include <gdl/gdl_hstruct_config.h>
#include <gdl/gdl_hstruct_model_chromosome.h>
#include <gdl/gdl_hstruct_model_partition.h>
#include <gdl/gdl_hstruct_result.h>

struct _gdl_hstruct_model_chromosome
{
	size_t idx;
	const gdl_gligation_type * ltype;
	const gdl_hstruct_data * data;
	gdl_mask                     * gmask;
	gdl_gpartition               * partition;
	gdl_list                     * configs;
	gdl_hashtable                * config_table;
	gdl_hstruct_model_partition  * uniq;
};

gdl_hstruct_model_chromosome *
gdl_hstruct_model_chromosome_alloc (const gdl_hstruct_data * d, size_t idx)
{
	gdl_hstruct_model_chromosome * m;
	
	m = GDL_CALLOC (gdl_hstruct_model_chromosome, 1);
	
	m->idx          = idx;
	m->data         = d;
	m->configs      = gdl_list_alloc (gdl_list_default);
	m->config_table = gdl_hashtable_alloc (gdl_hash_default, 1);
	
	return m;
}

static void
_gdl_hstruct_model_chromosome_gpartition_free (void * data)
{
	gdl_hstruct_model_partition * p = (gdl_hstruct_model_partition *) data;
	gdl_hstruct_model_partition_free (p);
}

void
gdl_hstruct_model_chromosome_free (gdl_hstruct_model_chromosome * m)
{
	if (m)
	{
		gdl_hstruct_model_partition_free (m->uniq);
		gdl_gpartition_free (m->partition, &_gdl_hstruct_model_chromosome_gpartition_free);
		gdl_list_free (m->configs);
		gdl_hashtable_free (m->config_table);
		gdl_mask_free (m->gmask);
		GDL_FREE (m);
	}	
}

static void
_gdl_hstruct_model_chromosome_build_gmask (gdl_hstruct_model_chromosome * m)
{
	m->gmask = gdl_mask_alloc ();
		
	gdl_mask_set (m->gmask, GDL_LOCUS, gdl_chromosome_get_mask (m->data->chrom, NULL), 1);
	
	gdl_mask_set (m->gmask, GDL_ACCESSION, gdl_mask_get_clone (m->data->gmask, GDL_ACCESSION), 1);
}

int
gdl_hstruct_model_chromosome_init (gdl_hstruct_model_chromosome * m, size_t k)
{
	_gdl_hstruct_model_chromosome_build_gmask (m);
		
	m->uniq = gdl_hstruct_model_partition_alloc (m->data, m->gmask, m->idx, 0, 0);
	
	return GDL_SUCCESS;
}

static gdl_gpartition_state **
_create_partition (gdl_hstruct_model_chromosome * m, gdl_hstruct_partition_fonction f, void * extra, size_t * np)
{
	size_t i, j;
	gdl_mask ** pmask;
	gdl_gpartition_state ** pstate;
	
	pmask = (f)(m->data, m->data->chrom, extra, np);
	
	if (pmask)
	{
		
		pstate = GDL_MALLOC (gdl_gpartition_state *, *np);
		
		for (j = i = 0; i < *np; i++)
		{
			pstate[i]       = GDL_MALLOC (gdl_gpartition_state, 1);
			pstate[i]->mask = pmask[i];
			pstate[i]->data = gdl_hstruct_model_partition_alloc (m->data, pmask[i], m->idx, i, j);
			j += gdl_mask_get_size (pmask[i], GDL_LOCUS);
		}
		
		GDL_FREE (pmask);
	}
	else
	{
		pstate = NULL;	
	}
	
	return pstate;
}

int
gdl_hstruct_model_chromosome_init_pl (gdl_hstruct_model_chromosome * m, size_t k, gdl_hstruct_partition_fonction f, void * extra, const gdl_gligation_type * L)
{
	size_t np = 0;
	gdl_gpartition_state ** pstate;
	
	_gdl_hstruct_model_chromosome_build_gmask (m);
	
	pstate = _create_partition (m, f, extra, &np);
	
	if (pstate)
	{
		m->partition = gdl_gpartition_new (pstate, np);
	}
	else
	{
		GDL_ERROR_VAL ("Unable to create partition", GDL_FAILURE, GDL_FAILURE);	
	}
	
	m->ltype = L;
	
	return GDL_SUCCESS;
}

const gdl_mask *
gdl_hstruct_model_chromosome_gmask (const gdl_hstruct_model_chromosome * m)
{
	return m->gmask;	
}

size_t
gdl_hstruct_model_chromosome_size (const gdl_hstruct_model_chromosome * m)
{
	if (m->partition)
	{
		return gdl_gpartition_size (m->partition);
	}
	else
	{
		return 1;
	}
}

size_t
gdl_hstruct_model_chromosome_eligible_size (const gdl_hstruct_model_chromosome * m, const gdl_gligation_type * T)
{
	if (m->partition)
	{
		return gdl_gpartition_eligible_size (m->partition, (T) ? T : m->ltype);
	}
	else
	{
		return 1;
	}
}

gdl_hstruct_model_partition *
gdl_hstruct_model_chromosome_get_partition (const gdl_hstruct_model_chromosome * m, size_t p)
{
	if (m->partition)
	{
		gdl_gpartition_state * gs = gdl_gpartition_get (m->partition, p);
		return (gdl_hstruct_model_partition *)gs->data;
	}
	else
	{
		return m->uniq;
	}
}

int
gdl_hstruct_model_chromosome_fuzzy2hmm (const gdl_hstruct_model_chromosome * m)
{
	if (m->partition)
	{
		size_t p, np;
		
		np = gdl_hstruct_model_chromosome_size (m);
		
		for (p = 0; p < np; p++)
		{
			gdl_gpartition_state * gs = gdl_gpartition_get (m->partition, p);
		    gdl_hstruct_model_partition_fuzzy2hmm ((gdl_hstruct_model_partition *)gs->data);
		}
		
		return GDL_SUCCESS;
	}
	else
	{
		return gdl_hstruct_model_partition_fuzzy2hmm (m->uniq);
	}	
}

double
gdl_hstruct_model_chromosome_loglikelihood (const gdl_hstruct_model_chromosome * m)
{
	double y = 0.;
	
	if (m->partition)
	{
		size_t p, np;
		
		np = gdl_hstruct_model_chromosome_size (m);
		
		for (p = 0; p < np; p++)
		{
			gdl_gpartition_state * gs = gdl_gpartition_get (m->partition, p);
		    y += gdl_hstruct_model_partition_loglikelihood ((gdl_hstruct_model_partition *)gs->data);
		}		
	}
	else
	{
		y = gdl_hstruct_model_partition_loglikelihood (m->uniq);
	}
	
	return y;
}

static void
_partition_free (void * vp)
{
	gdl_hstruct_model_partition * p;
	
	p = (gdl_hstruct_model_partition *) vp;
	
	gdl_hstruct_model_partition_free (p);
}

static int
_partition_ligation (gdl_gpartition_state * cur, const gdl_gpartition_state * next, const gdl_gligation_type * T, void * extra)
{
	gdl_mask * gmask;
	gdl_hstruct_model_partition * p1, * p2;
	gdl_hstruct_parameters * pr = (gdl_hstruct_parameters *) extra;
	
	p1 = (gdl_hstruct_model_partition *) cur->data;
	p2 = (gdl_hstruct_model_partition *) next->data;
	
	printf ("LIGATION [%d, %d]\n", gdl_hstruct_model_partition_get_pidx (p1), gdl_hstruct_model_partition_get_pidx (p2));
	
	gmask = gdl_hstruct_model_partition_ligation (pr, p1, p2, T, NULL);
	
	gdl_mask_free (cur->mask);
	
	cur->mask = gmask;
	cur->data = p1;
	
	return GDL_SUCCESS;
}

int _partition_update (gdl_gpartition_state * cur, const gdl_gligation_type * T, void * extra);

int
gdl_hstruct_model_chromosome_ligation (gdl_hstruct_parameters * pr, gdl_hstruct_model_chromosome * m, const gdl_gligation_type * T)
{
	if (m->partition)
	{
		return gdl_gpartition_ligation (m->partition, (T) ? T : m->ltype, &_partition_ligation, &_partition_free, &_partition_update, pr);
	}
	return GDL_SUCCESS;
}

const gdl_allele *
gdl_hstruct_model_chromosome_get_ancestral (const gdl_hstruct_model_chromosome * m, size_t k, size_t l)
{
	int aidx       = -1;
	gdl_locus * locus = NULL;
	
	if (m->partition)
	{
		size_t p, np, npl, tnl=0;
		np = gdl_hstruct_model_chromosome_size (m);
		for (p = 0; p < np; p++)
		{
			gdl_gpartition_state * gs = gdl_gpartition_get (m->partition, p);
		   npl = gdl_hstruct_model_partition_size ((gdl_hstruct_model_partition *)gs->data);
		   if (l < tnl+npl)
		   {
		   	aidx  = gdl_hstruct_model_partition_get_ancestral ((gdl_hstruct_model_partition *)gs->data, k, l-tnl);
		   	locus = gdl_hstruct_model_partition_get_locus ((gdl_hstruct_model_partition *)gs->data, l-tnl);
		   	break;
		   }
		   tnl += npl;
		}	
	}
	else
	{
		aidx  = gdl_hstruct_model_partition_get_ancestral (m->uniq, k, l);
		locus = gdl_hstruct_model_partition_get_locus (m->uniq, l);
	}
	if (aidx >=0 && locus)
	{
		gdl_allele * a = gdl_locus_get_allele (locus, aidx);
		return gdl_locus_get_allele (locus, aidx);
	}
	else
	{
		return NULL;	
	}
}

void
gdl_hstruct_model_chromosome_set_ancestral (const gdl_hstruct_model_chromosome * m, size_t k, size_t l, size_t a)
{
	if (m->partition)
	{
		size_t p, np, npl, tnl=0;
		np = gdl_hstruct_model_chromosome_size (m);
		for (p = 0; p < np; p++)
		{
			gdl_gpartition_state * gs = gdl_gpartition_get (m->partition, p);
		   npl = gdl_hstruct_model_partition_size ((gdl_hstruct_model_partition *)gs->data);
		   if (l < tnl+npl)
		   {
		   	return gdl_hstruct_model_partition_set_ancestral ((gdl_hstruct_model_partition *)gs->data, k, l-tnl, a);
		   }
		   tnl += npl;
		}		
	}
	else
	{
		return gdl_hstruct_model_partition_set_ancestral (m->uniq, k, l, a);
	}
	
}

int gdl_hstruct_model_partition_setup (gdl_hstruct_model_partition * p, const gdl_hstruct_partition_result * pr, gdl_hstruct_parameters * params);

static gdl_hstruct_model_partition *
gdl_hstruct_model_chromosome_setup_partition (gdl_hstruct_model_chromosome * m, const gdl_hstruct_partition_result * pr, gdl_hstruct_parameters * params, size_t i, size_t j)
{
	const gdl_mask * mask;
	gdl_hstruct_model_partition     * part;
	
	mask = gdl_hstruct_partition_result_gmask (pr);
	
	part = gdl_hstruct_model_partition_alloc (m->data, mask, m->idx, i, j);
	
	gdl_hstruct_model_partition_setup (part, pr, params);
	
	return part;
}

static gdl_gpartition_state **
gdl_hstruct_model_chromosome_setup_pl (gdl_hstruct_model_chromosome * m, const gdl_hstruct_chromosome_result * r, gdl_hstruct_parameters * params)
{
	size_t i, j, np;
	gdl_gpartition_state     ** pstate;
	const gdl_hstruct_partition_result * pr; 
	
	np = gdl_hstruct_chromosome_result_size (r);
	
	pstate = GDL_MALLOC (gdl_gpartition_state *, np);
	
	for (j = i = 0; i < np; i++)
	{
		pr = gdl_hstruct_chromosome_result_partition (r, i);
		
		pstate[i]       = GDL_MALLOC (gdl_gpartition_state, 1);
		pstate[i]->mask = gdl_mask_clone (gdl_hstruct_partition_result_gmask (pr));
		pstate[i]->data = gdl_hstruct_model_chromosome_setup_partition (m, pr, params, i, j);
		
		j += gdl_mask_get_size (pstate[i]->mask, GDL_LOCUS);
	}
	
	return pstate;
}

int
gdl_hstruct_model_chromosome_setup (gdl_hstruct_model_chromosome * m, const gdl_hstruct_chromosome_result * cr, gdl_hstruct_parameters * params)
{
	size_t i, np;
	const gdl_hstruct_partition_result * pr;
	
	_gdl_hstruct_model_chromosome_build_gmask (m);
	
	np = gdl_hstruct_chromosome_result_size (cr);
	
	if (np > 1)
	{
		m->partition = gdl_gpartition_new (gdl_hstruct_model_chromosome_setup_pl (m, cr, params), np);
	}
	else
	{
		m->uniq = gdl_hstruct_model_chromosome_setup_partition (m, gdl_hstruct_chromosome_result_partition (cr, 0), params, 0, 0);
	}
	
	return GDL_SUCCESS;
}
