/*  
 *  hstruct/model.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_sort.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_fuzzy.h>
#include <gdl/gdl_hstruct_data.h>
#include <gdl/gdl_hstruct_point.h>
#include <gdl/gdl_hstruct_config.h>
#include <gdl/gdl_hstruct_parameters.h>
#include <gdl/gdl_hstruct_model_type.h>
#include <gdl/gdl_hstruct_model_chromosome.h>
#include <gdl/gdl_hstruct_model_partition.h>
#include <gdl/gdl_hstruct_model.h>

struct _gdl_hstruct_model_default_config
{
	gdl_hstruct_config * physic;
	gdl_hstruct_config * genetic;
	gdl_hashtable * physic_type;
	gdl_hashtable * genetic_type;
};

struct _gdl_hstruct_model
{
	gdl_hstruct_data * data;
	const gdl_gmap * gmap;
	const gdl_hmap * hmap;
	gdl_hstruct_parameters       * parameters;
	gdl_hstruct_data             ** cdata;
	gdl_hstruct_model_chromosome ** chroms;
	struct _gdl_hstruct_model_default_config       * default_config;
	size_t tnc;
	size_t k;
};

struct _gdl_hstruct_model_mode
{
	gdl_string * name;
	void (*alloc_default_config)(struct _gdl_hstruct_model_default_config **);
	void (*free_default_config)(struct _gdl_hstruct_model_default_config **);
	void (*get_default_config)(struct _gdl_hstruct_model_default_config **);
};

static gdl_hstruct_config *
_get_default_config (gdl_hstruct_model * model, const gdl_gdistance * d, const gdl_locus_type * type, struct _gdl_hstruct_model_default_config * dc)
{   
	switch (d->type)
	{
		case gdl_gdistance_kilo_base :
		case gdl_gdistance_base :
			if (!type && dc->physic == 0)
			{
				dc->physic = gdl_hstruct_parameters_new_config (model->parameters);
			}
			else if (type)
			{
				gdl_hstruct_config * c;
				if (!dc->physic_type)
				{
					dc->physic_type = gdl_hashtable_alloc (gdl_hash_default, 0);
				}
				c = (gdl_hstruct_config *) gdl_hashtable_lookup (dc->physic_type, type->name);
				if (!c)
				{
					void * p;
					if (!dc->physic)
					{
						dc->physic = gdl_hstruct_parameters_new_config (model->parameters);
					}
					c = gdl_hstruct_parameters_clone_config (model->parameters, dc->physic);
					p = gdl_hstruct_parameters_new_point (model->parameters, gdl_hstruct_point_mu, 0);
					gdl_hstruct_config_set_point (c, gdl_hstruct_point_mu, p);
					gdl_hashtable_add (dc->physic_type, type->name, c, 0);
				}
				return c;
			}
			return dc->physic;
		case gdl_gdistance_morgan :
		case gdl_gdistance_centi_morgan :
			if (!type && dc->genetic == 0)
			{
				dc->genetic = gdl_hstruct_parameters_new_config (model->parameters);
			}
			else if (type)
			{
				gdl_hstruct_config * c;
				if (!dc->genetic_type)
				{
					dc->genetic_type = gdl_hashtable_alloc (gdl_hash_default, 0);
				}
				c = (gdl_hstruct_config *) gdl_hashtable_lookup (dc->genetic_type, type->name);
				if (!c)
				{
					void * p;
					
					if (!dc->genetic)
					{
						dc->genetic = gdl_hstruct_parameters_new_config (model->parameters);
					}
					c = gdl_hstruct_parameters_clone_config (model->parameters, dc->genetic);
					p = gdl_hstruct_parameters_new_point (model->parameters, gdl_hstruct_point_mu, 0);
					gdl_hstruct_config_set_point (c, gdl_hstruct_point_mu, p);
					gdl_hashtable_add (dc->genetic_type, type->name, c, 0);
				}
				return c;
			}
			return dc->genetic;
		default:
		    GDL_ERROR_NULL ("Undefined chromosome distance type", GDL_FAILURE);
	}
}

static void
_gdl_hstruct_model_init_partition_clean_default_config (gdl_hstruct_parameters * r, struct _gdl_hstruct_model_default_config * dc)
{
	gdl_hashtable_itr * itr;
	gdl_hstruct_parameters_clean_config (r, dc->physic);
	gdl_hstruct_parameters_clean_config (r, dc->genetic);
	if (dc->physic_type && gdl_hashtable_size (dc->physic_type))
	{
		itr = gdl_hashtable_iterator (dc->physic_type);
		do
		{
			gdl_hstruct_config * c = (gdl_hstruct_config *) gdl_hashtable_iterator_value (itr);
			gdl_hstruct_parameters_clean_config (r, c);
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	if (dc->genetic_type && gdl_hashtable_size (dc->genetic_type))
	{
		itr = gdl_hashtable_iterator (dc->genetic_type);
		do
		{
			gdl_hstruct_config * c = (gdl_hstruct_config *) gdl_hashtable_iterator_value (itr);
			gdl_hstruct_parameters_clean_config (r, c);
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
}

static int
_gdl_hstruct_model_init_partition (gdl_hstruct_model * model,
                                    gdl_hstruct_model_partition * pmodel,
                                    const gdl_hstruct_model_mode * mode,
                                    size_t k)
{
	size_t l, nl, from, to;
	const gdl_chromosome * chrom;
	const gdl_gdistance  * d;
	const gdl_locus_type * type;
	gdl_hstruct_config * conf;
	gdl_hstruct_point_handler * handler;
	gdl_hstruct_config_block_itr * itr;
	
	(mode->get_default_config)(&(model->default_config));
	
	nl = gdl_hstruct_model_partition_size (pmodel);
	
	for (l = 0; l < nl; l++)
	{
		type = gdl_hstruct_model_partition_get_locus_type (pmodel, l);
		if (l < nl - 1)
		{
			d = gdl_hstruct_model_partition_get_distance (pmodel, l);
		}
		conf = _get_default_config (model, d, type, model->default_config);
		gdl_hstruct_model_attach_config (model, pmodel, conf, l);
	}
	
	_gdl_hstruct_model_init_partition_clean_default_config (model->parameters, model->default_config);
		
	gdl_hstruct_model_partition_init (pmodel, k);
	
	// Now set a specific F point for the partition
	handler = gdl_hstruct_parameters_new_handler (model->parameters, gdl_hstruct_point_f, 0);
	itr = gdl_hstruct_model_partition_config_block_iterator (pmodel, 0);
	do
	{
		gdl_hstruct_config_block  * block   = gdl_hstruct_config_block_iterator_get (itr);
		gdl_hstruct_config_buffer * buffer  = gdl_hstruct_config_block_buffer (block);
		
		gdl_hstruct_config_buffer_set (buffer, gdl_hstruct_point_f, gdl_hstruct_point_handler_point_ptr (handler));
		
		from = gdl_hstruct_config_block_from (block);
		to   = gdl_hstruct_config_block_to (block);
		
		for (l = from; l <= to; l++)
		{
			gdl_hstruct_model_partition_update (pmodel, model->parameters, buffer, l);
		}
		
		gdl_hstruct_config_block_update (model->parameters, block);
		
	}
	while (gdl_hstruct_config_block_iterator_next (itr, gdl_true));
	gdl_hstruct_config_block_iterator_free (itr);
	
	return GDL_SUCCESS;
}

static int
_gdl_hstruct_model_init_chromosome (gdl_hstruct_model * model, 
                                     const gdl_hstruct_model_mode * mode,
                                     gdl_hstruct_model_chromosome * cmodel,
                                     size_t k)
{
	gdl_hstruct_model_partition  * pmodel;
	
	gdl_hstruct_model_chromosome_init (cmodel, k);
	
	pmodel = gdl_hstruct_model_chromosome_get_partition (cmodel, 0);
	
	return _gdl_hstruct_model_init_partition (model, pmodel, mode, k);
}

static int
_gdl_hstruct_model_init_pl_chromosome (gdl_hstruct_model * model,
                                        gdl_hstruct_model_chromosome * cmodel,
                                        const gdl_hstruct_model_mode * mode,
                                        size_t k,
                                        gdl_hstruct_partition_fonction f,
                                        void * extra,
                                        const gdl_gligation_type * L)
{
	size_t i, np;
	gdl_hstruct_model_partition  * pmodel;
	
	gdl_hstruct_model_chromosome_init_pl (cmodel, k, f, extra, L);
	
	np = gdl_hstruct_model_chromosome_size (cmodel);
	
	for (i = 0; i < np; i++)
	{
		pmodel = gdl_hstruct_model_chromosome_get_partition (cmodel, i);
		_gdl_hstruct_model_init_partition (model, pmodel, mode, k);
	}
	
	return GDL_SUCCESS;
}

static int
gdl_hstruct_model_genome_alloc (gdl_hstruct_model * t)
{
	size_t i, j, l, ng, nc;
	
	t->tnc  = gdl_gmap_chrom_number (t->gmap);
	
	t->cdata = GDL_MALLOC (gdl_hstruct_data *, t->tnc);
	
	t->chroms = GDL_MALLOC (gdl_hstruct_model_chromosome *, t->tnc);
	
	ng = gdl_gmap_size (t->gmap);
	
	for (l = i = 0; i < ng; i++)
	{
		nc = gdl_gmap_genome_size (t->gmap, i);
		
		for (j = 0;	j < nc; j++, l++)
		{
			gdl_hstruct_data * cd;
			
			cd = gdl_hstruct_data_alloc ();
			
			cd->chrom = gdl_gmap_get_chromosome (t->gmap, i, j);
			cd->gview = gdl_hstruct_data_get_gview (t->data);
			cd->gmask = gdl_hstruct_data_get_gmask (t->data);
			cd->rng   = gdl_hstruct_data_get_rng (t->data);
			cd->locus_type = gdl_hstruct_data_get_locus_type (t->data);
			
			t->chroms[l]  = gdl_hstruct_model_chromosome_alloc (cd, l);
			t->cdata[l]   = cd;
		}
	}
	
	return GDL_SUCCESS;
}

gdl_hstruct_model *
gdl_hstruct_model_alloc (gdl_hstruct_data * D)
{
	gdl_hstruct_model * t;
	
	t = GDL_CALLOC (gdl_hstruct_model, 1);
	
	t->data = D;
	
	if (t->data->rng == 0)
	{
		GDL_FREE (t);
		GDL_ERROR_VAL ("No random generator defined",
		               GDL_EINVAL,
		               0);	
	}
	
	t->gmap = gdl_hstruct_data_get_gmap (D);
		
	if (t->gmap == 0)
	{
		GDL_FREE (t);
		GDL_ERROR_VAL ("Unable to extract gmap",
		               GDL_EINVAL,
		               0);	
	}
		
	gdl_hstruct_model_genome_alloc (t);
	
	return t;
}

void
gdl_hstruct_model_free (gdl_hstruct_model * t)
{
	if (t)
	{
		size_t i;
		for (i = 0; i < t->tnc; i++)
		{
			gdl_hstruct_model_chromosome_free (t->chroms[i]);
		}
		GDL_FREE (t->chroms);
		gdl_hstruct_parameters_free (t->parameters);
		gdl_hstruct_data_free (t->data);
		GDL_FREE (t);
	}	
}

int
gdl_hstruct_model_init (gdl_hstruct_model * t, const gdl_hstruct_model_mode * mode, size_t k)
{
	size_t i, j, l, ng, nc;
	gdl_hstruct_config * pconf;
	
	(mode->alloc_default_config)(&(t->default_config));
	
	t->k = k;
	
	t->parameters = gdl_hstruct_parameters_alloc (k);
	
	ng = gdl_gmap_size (t->gmap);
	
	for (l = i = 0; i < ng; i++)
	{
		nc = gdl_gmap_genome_size (t->gmap, i);
		
		for (j = 0;	j < nc; j++, l++)
		{
			_gdl_hstruct_model_init_chromosome (t, mode, t->chroms[l], k);
		}
	}
	
	(mode->free_default_config)(&(t->default_config));
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_model_init_pl (gdl_hstruct_model * t, const gdl_hstruct_model_mode * mode, size_t k, gdl_hstruct_partition_fonction f, void * extra, const gdl_gligation_type * L)
{
	size_t i, j, l, ng, nc;
	gdl_hstruct_config * pconf;
	struct _gdl_hstruct_model_default_config * dc;
	
	(mode->alloc_default_config)(&(t->default_config));
	
	t->k = k;
	
	t->parameters = gdl_hstruct_parameters_alloc (k);
	
	ng = gdl_gmap_size (t->gmap);
	
	for (l = i = 0; i < ng; i++)
	{
		nc = gdl_gmap_genome_size (t->gmap, i);
		
		for (j = 0;	j < nc; j++, l++)
		{
			_gdl_hstruct_model_init_pl_chromosome (t, t->chroms[l], mode, k, f, extra, L);
		}
	}
	
	(mode->free_default_config)(&(t->default_config));
	
	return GDL_SUCCESS;
}

gdl_hstruct_data *
gdl_hstruct_model_data (const gdl_hstruct_model * m)
{
	return m->data;	
}

const gdl_hstruct_parameters *
gdl_hstruct_model_parameters (const gdl_hstruct_model * m)
{
	return m->parameters;
}

const gdl_gmap *
gdl_hstruct_model_locus_map (const gdl_hstruct_model * m)
{
	return m->gmap;
}

size_t
gdl_hstruct_model_size (const gdl_hstruct_model * m)
{
	return m->tnc;	
}

gdl_hstruct_model_chromosome *
gdl_hstruct_model_get_chromosome (const gdl_hstruct_model * m, size_t i)
{
	return m->chroms[i];
}

int
gdl_hstruct_model_fuzzy2hmm (gdl_hstruct_model * m)
{
	size_t i;
	gdl_hstruct_fpoint * fpoint;
	gdl_hstruct_parameter_itr * itr;
	
	for (i = 0; i < m->tnc; i++)
	{
		gdl_hstruct_model_chromosome_fuzzy2hmm (m->chroms[i]);
	}
	
	itr = gdl_hstruct_parameter_iterate (m->parameters, gdl_hstruct_point_f);
	
	if (itr)
	{
		do
		{
			fpoint = (gdl_hstruct_fpoint *) gdl_hstruct_parameter_iterate_point (itr);
			gdl_hstruct_fpoint_update (fpoint, 0, 0);
		}
		while (gdl_hstruct_parameter_iterate_next (itr));
		
		gdl_hstruct_parameter_iterate_free (itr);
	}
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_model_attach_config (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config * conf, size_t l)
{
	return gdl_hstruct_model_partition_attach_config (pm, m->parameters, conf, l);
}

gdl_hstruct_model_partition *
gdl_hstruct_model_get_partition (const gdl_hstruct_model * m, const gdl_hstruct_parameter_domain * d, size_t i)
{
	size_t c = gdl_hstruct_parameter_domain_get_cidx (d, i);
	size_t p = gdl_hstruct_parameter_domain_get_pidx (d, i);
	return gdl_hstruct_model_chromosome_get_partition (m->chroms[c], p);
}

gdl_hstruct_point_handler *
gdl_hstruct_model_new_handler (gdl_hstruct_model * m, const gdl_hstruct_point_type * T, void * extra)
{
	return gdl_hstruct_parameters_new_handler (m->parameters, T, extra);
}

void
gdl_hstruct_model_clean_handler (gdl_hstruct_model * m, gdl_hstruct_point_handler * h)
{
	gdl_hstruct_parameters_clean_handler (m->parameters, h);	
}

gdl_hstruct_config_buffer *
gdl_hstruct_model_new_buffer (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm)
{
	return gdl_hstruct_model_partition_new_buffer (pm);
}

int
gdl_hstruct_model_update (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm,  gdl_hstruct_config_buffer * t, size_t l)
{
	return gdl_hstruct_model_partition_update (pm, m->parameters, t, l);	
}

int
gdl_hstruct_model_restore (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm,  gdl_hstruct_config_buffer * t, size_t l)
{
	return gdl_hstruct_model_partition_restore (pm, m->parameters, t, l);
}

static int
gdl_hstruct_model_golden_point (gdl_hstruct_model * model,
		                        const gdl_hstruct_point_type * T,
		                        gdl_hstruct_point * point,
		                        double * res_abs,
		                        double * res_sq)
{
	size_t dn;
	double loglik,ax,bx,cx,fa,fb,fc,xmin,old,new,err,tol;
	const gdl_hstruct_parameter_domain * d;
	gdl_hstruct_model_partition * mp;
	gdl_hstruct_hmm * hmm;
	
	double func (double x) {
		size_t i;
		double y = 0;
		
		gdl_hstruct_point_update_scale (point, x);
		
		for (i = 0; i < dn; i++)
		{
			mp  = gdl_hstruct_model_get_partition (model, d, i);
			hmm = gdl_hstruct_model_partition_get_hmm (mp);
			y  -= gdl_hstruct_hmm_get_loglikelihood (hmm);
		}
		
		y -= log (gdl_hstruct_point_prior (point));
				
		return y;
	};
	
	tol = gdl_hstruct_point_get_golden_tolerance (point);
	
	d = gdl_hstruct_parameters_get_domain (model->parameters, T, point);
	
	if (d && gdl_hstruct_parameter_domain_size (d))
	{
		
		dn = gdl_hstruct_parameter_domain_size (d);
		
		//printf ("DOMAIN SIZE %d\n", dn);
			
		old = ax = gdl_hstruct_point_value (point);
		
		//printf ("CURRENT POINT %g\n", old);
		
		bx  = gdl_hstruct_point_rng_value (point, model->data->rng);
		
		ax = gdl_hstruct_point_scale (point, ax);
		
		bx = gdl_hstruct_point_scale (point, bx);
		
		mnbrak (&ax, &bx, &cx, &fa, &fb, &fc, &func);
		
		loglik = -dgolden (ax, bx, cx, &func, tol, &xmin);
		
		//printf("OLD POINT %g (%g) \n", old, -func(gdl_hstruct_point_scale(point, old)));
		
		gdl_hstruct_point_update_scale (point, xmin);
		
		new = gdl_hstruct_point_value (point);
		
		//printf("NEW POINT %g (%g) \n", new, -func(gdl_hstruct_point_scale(point, new)));
		
		err = (old - new);
		
		if (old)
		{
			err/=old;
		}
		
		*res_abs += fabs (err);
		
		*res_sq  += err*err;
	}
	
	return GDL_SUCCESS;	
} 

static int
gdl_hstruct_model_golden_fpoint (gdl_hstruct_model * model, const gdl_hstruct_point_type * T, gdl_hstruct_fpoint * point,  double * res_abs, double * res_sq)
{
	size_t i, dn;
	const gdl_hstruct_parameter_domain * d;
	gdl_hstruct_model_partition * mp;
	gdl_hstruct_hmm * hmm;
	gdl_boolean up = gdl_false;
	
	d = gdl_hstruct_parameters_get_domain (model->parameters, T, point);
	
	if (d && gdl_hstruct_parameter_domain_size (d))
	{
		dn = gdl_hstruct_parameter_domain_size (d);
		
		for (i = 0; i < dn; i++)
		{
			mp  = gdl_hstruct_model_get_partition (model, d, i);
			hmm = gdl_hstruct_model_partition_get_hmm (mp);
			gdl_hstruct_hmm_fb_update (hmm, &up);
		}
		if (up)
		{
			double old, new, abs=0, sq=0;
			gdl_hstruct_fpoint * buf;
			
			old = gdl_hstruct_model_loglikelihood (model);
			
			buf = gdl_hstruct_fpoint_clone (point);
			gdl_hstruct_fpoint_init_collect (buf);
			
			gdl_hstruct_fpoint_update (point, &abs, &sq);
			
			new = gdl_hstruct_model_loglikelihood (model);
			
			if (new >= old)
			{
				(*res_abs) += abs;
				(*res_sq)  += sq;
			}
			else
			{
				gdl_hstruct_fpoint_copy (point, buf);
			}
			
			gdl_hstruct_fpoint_free (buf);
		}
	}
	
	return GDL_SUCCESS;	
}

int
gdl_hstruct_model_golden_parameter (gdl_hstruct_model * model, const gdl_hstruct_point_type * T, void * p, double * res_abs, double * res_sq)
{
	if (p)
	{
		if (T == gdl_hstruct_point_f && gdl_hstruct_fpoint_is_eligible ((gdl_hstruct_fpoint *) p))
		{
			return gdl_hstruct_model_golden_fpoint (model, T, (gdl_hstruct_fpoint *) p, res_abs, res_sq);
		}
		else if (gdl_hstruct_point_is_eligible ((gdl_hstruct_point *) p))
		{
			if (T == gdl_hstruct_point_hot)
			{
				//printf (">>HOT SPOT IN = %g (%d)",gdl_hstruct_point_value ((gdl_hstruct_point*) p), gdl_hstruct_point_is_eligible ((gdl_hstruct_point *) p));
				gdl_hstruct_model_golden_point (model, T, (gdl_hstruct_point *) p, res_abs, res_sq);
				//printf (" OUT = %g\n", gdl_hstruct_point_value ((gdl_hstruct_point*) p));
			}
			else
			{
				gdl_hstruct_model_golden_point (model, T, (gdl_hstruct_point *) p, res_abs, res_sq);
			}
			return GDL_SUCCESS;
		}
	}	
}

int
gdl_hstruct_model_golden_partition (gdl_hstruct_model * model, gdl_hstruct_model_partition * part, double * res_abs, double * res_sq)
{
	size_t nres = 0;
	gdl_hstruct_config * config;
	gdl_hstruct_config_collector * collector;
	gdl_hstruct_config_itr * citr;
	gdl_hstruct_config_block * block;
	gdl_hstruct_config_block_itr * itr;
	void * point;
	gdl_hashtable * buffer;
	
	*res_abs = *res_sq = 0;
	
	buffer = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	itr = gdl_hstruct_model_partition_config_block_iterator (part, NULL);
	
	do
	{
		block     = gdl_hstruct_config_block_iterator_get (itr);
		collector = gdl_hstruct_config_block_config_collector (block);
		
		citr = gdl_hstruct_config_collector_iterator (collector);
		do
		{
			config = gdl_hstruct_config_collector_iterator_config (citr);
			if (config->k > 1)
			{
				// rho
				point  = gdl_hstruct_config_get_point (config, gdl_hstruct_point_rho);
				if (point && gdl_hashtable_vlookup (buffer, point)==0)
				{
					gdl_hstruct_model_golden_parameter (model, gdl_hstruct_point_rho, point, res_abs, res_sq);
					gdl_hashtable_vadd (buffer, point, point, 0);
					nres++;
				}
				// hot
				point  = gdl_hstruct_config_get_point (config, gdl_hstruct_point_hot);
				if (point && gdl_hashtable_vlookup (buffer, point)==0)
				{
					//printf ("HOT SPOT [%d, %d] IN = %g (%d)", gdl_hstruct_config_block_from (block), gdl_hstruct_config_block_to (block), gdl_hstruct_point_value ((gdl_hstruct_point*) point), gdl_hstruct_point_is_eligible ((gdl_hstruct_point *) point));
					gdl_hstruct_model_golden_parameter (model, gdl_hstruct_point_hot, point, res_abs, res_sq);
					//printf (" OUT = %g\n", gdl_hstruct_point_value ((gdl_hstruct_point*) point));
					gdl_hashtable_vadd (buffer, point, point, 0);
					nres++;
				}
			}
			// mu
			point  = gdl_hstruct_config_get_point (config, gdl_hstruct_point_mu);
			if (point && gdl_hashtable_vlookup (buffer, point)==0)
			{
				gdl_hstruct_model_golden_parameter (model, gdl_hstruct_point_mu, point, res_abs, res_sq);
				gdl_hashtable_vadd (buffer, point, point, 0);
				nres++;
			}
			if (config->k > 1)
			{
				// f
				point  = gdl_hstruct_config_get_point (config, gdl_hstruct_point_f);
				if (point && gdl_hashtable_vlookup (buffer, point)==0)
				{
					gdl_hstruct_model_golden_parameter (model, gdl_hstruct_point_f, point, res_abs, res_sq);
					gdl_hashtable_vadd (buffer, point, point, 0);
					nres++;
				}
			}
		}
		while (gdl_hstruct_config_collector_iterator_next (citr));
		
		gdl_hstruct_config_collector_iterator_free (citr);
		
	}
	while (gdl_hstruct_config_block_iterator_next (itr, gdl_true));
	
	gdl_hstruct_config_block_iterator_free (itr);
	
	if (nres)
	{
		*res_abs /= (double)nres;
		*res_sq  /= (double)nres;
	}
	
	gdl_hashtable_free (buffer);
	
	return GDL_SUCCESS;
}

gdl_hstruct_parameter_itr *
gdl_hstruct_model_parameter_iterate (const gdl_hstruct_model * model, const gdl_hstruct_point_type * T)
{
	return gdl_hstruct_parameter_iterate (model->parameters, T);
}

size_t
gdl_hstruct_model_parameter_size (const gdl_hstruct_model * model)
{
	return gdl_hstruct_parameters_size (model->parameters);
}

size_t
gdl_hstruct_model_data_size (const gdl_hstruct_model * model)
{
	return gdl_hstruct_data_size (model->data);
}

static gdl_hstruct_model_point_prior (const gdl_hstruct_model * model, const gdl_hstruct_point_type * T)
{
	double prior = 0;
	
	if (T != gdl_hstruct_point_f)
	{
		gdl_hstruct_parameter_itr * itr;
		
		itr = gdl_hstruct_model_parameter_iterate (model, T);
		
		if (itr)
		{
			do
			{
				void * point = gdl_hstruct_parameter_iterate_point (itr);
				prior += log (gdl_hstruct_point_prior (point));
			}
			while (gdl_hstruct_parameter_iterate_next (itr));
			gdl_hstruct_parameter_iterate_free (itr);
		}
		
	}
	
	//printf ("LOG PRIOR %s ==> %g\n", T->name, prior);
	
	return prior;
}
static double
gdl_hstruct_model_prior (const gdl_hstruct_model * model)
{
	double prior;
	
	prior  = gdl_hstruct_model_point_prior (model, gdl_hstruct_point_rho);
	prior += gdl_hstruct_model_point_prior (model, gdl_hstruct_point_hot);
	prior += gdl_hstruct_model_point_prior (model, gdl_hstruct_point_mu);
	
	return prior;
}

double
gdl_hstruct_model_loglikelihood (const gdl_hstruct_model * model)
{
	size_t i;
	double y = 0;
	
	for (i = 0; i < model->tnc; i++)
	{
		y += gdl_hstruct_model_chromosome_loglikelihood (model->chroms[i]);
	}
	// add prior
	y += gdl_hstruct_model_prior (model);
	
	return y;
}

static int
_find_best_transition (gdl_matrix * tr, const gdl_hstruct_config * c1, const gdl_hstruct_config * c2, size_t ** permut, gdl_boolean * backward)
{
	size_t i, j, l, k1, k2, kmin, imx, jmx, * idx;
	int * ligate;
	double * freq;
	gdl_boolean * seen;
	double max;
	const gdl_hstruct_fpoint * q;
	
	k1 = c1->k;
	k2 = c2->k;
	
	kmin = GDL_MIN (k1, k2);
	
	ligate = GDL_MALLOC (int, k1);
	for (i = 0; i < k1; i++) ligate[i]=-1;
	seen   = GDL_CALLOC (gdl_boolean, k2);
	
//	idx = GDL_CALLOC (size_t, k1);
//	if (k1 > 1)
//	{
//		freq = GDL_MALLOC (double, k1);
//		q = (gdl_hstruct_fpoint *) gdl_hstruct_config_get_point (c1, gdl_hstruct_point_f);
//		for (i = 0; i < k1; i++)
//		{
//			freq[i]=gdl_hstruct_fpoint_value (q, i);
//		}
//		gdl_sort_index (idx, freq, 1, k1);
//		GDL_FREE (freq);
//	}
//	
	for (i = 0; i < k1; i++)
	{
		printf ("TR %d", i);
		for (j = 0; j <  k2; j++)
		{
			printf (" %.1f", gdl_matrix_get (tr,i,j));
		}
		printf ("\n");
	}
	
	for (l = 0; l < kmin; l++)
	{
		max = GDL_NEGINF;
		for (i = 0; i < k1; i++)
		{
			if (ligate[i]==-1)
			{
				for (j = 0; j < k2; j++)
				{
					if (!seen[j] && gdl_matrix_get (tr,i,j) > max)
					{
						max = gdl_matrix_get (tr,i,j);
						imx = i;
						jmx = j;
					}
				}
			}					
		}
		ligate[imx] = jmx;
		seen[jmx]   = gdl_true;
		printf ("LIGATE [%d] - [%d]\n", imx, jmx);
	}
	
	*permut   = GDL_MALLOC (size_t, GDL_MAX (k1, k2));
	*backward = (k1 > k2) ? gdl_true : gdl_false;
	
	for (j = i = 0; i < k1; i++)
	{
		if (ligate[i] != -1)
		{
			(*permut)[j] = ligate[i];
			j++;
		}
	}
	
	if (k1 < k2)
	{
		for (i = k1, j = 0; j < k2; j++)
		{
			if (!seen[j])
			{
				(*permut)[i++] = j;
				seen[j] = gdl_true;
			}	
		}
	}
	else if (k1 > k2)
	{
		for (i = k2, j = 0; j < k1; j++)
		{
			if (ligate[j]==-1)
			{
				(*permut)[i++] = j;
				ligate[j] = 0;
			}	
		}
	}
	
	for (i = 0; i < GDL_MAX (k1, k2); i++)
	{
		printf ("PERMUT %d %d\n", i, (*permut)[i]);
	}
	
	GDL_FREE (ligate);
	GDL_FREE (seen);
	GDL_FREE (idx);
	
	return GDL_SUCCESS;
}

static void
_ligation_ancestral (gdl_hstruct_model * m, gdl_hstruct_model_partition * p)
{
	if (gdl_hstruct_model_partition_ancestral_size (p) == 1)
	{
		return;	
	}
	else if (gdl_hstruct_model_partition_ligation_site (p) == 0)
	{
		return;
	}	
	else
	{	
		size_t site, i, j, k, l, a, nl, nk, imx, jmx, swap, * permut;
		gdl_boolean backward;
		double max, t1, t2;
		gdl_matrix * trmx;
		gdl_hstruct_config * c1, * c2;
		gdl_hstruct_point_handler * rh;
		gdl_hstruct_config_buffer * t;
		
		// get the index of the ligation site
		site = gdl_hstruct_model_partition_ligation_site (p);
		// get the configurations at the left and at the right of the site
		c1 = gdl_hstruct_model_partition_get_config (p, site);
		c2 = gdl_hstruct_model_partition_get_config (p, site+1);
		// Set the ligation site independent
		t = gdl_hstruct_model_new_buffer (m, p);
	   gdl_hstruct_config_buffer_set (t, gdl_hstruct_point_rho, 0); 
	   // Estimate the value of this rec parameter
	   gdl_hstruct_model_update (m, p, t, site);
		// Get now the transition matrix at the ligation site
		trmx = gdl_hstruct_hmm_get_viterbi_transition_matrix (gdl_hstruct_model_partition_get_hmm (p), site);
		// Find the best transition between ancestral haplotypes
		_find_best_transition (trmx, c1, c2, &permut, &backward);
		
		if (backward)
		{
			gdl_hstruct_model_partition_permut_ancestral (p, 0, site, permut, c1->k);
		}
		else
		{
			nl = gdl_hstruct_model_partition_size (p);
			gdl_hstruct_model_partition_permut_ancestral (p, site+1, nl-1, permut, c2->k);
		}
		
		gdl_hstruct_model_partition_reset_ligation_site (p);
		
		// remove the extra rho parameter
		gdl_hstruct_model_restore (m, p, t, site);
		// Clean
		GDL_FREE (permut);
      gdl_matrix_free (trmx);
		gdl_hstruct_config_buffer_free (t);
	}
}

int
gdl_hstruct_model_ligation (gdl_hstruct_model * m, const gdl_gligation_type * T)
{
	int status = GDL_SUCCESS;
	size_t i, j, np;
	
	for (i = 0; i < m->tnc; i++)
	{
		np     = gdl_hstruct_model_chromosome_size (m->chroms[i]);
		status = gdl_hstruct_model_chromosome_ligation (m->parameters, m->chroms[i], T);
		if (np > 1)
		{
			np = gdl_hstruct_model_chromosome_eligible_size (m->chroms[i], T);
			for (j = 0; j < np; j++)
			{
				_ligation_ancestral (m, gdl_hstruct_model_chromosome_get_partition (m->chroms[i], j));
			}
		}
	}
	
	return status;
}

int
gdl_hstruct_model_backup_points (gdl_hstruct_model * m)
{
	size_t i, j, np;
	
	for (i = 0; i < m->tnc; i++)
	{
		np = gdl_hstruct_model_chromosome_size (m->chroms[i]);
		for (j = 0; j < np; j++)
		{
			gdl_hstruct_model_partition_backup_points (
			     gdl_hstruct_model_chromosome_get_partition (m->chroms[i], j),
			     m->parameters);
		}
	}
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_model_backup_partition_points (gdl_hstruct_model * m,
                                           gdl_hstruct_model_partition * pm)
{
	gdl_hstruct_model_partition_backup_points (pm, m->parameters);
}

int
gdl_hstruct_model_restore_points (gdl_hstruct_model * m)
{
	size_t i, j, np;
	
	for (i = 0; i < m->tnc; i++)
	{
		np = gdl_hstruct_model_chromosome_size (m->chroms[i]);
		for (j = 0; j < np; j++)
		{
			gdl_hstruct_model_partition_restore_points (
			     gdl_hstruct_model_chromosome_get_partition (m->chroms[i], j),
			     m->parameters);
		}
	}
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_model_restore_partition_points (gdl_hstruct_model * m,
                                           gdl_hstruct_model_partition * pm)
{
	gdl_hstruct_model_partition_restore_points (pm, m->parameters);
}

gdl_hstruct_point_handler *
gdl_hstruct_model_collapse_ancestral_block (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config_block * block, size_t k)
{
	return gdl_hstruct_model_partition_collapse_ancestral_block (pm, m->parameters, block, k);
}

void
gdl_hstruct_model_restore_ancestral_block (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config_block * block, size_t k)
{
	gdl_hstruct_model_partition_restore_ancestral_block (pm, m->parameters, block, k);
}

gdl_hstruct_point_handler *
gdl_hstruct_model_collapse_ancestral_block2 (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config_block * hot1, gdl_hstruct_config_block * block, gdl_hstruct_config_block * hot2, size_t k)
{
	return gdl_hstruct_model_partition_collapse_ancestral_block2 (pm, m->parameters, hot1, block, hot2, k);
}

void
gdl_hstruct_model_restore_ancestral_block2 (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config_block * hot1, gdl_hstruct_config_block * block, gdl_hstruct_config_block * hot2, size_t k)
{
	gdl_hstruct_model_partition_restore_ancestral_block2 (pm, m->parameters, hot1, block, hot2, k);
}

gdl_hstruct_point_handler *
gdl_hstruct_model_remove_hotspot_block (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config_block * block)
{
	return gdl_hstruct_model_partition_remove_hotspot_block (pm, m->parameters, block);
}

void
gdl_hstruct_model_restore_hotspot_block (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm, gdl_hstruct_config_block * block)
{
	gdl_hstruct_model_partition_restore_hotspot_block (pm, m->parameters, block);
}

void
gdl_hstruct_model_check_config_block (gdl_hstruct_model * m, gdl_hstruct_model_partition * pm)
{
	gdl_hstruct_model_partition_check_config_block (pm, m->parameters);
}

void
gdl_hstruct_model_config_block_update (gdl_hstruct_model * m, gdl_hstruct_config_block * b)
{
	gdl_hstruct_config_block_update (m->parameters, b);
}


static void
_reset_default_config (struct _gdl_hstruct_model_default_config ** c)
{
	if (*c)
	{
		gdl_hashtable_free ((*c)->physic_type);
		gdl_hashtable_free ((*c)->genetic_type);
		GDL_FREE (*c);
		*c=NULL;
	}
}

static void
_get_homogeneous_default_config (struct _gdl_hstruct_model_default_config ** c)
{
	if (*c)
	{
		return;
	}
	else
	{
		*c = GDL_CALLOC (struct _gdl_hstruct_model_default_config, 1);
	}
}

static const gdl_hstruct_model_mode _homogeneous =
{
	"gdl_hstruct_model_homogeneous",
	&_reset_default_config,
	&_reset_default_config,
	&_get_homogeneous_default_config
};

static void
_get_heterogeneous_default_config (struct _gdl_hstruct_model_default_config ** c)
{
	struct _gdl_hstruct_model_default_config * n = GDL_CALLOC (struct _gdl_hstruct_model_default_config, 1);
	if (*c)
	{
		gdl_hashtable_free ((*c)->physic_type);
		gdl_hashtable_free ((*c)->genetic_type);
		GDL_FREE (*c);
	}
	*c = n;
}

static const gdl_hstruct_model_mode _heterogeneous =
{
	"gdl_hstruct_model_heterogeneous",
	&_reset_default_config,
	&_reset_default_config,
	&_get_heterogeneous_default_config
};

const gdl_hstruct_model_mode * gdl_hstruct_model_homogeneous   = &_homogeneous;
const gdl_hstruct_model_mode * gdl_hstruct_model_heterogeneous = &_heterogeneous;

static const gdl_hstruct_model_type _gdl_hstruct_model_gmap =
{
	"gdl_hstruct_model_gmap"
};

static const gdl_hstruct_model_type _gdl_hstruct_model_hmap =
{
	"gdl_hstruct_model_hmap"
};

static const gdl_hstruct_model_type _gdl_hstruct_model_hmap_unknown =
{
	"gdl_hstruct_model_hmap_unknown"
};

const gdl_hstruct_model_type * gdl_hstruct_model_gmap = &_gdl_hstruct_model_gmap;
const gdl_hstruct_model_type * gdl_hstruct_model_hmap = &_gdl_hstruct_model_hmap;
const gdl_hstruct_model_type * gdl_hstruct_model_hmap_unknown = &_gdl_hstruct_model_hmap_unknown;

#include <gdl_hstruct_result.h>

int gdl_hstruct_model_chromosome_setup (gdl_hstruct_model_chromosome * m, const gdl_hstruct_chromosome_result * cr, gdl_hstruct_parameters * params);

gdl_hstruct_model *
gdl_hstruct_model_setup (const gdl_hstruct_result * r, const gdl_view * v, const gdl_rng * rng)
{
	size_t i, j, k, l, ng, nc, nl;
	gdl_hstruct_data       * data;
	gdl_hstruct_model      * model;
	const gdl_hstruct_genome_result * gr;
	const gdl_hstruct_chromosome_result * cr;
	
	data       = gdl_hstruct_data_alloc ();
	data->view = v;
	data->gmap = gdl_gmap_clone (gdl_hstruct_result_gmap (r));
	data->rng  = rng;
	data->locus_type = gdl_view_get_locus_type (v);
	
	model             = gdl_hstruct_model_alloc (data);
	model->parameters = gdl_hstruct_parameters_setup (gdl_hstruct_result_parameter_registry (r));
	
	ng = gdl_hstruct_result_size (r);
	
	for (k = i = 0; i < ng; i++)
	{
		gr = gdl_hstruct_result_genome (r, i);
		
		nc = gdl_hstruct_genome_result_size (gr);
		
		for (j = 0; j < nc; j++, k++)
		{
			cr = gdl_hstruct_genome_result_chromosome (gr, j);
			model->cdata[k] = gdl_hstruct_data_alloc ();
			model->cdata[k]->gview = gdl_hstruct_data_get_gview (model->data);
			model->cdata[k]->gmask = gdl_hstruct_data_get_gmask (model->data);
			model->cdata[k]->chrom = gdl_gmap_get_chromosome (data->gmap, i, j);
			model->cdata[k]->rng   = rng;
			model->cdata[k]->locus_type = data->locus_type;
			model->chroms[k]       = gdl_hstruct_model_chromosome_alloc (model->cdata[k], k);
			gdl_hstruct_model_chromosome_setup (model->chroms[k], cr, model->parameters);
		}
	}
	
	return model;
}
