/*  
 * 	hstruct/params.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_hstruct_point.h>
#include <gdl/gdl_hstruct_config.h>
#include <gdl/gdl_hstruct_parameters.h>

struct _gdl_hstruct_parameter_domain
{
	size_t _size;
	size_t size;
	size_t * cidx;
	size_t * pidx;
};

size_t
gdl_hstruct_parameter_domain_get_cidx (const gdl_hstruct_parameter_domain * d, size_t i)
{
	return d->cidx[i];
}

size_t
gdl_hstruct_parameter_domain_get_pidx (const gdl_hstruct_parameter_domain * d, size_t j)
{
	return d->pidx[j];
}


gdl_hstruct_parameter_domain *
gdl_hstruct_parameter_domain_alloc (size_t size)
{
	gdl_hstruct_parameter_domain * d;
	
	d = GDL_MALLOC (gdl_hstruct_parameter_domain, 1);
	
	d->_size = size;
	d->size  = 0;
	
	if (d->_size)
	{
		d->cidx = GDL_MALLOC (size_t, d->_size);	
		d->pidx = GDL_MALLOC (size_t, d->_size);	
	}
	else
	{
		d->cidx = d->pidx = NULL;
	}
	
	return d;
}

void
gdl_hstruct_parameter_domain_free (gdl_hstruct_parameter_domain * d)
{
	if (d)
	{
		GDL_FREE (d->cidx);
		GDL_FREE (d->pidx);
		GDL_FREE (d);
	}	
}

static int
_realloc (gdl_hstruct_parameter_domain * d)
{
	size_t * n;
	d->_size += 5;
	n = GDL_MALLOC (size_t, d->_size);
	if (d->cidx)
	{
		memcpy (n, d->cidx, sizeof (size_t)*d->size);
		GDL_FREE (d->cidx);
	}
	d->cidx = n;
	n = GDL_MALLOC (size_t, d->_size);
	if (d->pidx)
	{
		memcpy (n, d->pidx, sizeof (size_t)*d->size);
		GDL_FREE (d->pidx);
	}
	d->pidx = n;
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameter_domain_add (gdl_hstruct_parameter_domain * d, size_t i, size_t j)
{
	size_t k;
	
	for (k = 0; k < d->size; k++)
	{
		if (d->cidx[k]==i && d->pidx[k]==j)
		{
			break;
		}
	}
	if (k == d->size)
	{
		if (d->_size == d->size)
		{
			_realloc (d);
		}
		d->cidx[k]=i;
		d->pidx[k]=j;
		d->size++;
	}
	
	return k;
}

int
gdl_hstruct_parameter_domain_remove (gdl_hstruct_parameter_domain * d, size_t i, size_t j)
{
	size_t k;
	
	for (k = 0; k < d->size; k++)
	{
		if (d->cidx[k]==i && d->pidx[k]==j)
		{
			break;
		}
	}
	
	if (k < d->size)
	{
		size_t l;
		
		for (l = k+1; l < d->size; l++)
		{
			d->cidx[l-1] = d->cidx[l];
			d->pidx[l-1] = d->pidx[l];
		}
		(d->size)--;
	}
	
	return d->size;	
}

size_t
gdl_hstruct_parameter_domain_size (const gdl_hstruct_parameter_domain * d)
{
	return d->size;
}

struct _gdl_hstruct_parameter_collector
{
	gdl_hashtable * table;
};

gdl_hstruct_parameter_collector *
gdl_hstruct_parameter_collector_alloc ()
{
	gdl_hashtable * tmp;
	gdl_hstruct_parameter_collector * c;
	
	c = GDL_MALLOC (gdl_hstruct_parameter_collector, 1);
	
	c->table = gdl_hashtable_alloc (gdl_hash_default, 4);
	
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (c->table, gdl_hstruct_point_rho, tmp, 0);
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (c->table, gdl_hstruct_point_hot, tmp, 0);
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (c->table, gdl_hstruct_point_mu, tmp, 0);
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (c->table, gdl_hstruct_point_f, tmp, 0);	
	
	return c;
}

static void
_clean_table (void * vtmp)
{
	gdl_hashtable * tmp = (gdl_hashtable *) vtmp;
	
	if (gdl_hashtable_size (tmp))
	{
		gdl_hashtable_itr * itr = gdl_hashtable_iterator (tmp);
		
		do
		{
			void * v = gdl_hashtable_iterator_value (itr);
			GDL_FREE (v);
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
	}
	
	gdl_hashtable_free (tmp);
}

void
gdl_hstruct_parameter_collector_free (gdl_hstruct_parameter_collector * c)
{
	if (c)
	{
		_clean_table (gdl_hashtable_vlookup (c->table, gdl_hstruct_point_rho));
		_clean_table (gdl_hashtable_vlookup (c->table, gdl_hstruct_point_hot));
		_clean_table (gdl_hashtable_vlookup (c->table, gdl_hstruct_point_mu));
		_clean_table (gdl_hashtable_vlookup (c->table, gdl_hstruct_point_f));
		gdl_hashtable_free (c->table);
	}
}

size_t
gdl_hstruct_parameter_collector_add (gdl_hstruct_parameter_collector * c, const gdl_hstruct_point_type * T, void * p)
{
	size_t * n;
	gdl_hashtable * tmp;
	
	tmp = (gdl_hashtable *) gdl_hashtable_vlookup (c->table, T);
	
	n = (size_t *) gdl_hashtable_vlookup (tmp, p);
	
	if (n == 0)
	{
		n = GDL_MALLOC (size_t, 1);
		gdl_hashtable_vadd (tmp, p, n, 0);
	}
	
	(*n)++;
	
	return *n;
}

struct _gdl_hstruct_parameter
{
	size_t idx;
	void * p;
	void * b;
	gdl_hstruct_parameter_domain * d;
};

typedef struct _gdl_hstruct_parameter gdl_hstruct_parameter;

static gdl_hstruct_parameter *
gdl_hstruct_parameter_alloc (void * p)
{
	gdl_hstruct_parameter * pr = GDL_MALLOC (gdl_hstruct_parameter, 1);
	
	pr->p = p;
	pr->b = NULL;
	pr->d = gdl_hstruct_parameter_domain_alloc (5);
	
	return pr;
}

struct _gdl_hstruct_parameters
{
	size_t k;
	size_t size;
	gdl_hashtable * points;
	gdl_list      * configs;
	gdl_hstruct_config_collector * collector;
	gdl_hashtable * setup;
};

static gdl_hashtable *
_get_table (const gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T)
{
	return gdl_hashtable_vlookup (r->points, T);
}

static void
_add_point (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * p)
{
	gdl_hashtable * table      = _get_table (r, T);
	gdl_hstruct_parameter * pr = gdl_hstruct_parameter_alloc (p);
	
	pr->idx = gdl_hashtable_size (table);
	
	gdl_hashtable_vadd (table, p, pr, 0);
}

static void
_rm_point (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * p)
{
	gdl_hashtable * table      = _get_table (r, T);
	gdl_hstruct_parameter * pr = (gdl_hstruct_parameter *) gdl_hashtable_vlookup (table, p);
	
	if (pr)
	{
		gdl_hstruct_parameter_domain_free (pr->d);
		
		gdl_hashtable_vremove (table, p);
		
		// Update point indexes
		if (gdl_hashtable_size (table))
		{
			gdl_hashtable_itr * itr = gdl_hashtable_iterator (table);
			
			do
			{
				gdl_hstruct_parameter * tpr = (gdl_hstruct_parameter *) gdl_hashtable_iterator_value (itr);
				if (tpr->idx > pr->idx)
				{
					(tpr->idx)--;
				}
			} while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		GDL_FREE (pr);
	}
}

static gdl_hstruct_parameter *
_get_parameter (const gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * p)
{
	gdl_hashtable * table      = _get_table (r, T);
	gdl_hstruct_parameter * pr = gdl_hashtable_vlookup (table, p);
	return pr;	
}

static gdl_hstruct_parameter_domain *
_get_domain (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * p)
{
	gdl_hstruct_parameter * pr = _get_parameter (r, T, p);
	return pr->d;
}

static void
_extend_point_domain (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * pt, size_t c, size_t p)
{
	if (pt)
	{
		gdl_hstruct_parameter_domain * d = _get_domain (r, T, pt);
		gdl_hstruct_parameter_domain_add (d, c, p);
		if (T == gdl_hstruct_point_f)
		{
			r->size += gdl_hstruct_fpoint_size((gdl_hstruct_fpoint *)pt);
			//printf ("PARAM FPOINT EXTEND + %d\n", gdl_hstruct_fpoint_size((gdl_hstruct_fpoint *)pt));
		}
	}
}

static void
_collapse_point_domain (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * pt, size_t c, size_t p)
{
	if (pt)
	{
		gdl_hstruct_parameter_domain * d = _get_domain (r, T, pt);
		gdl_hstruct_parameter_domain_remove (d, c, p);
		if (T == gdl_hstruct_point_f)
		{
			r->size -= gdl_hstruct_fpoint_size((gdl_hstruct_fpoint *)pt);
		}
	}
}

static void
_backup_point (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * pt)
{
	if (pt)
	{
		gdl_hstruct_parameter * pr = _get_parameter (r, T, pt);
		if (pr->b)
		{
			(T->free)(pr->b);
		}
		pr->b = (T->clone)(pt);
	}	
}

static void
_restore_point (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * pt)
{
	if (pt)
	{
		gdl_hstruct_parameter * pr = _get_parameter (r, T, pt);
		if (pr->b)
		{
			(T->copy)(pt, pr->b);
		}
	}	
}

static void
_extend_config_domain (gdl_hstruct_parameters * r, const gdl_hstruct_config * conf, size_t c, size_t p)
{
	void * pt;
	
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_f);
	_extend_point_domain (r, gdl_hstruct_point_f, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_f))
	{
		r->size += gdl_hstruct_fpoint_size((gdl_hstruct_fpoint *)pt)-1;
		//printf ("PARAM FPOINT + %d\n", gdl_hstruct_fpoint_size((gdl_hstruct_fpoint *)pt)-1);
	}
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_mu);
	_extend_point_domain (r, gdl_hstruct_point_mu, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_mu))
	{
		r->size += 1;
		//printf ("PARAM MU POINT + %d\n", 1);
	}
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_hot);
	_extend_point_domain (r, gdl_hstruct_point_hot, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_hot))
	{
		r->size += 1;
		//printf ("PARAM HOT POINT + %d\n", 1);
	}
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_rho);
	_extend_point_domain (r, gdl_hstruct_point_rho, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_rho))
	{
		r->size += 1;
		//printf ("PARAM RHO POINT + %d\n", 1);
	}
}

static void
_collapse_config_domain (gdl_hstruct_parameters * r, const gdl_hstruct_config * conf, size_t c, size_t p)
{
	void * pt;
	
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_f);
	_collapse_point_domain (r, gdl_hstruct_point_f, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_f))
	{
		r->size -= 	gdl_hstruct_fpoint_size((gdl_hstruct_fpoint *)pt)-1;
	}
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_mu);
	_collapse_point_domain (r, gdl_hstruct_point_mu, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_mu))
	{
		r->size -= 1;
	}
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_hot);
	_collapse_point_domain (r, gdl_hstruct_point_hot, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_hot))
	{
		r->size -= 1;
	}
	pt = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_rho);
	_collapse_point_domain (r, gdl_hstruct_point_rho, pt, c, p);
	if (pt && !gdl_hstruct_config_point_inuse (conf, gdl_hstruct_point_rho))
	{
		r->size -= 1;
	}
}

static void
_backup_config_points (gdl_hstruct_parameters * r, const gdl_hstruct_config * conf)
{
	_backup_point (r, gdl_hstruct_point_f, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_f));
	_backup_point (r, gdl_hstruct_point_mu, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_mu));
	_backup_point (r, gdl_hstruct_point_hot, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_hot));
	_backup_point (r, gdl_hstruct_point_rho, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_rho));
}

static void
_restore_config_points (gdl_hstruct_parameters * r, const gdl_hstruct_config * conf)
{
	//printf ("RESTORE CONFIG %p (%d)\n", conf, conf->k);
	_restore_point (r, gdl_hstruct_point_f, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_f));
	_restore_point (r, gdl_hstruct_point_mu, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_mu));
	_restore_point (r, gdl_hstruct_point_hot, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_hot));
	_restore_point (r, gdl_hstruct_point_rho, gdl_hstruct_config_get_point (conf, gdl_hstruct_point_rho));
}

gdl_hstruct_parameters *
gdl_hstruct_parameters_alloc (size_t k)
{
	gdl_hashtable * tmp;
	gdl_hstruct_parameters * p;
	
	p = GDL_CALLOC (gdl_hstruct_parameters, 1);
	
	p->size    = 0;
	p->k       = k;
	p->points  = gdl_hashtable_alloc (gdl_hash_default, 4);
	
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (p->points, gdl_hstruct_point_rho, tmp, 0);
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (p->points, gdl_hstruct_point_hot, tmp, 0);
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (p->points, gdl_hstruct_point_mu, tmp, 0);
	tmp = gdl_hashtable_alloc (gdl_hash_default, 5);
	gdl_hashtable_vadd (p->points, gdl_hstruct_point_f, tmp, 0);
	
	p->configs   = gdl_list_alloc (gdl_list_default);
	p->collector = gdl_hstruct_config_collector_alloc (10);
	
	return p;
}

static void
_gdl_hstruct_parameters_free_table (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T)
{
	gdl_hstruct_parameter_itr * itr;
		
	itr = gdl_hstruct_parameter_iterate (r, T);
	if (itr)
	{
		do
		{
			void * p = gdl_hstruct_parameter_iterate_point (itr);
			gdl_hstruct_parameters_clean_point (r, T, &p);
		}
		while (gdl_hstruct_parameter_iterate_next (itr));
		gdl_hstruct_parameter_iterate_free (itr);
	}	
}

void
gdl_hstruct_parameters_free (gdl_hstruct_parameters * r)
{
	if (r)
	{
		gdl_hashtable_itr * itr;
		
		_gdl_hstruct_parameters_free_table (r, gdl_hstruct_point_rho);
		_gdl_hstruct_parameters_free_table (r, gdl_hstruct_point_hot);
		_gdl_hstruct_parameters_free_table (r, gdl_hstruct_point_mu);
		_gdl_hstruct_parameters_free_table (r, gdl_hstruct_point_f);
		
		itr = gdl_hashtable_iterator (r->points);
		do
		{
			gdl_hashtable_free ((gdl_hashtable *) gdl_hashtable_iterator_value (itr));
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
		gdl_hstruct_config_collector_free (r->collector);
		gdl_list_free (r->configs);
		GDL_FREE (r);
	}	
}

size_t
gdl_hstruct_parameters_size (const gdl_hstruct_parameters * r)
{
	return r->size;
}

gdl_hstruct_config *
gdl_hstruct_parameters_new_config (gdl_hstruct_parameters * r)
{
	gdl_hstruct_parameter_domain * d;
	gdl_hstruct_config * c = gdl_hstruct_config_alloc (r->k);
	void ** p;
	
	p = gdl_hstruct_parameters_new_point (r, gdl_hstruct_point_mu, NULL);
	gdl_hstruct_config_set_point (c, gdl_hstruct_point_mu, p);
	if (r->k > 1)
	{
		p = gdl_hstruct_parameters_new_point (r, gdl_hstruct_point_rho, NULL);
		gdl_hstruct_config_set_point (c, gdl_hstruct_point_rho, p);
		p = gdl_hstruct_parameters_new_point (r, gdl_hstruct_point_f, NULL);
		gdl_hstruct_config_set_point (c, gdl_hstruct_point_f, p);
	}
	
	gdl_list_push_front (r->configs, c, 0);
	
	return c;
}

const gdl_hstruct_config *
gdl_hstruct_parameters_last_config (gdl_hstruct_parameters * r)
{
	return (gdl_hstruct_config *) gdl_list_get (r->configs, 0);	
}

gdl_hstruct_config *
gdl_hstruct_parameters_clone_config (gdl_hstruct_parameters * r, const gdl_hstruct_config * c)
{
	gdl_hstruct_config * nc = gdl_hstruct_config_clone (c);
	gdl_list_push_front (r->configs, nc, 0);
	return nc;
}

gdl_hstruct_point_handler *
gdl_hstruct_parameters_new_handler (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * extra)
{
	void ** p = gdl_hstruct_parameters_new_point (r, T, extra);
	return gdl_hstruct_point_handler_alloc (T, p);
}

void
gdl_hstruct_parameters_clean_handler (gdl_hstruct_parameters * r, gdl_hstruct_point_handler * h)
{
	const gdl_hstruct_point_type * T = gdl_hstruct_point_handler_type (h);
	void ** p = gdl_hstruct_point_handler_point_ptr	(h);
	gdl_hstruct_parameters_clean_point (r, T, p);
	gdl_hstruct_point_handler_free (h);
}

void **
gdl_hstruct_parameters_new_point (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * extra)
{
	void ** p = GDL_CALLOC (void *, 1);
	
	if (T == gdl_hstruct_point_f)
	{
		if (extra)
		{
			size_t k = *((size_t *)extra);
			if (k > 1)
			{
				*p = gdl_hstruct_fpoint_alloc (k);
			}
			else
			{
				*p = NULL;
				return p;
			}
		}
		else
		{
		    *p = gdl_hstruct_fpoint_alloc (r->k);
		}
	}
	else if (T == gdl_hstruct_point_rho)
	{
		*p = gdl_hstruct_point_alloc (log(10), 0, GDL_POSINF, NULL);
	}
	else if (T == gdl_hstruct_point_mu)
	{
		*p = gdl_hstruct_point_alloc (1.e-4, 0, 1, &gdl_hstruct_point_mut_prior);
	}
	else if (T == gdl_hstruct_point_hot)
	{
		*p = gdl_hstruct_point_alloc (10, 1, GDL_POSINF, &gdl_hstruct_point_hot_prior);
	}
	
	_add_point (r, T, *p);
	
	return p;
}

void
gdl_hstruct_config_block_update (gdl_hstruct_parameters * params, gdl_hstruct_config_block * b)
{
	size_t k;
	gdl_hstruct_config_collector * collector =  gdl_hstruct_config_block_config_collector (b);
	gdl_hstruct_config_collector * new_collector;
	gdl_hstruct_config_itr * itr;
	
	new_collector = gdl_hstruct_config_collector_alloc (1);
	
	itr = gdl_hstruct_config_collector_iterator (collector);
	do
	{
		gdl_hstruct_config * c = gdl_hstruct_config_collector_iterator_config (itr);
		gdl_hstruct_parameters_clean_config (params, c);
		c = gdl_hstruct_config_buffer_new (b->buffer, c);
		k = c->k;
		gdl_hstruct_config_collector_attach (new_collector, c);
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	gdl_hstruct_config_collector_iterator_free (itr);
		
	gdl_hstruct_config_collector_free (b->collector);
	gdl_hstruct_config_buffer_free (b->buffer);
	
	b->collector = new_collector;
	b->buffer    = gdl_hstruct_config_buffer_alloc (new_collector);
	b->k         = k;
}

void
gdl_hstruct_parameters_clean_config (gdl_hstruct_parameters * r, gdl_hstruct_config * c)
{
	if (r && c)
	{
		void ** p;
		
		p = gdl_hstruct_config_get_point_ptr (c, gdl_hstruct_point_rho);
		if (p && !gdl_hstruct_config_point_inuse (c, gdl_hstruct_point_rho))
		{
			gdl_hstruct_parameters_clean_point (r, gdl_hstruct_point_rho, p);
		}
		p = gdl_hstruct_config_get_point_ptr (c, gdl_hstruct_point_hot);
		if (p && !gdl_hstruct_config_point_inuse (c, gdl_hstruct_point_hot))
		{
			gdl_hstruct_parameters_clean_point (r, gdl_hstruct_point_hot, p);
		}
		p = gdl_hstruct_config_get_point_ptr (c, gdl_hstruct_point_mu);
		if (p && !gdl_hstruct_config_point_inuse (c, gdl_hstruct_point_mu))
		{
			gdl_hstruct_parameters_clean_point (r, gdl_hstruct_point_mu, p);
		}
		p = gdl_hstruct_config_get_point_ptr (c, gdl_hstruct_point_f);
		if (p && !gdl_hstruct_config_point_inuse (c, gdl_hstruct_point_f))
		{
			gdl_hstruct_parameters_clean_point (r, gdl_hstruct_point_f, p);
		}
	}
}

void
gdl_hstruct_parameters_clean_point (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void ** p)
{
	if (*p)
	{
		_rm_point (r, T, *p);
		
		if (T == gdl_hstruct_point_f)
		{
			gdl_hstruct_fpoint_free ((gdl_hstruct_fpoint *)*p);
			(*p)=NULL;
		}
		else 
		{
			gdl_hstruct_point_free ((gdl_hstruct_point *)*p);
			(*p)=NULL;
		}
	}	
}

int
gdl_hstruct_parameters_get_point_index (const gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * p)
{
	if (p)
	{
		gdl_hstruct_parameter * pr = _get_parameter (r, T, p);
		if (pr)
		{
			return pr->idx;
		}
	}
	return -1;
}

gdl_hstruct_config *
gdl_hstruct_parameters_attach_config (gdl_hstruct_parameters * r, size_t c, size_t p, gdl_hstruct_config * conf, gdl_locus * locus)
{
	gdl_hstruct_config * old = gdl_hstruct_config_detach (locus);
	
	if (old)
	{
		_collapse_config_domain (r, old, c, p);
		gdl_hstruct_config_collector_detach (r->collector, old);
	}
	
	_extend_config_domain (r, conf, c, p);
	
	gdl_hstruct_config_attach (conf, locus);
	gdl_hstruct_config_collector_attach (r->collector, conf);
	
	return old;
}

gdl_hstruct_config *
gdl_hstruct_parameters_detach_config (gdl_hstruct_parameters * r, size_t c, size_t p, gdl_locus * locus)
{
	gdl_hstruct_config * old = gdl_hstruct_config_detach (locus);
	
	if (old)
	{
		_collapse_config_domain (r, old, c, p);
		gdl_hstruct_config_collector_detach (r->collector, old);
	}
	
	return old;
}

int
gdl_hstruct_parameters_update_config_domain (gdl_hstruct_parameters * r, gdl_hstruct_config * conf, size_t oc, size_t op, size_t c, size_t p)
{
	_collapse_config_domain (r, conf, oc, op);
	_extend_config_domain (r, conf, c, p);
}

int
gdl_hstruct_parameters_attach_handler (gdl_hstruct_parameters * r, size_t c, size_t p, gdl_hstruct_point_handler * h)
{
	_extend_point_domain (r,
                             gdl_hstruct_point_handler_type (h),
                             gdl_hstruct_point_handler_point (h),
                             c,
                             p);
    
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameters_detach_handler (gdl_hstruct_parameters * r, size_t c, size_t p, gdl_hstruct_point_handler * h)
{
	_collapse_point_domain (r,
                             gdl_hstruct_point_handler_type (h),
                             gdl_hstruct_point_handler_point (h),
                             c,
                             p);
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameters_flush_buffer (gdl_hstruct_parameters * r, gdl_hstruct_config_buffer * t, gdl_hstruct_config * new)
{
	gdl_hstruct_config_buffer_flush (t, new);
	gdl_list_push_front (r->configs, new, 0);
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameters_unflush_buffer (gdl_hstruct_parameters * r, gdl_hstruct_config_buffer * t, gdl_hstruct_config * new)
{
	gdl_hstruct_config_buffer_unflush (t, new);
	gdl_list_remove_all (r->configs, new);
	return GDL_SUCCESS;
}


const gdl_hstruct_parameter_domain *
gdl_hstruct_parameters_get_domain (gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, void * p)
{
	return _get_domain (r, T, p);
}

int
gdl_hstruct_parameters_backup_points (gdl_hstruct_parameters * r, const gdl_hstruct_config_collector * collector)
{
	gdl_hstruct_config * conf;
	gdl_hstruct_config_itr * itr;
	
	itr = gdl_hstruct_config_collector_iterator (collector);
	
	if (itr)
	{
		do
		{
			conf  = gdl_hstruct_config_collector_iterator_config (itr);
			if (gdl_hstruct_config_collector_size (collector, conf))
			{
				gdl_hstruct_parameters_backup_config_points (r, conf);
			}
		}
		while (gdl_hstruct_config_collector_iterator_next (itr));
		
		gdl_hstruct_config_collector_iterator_free (itr);
	}
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameters_backup_config_points (gdl_hstruct_parameters * r, const gdl_hstruct_config * config)
{
	_backup_config_points (r, config);
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameters_restore_points (gdl_hstruct_parameters * r, const gdl_hstruct_config_collector * collector)
{
	gdl_hstruct_config * conf;
	gdl_hstruct_config_itr * itr;
	
	itr = gdl_hstruct_config_collector_iterator (collector);
	
	if (itr)
	{
		do
		{
			conf  = gdl_hstruct_config_collector_iterator_config (itr);
			if (gdl_hstruct_config_collector_size (collector, conf))
			{
				gdl_hstruct_parameters_restore_config_points (r, conf);
			}
		}
		while (gdl_hstruct_config_collector_iterator_next (itr));
		
		gdl_hstruct_config_collector_iterator_free (itr);
	}
	return GDL_SUCCESS;
}

int
gdl_hstruct_parameters_restore_config_points (gdl_hstruct_parameters * r, const gdl_hstruct_config * config)
{
	_restore_config_points (r, config);
	return GDL_SUCCESS;
}

void *
gdl_hstruct_parameters_get_point (const gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T, size_t i)
{
	gdl_hstruct_parameter_itr * itr;
	void * p = NULL;
	
	itr = gdl_hstruct_parameter_iterate (r, T);
	
	do
	{
		if (i == gdl_hstruct_parameter_iterate_point_idx (itr))
		{
			p = gdl_hstruct_parameter_iterate_point (itr);
			break;	
		}
	}
	while (gdl_hstruct_parameter_iterate_next (itr));
	
	gdl_hstruct_parameter_iterate_free (itr);
	
	return p;
}

gdl_hstruct_config *
gdl_hstruct_parameters_setup_config (gdl_hstruct_parameters * r, const gdl_hstruct_static_config * sconf)
{
	if (!sconf)
	{
		return NULL;	
	}
	
	gdl_string * template;
	
	template = gdl_string_sprintf ("%d-%d-%d-%d-%d", sconf->k, sconf->rho_idx, sconf->hot_idx, sconf->mu_idx, sconf->f_idx); 
	
	gdl_hstruct_config * conf = gdl_hashtable_lookup (r->setup, template);
	
	if (!conf)
	{
		void ** p;
		
		conf = gdl_hstruct_config_alloc (sconf->k);
		
		if (sconf->rho_idx>=0)
		{
			p = GDL_MALLOC (void *, 1);
			*p = gdl_hstruct_parameters_get_point (r, gdl_hstruct_point_rho, sconf->rho_idx);
			gdl_hstruct_config_set_point (conf, gdl_hstruct_point_rho, p);
		}
		if (sconf->hot_idx>=0)
		{
			p  = GDL_MALLOC (void *, 1);
			*p = gdl_hstruct_parameters_get_point (r, gdl_hstruct_point_hot, sconf->hot_idx);
			gdl_hstruct_config_set_point (conf, gdl_hstruct_point_hot, p);
		}
		if (sconf->mu_idx>=0)
		{
			p  = GDL_MALLOC (void *, 1);
			*p = gdl_hstruct_parameters_get_point (r, gdl_hstruct_point_mu, sconf->mu_idx);
			gdl_hstruct_config_set_point (conf, gdl_hstruct_point_mu, p);
		}
		if (sconf->f_idx>=0)
		{
			p = GDL_MALLOC (void *, 1);
			*p = gdl_hstruct_parameters_get_point (r, gdl_hstruct_point_f, sconf->f_idx);
			gdl_hstruct_config_set_point (conf, gdl_hstruct_point_f, p);
		}
	
		gdl_list_push_front (r->configs, conf, 0);
		
		gdl_hashtable_add (r->setup, template, conf, 0);
	}
	
	gdl_string_free (template);
	
	return conf;
}

gdl_hstruct_point_array *
gdl_hstruct_parameters_get_point_array (const gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T)
{
	size_t idx, size;
	void * point;
	gdl_hstruct_point_array * a;
	gdl_hstruct_parameter_itr * itr;
   
   size  = 0;
   itr   = gdl_hstruct_parameter_iterate	(r, T);
   if (!itr)
   {
   	return gdl_hstruct_point_array_alloc (T, 0);
   }
   do
   {
   	(size)++;
   }
   while (gdl_hstruct_parameter_iterate_next (itr));
   gdl_hstruct_parameter_iterate_free (itr);
   
   a = gdl_hstruct_point_array_alloc (T, size);
   itr = gdl_hstruct_parameter_iterate	(r, T);
   do
   {
   	point       = gdl_hstruct_parameter_iterate_point (itr);
   	idx         = gdl_hstruct_parameter_iterate_point_idx (itr);
   	gdl_hstruct_point_array_set (a, idx, (T->clone)(point));
   }
   while (gdl_hstruct_parameter_iterate_next (itr));
   gdl_hstruct_parameter_iterate_free (itr);
   
   return a;	
}

typedef gdl_hashtable_itr _gdl_hstruct_parameter_itr;

gdl_hstruct_parameter_itr *
gdl_hstruct_parameter_iterate (const gdl_hstruct_parameters * r, const gdl_hstruct_point_type * T)
{
	gdl_hashtable * table = _get_table (r, T);
	
	if (gdl_hashtable_size (table))
	{
		return gdl_hashtable_iterator (table);
	}
	else
	{
		return NULL;
	}
}

gdl_boolean
gdl_hstruct_parameter_iterate_next (gdl_hstruct_parameter_itr * itr)
{
	return (gdl_hashtable_iterator_next (itr)) ? gdl_true : gdl_false;
}

void *
gdl_hstruct_parameter_iterate_point (gdl_hstruct_parameter_itr * itr)
{
	gdl_hstruct_parameter * pr 
	   = (gdl_hstruct_parameter *) gdl_hashtable_iterator_value (itr);
	return pr->p;
}

size_t
gdl_hstruct_parameter_iterate_point_idx (gdl_hstruct_parameter_itr * itr)
{
	gdl_hstruct_parameter * pr 
	   = (gdl_hstruct_parameter *) gdl_hashtable_iterator_value (itr);
	return pr->idx;
}

void
gdl_hstruct_parameter_iterate_free (gdl_hstruct_parameter_itr * itr)
{
	gdl_hashtable_iterator_free (itr);	
}

struct _gdl_hstruct_static_parameter_registry
{
	gdl_hashtable * points;	
};

struct _gdl_hstruct_static_parameters
{
	size_t owner;
	gdl_hstruct_static_parameter_registry * registry;
	gdl_hashtable * configs;
};

static gdl_hstruct_static_parameter_registry *
_gdl_hstruct_static_parameter_registry_alloc (const gdl_hstruct_parameters * r)
{
	gdl_hstruct_static_parameter_registry * reg;
	
	reg = GDL_MALLOC (gdl_hstruct_static_parameter_registry, 1);
	
	reg->points = gdl_hashtable_alloc (gdl_hstruct_point_array_interface, 4);
	
	gdl_hashtable_add (reg->points, gdl_hstruct_point_rho->name, gdl_hstruct_parameters_get_point_array (r, gdl_hstruct_point_rho), 1);
	gdl_hashtable_add (reg->points, gdl_hstruct_point_mu->name, gdl_hstruct_parameters_get_point_array (r, gdl_hstruct_point_mu), 1);
	gdl_hashtable_add (reg->points, gdl_hstruct_point_hot->name, gdl_hstruct_parameters_get_point_array (r, gdl_hstruct_point_hot), 1);
	gdl_hashtable_add (reg->points, gdl_hstruct_point_f->name, gdl_hstruct_parameters_get_point_array (r, gdl_hstruct_point_f), 1);
	
	return reg;
}

static gdl_hstruct_static_config * 
_get_static_config (const gdl_hstruct_parameters * r, const gdl_hstruct_config * c)
{
	gdl_hstruct_static_config * sc = gdl_hstruct_static_config_alloc (c->k);
	
	sc->rho_idx = gdl_hstruct_parameters_get_point_index (r, gdl_hstruct_point_rho, gdl_hstruct_config_get_point (c, gdl_hstruct_point_rho));
	sc->mu_idx  = gdl_hstruct_parameters_get_point_index (r, gdl_hstruct_point_mu, gdl_hstruct_config_get_point (c, gdl_hstruct_point_mu));
	sc->hot_idx = gdl_hstruct_parameters_get_point_index (r, gdl_hstruct_point_hot, gdl_hstruct_config_get_point (c, gdl_hstruct_point_hot));
	sc->f_idx   = gdl_hstruct_parameters_get_point_index (r, gdl_hstruct_point_f, gdl_hstruct_config_get_point (c, gdl_hstruct_point_f));
	
	return sc;	
}

static void
_static_add_configs (gdl_hstruct_static_parameters * rs, const gdl_hstruct_parameters * r)
{
	gdl_hstruct_config_itr * itr;
	
	itr = gdl_hstruct_config_collector_iterator (r->collector);
	
	do
	{
		const gdl_hstruct_config * c = gdl_hstruct_config_collector_iterator_config (itr);
		gdl_hashtable_vadd (rs->configs, c, _get_static_config (r, c), 0);
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	
	gdl_hstruct_config_collector_iterator_free (itr);
}

gdl_hstruct_static_parameters *
gdl_hstruct_static_parameters_alloc (const gdl_hstruct_parameters * r)
{
	gdl_hstruct_static_parameters * rs;
	
	rs = GDL_MALLOC (gdl_hstruct_static_parameters, 1);
	
	rs->owner     = 1;
	rs->registry  = _gdl_hstruct_static_parameter_registry_alloc (r);
	rs->configs   = gdl_hashtable_alloc (gdl_hash_default, gdl_list_size (r->configs));
	
	_static_add_configs (rs, r);
	
	return rs;
}

void
gdl_hstruct_static_parameters_free (gdl_hstruct_static_parameters * rs)
{
	if (rs)
	{
		size_t i;
		gdl_hashtable_itr * itr;
		
		if (rs->owner)
		{
			gdl_hstruct_static_parameter_registry_free (rs->registry);
		}
				
		itr = gdl_hashtable_iterator (rs->configs);
		
		do
		{
			gdl_hstruct_static_config * c =
			 (gdl_hstruct_static_config *) gdl_hashtable_iterator_value (itr);
			gdl_hstruct_static_config_free (c);
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
		
		gdl_hashtable_free (rs->configs);
		
		GDL_FREE (rs);
	}
}

const gdl_hstruct_static_parameter_registry *
gdl_hstruct_static_parameters_get_registry (const gdl_hstruct_static_parameters * rs)
{
	return rs->registry;	
}

gdl_hstruct_static_parameter_registry *
gdl_hstruct_static_parameters_detach_registry (gdl_hstruct_static_parameters * rs)
{
	rs->owner = 0;
	return rs->registry;	
}

gdl_hstruct_static_config *
gdl_hstruct_static_parameters_new_config (const gdl_hstruct_static_parameters * rs, const gdl_hstruct_config * c)
{
	return gdl_hstruct_static_config_clone ((gdl_hstruct_static_config *) gdl_hashtable_vlookup (rs->configs, c));
}

void
gdl_hstruct_static_parameter_registry_free (gdl_hstruct_static_parameter_registry * reg)
{
	if (reg)
	{
		gdl_hashtable_free (reg->points);
		GDL_FREE (reg);
	}
}

size_t
gdl_hstruct_static_parameter_registry_size (const gdl_hstruct_static_parameter_registry * rs, const gdl_hstruct_point_type * T)
{
	gdl_hstruct_point_array * a 
	  = (gdl_hstruct_point_array *) gdl_hashtable_lookup (rs->points, T->name);
	if (a)
	{
		return gdl_hstruct_point_array_size (a);
	}
	return 0;
}

void *
gdl_hstruct_static_parameter_registry_get (const gdl_hstruct_static_parameter_registry * rs, const gdl_hstruct_point_type * T, size_t i)
{
	gdl_hstruct_point_array * a 
	  = (gdl_hstruct_point_array *) gdl_hashtable_lookup (rs->points, T->name);
	if (a)
	{
		return gdl_hstruct_point_array_get (a, i);
	}
	return NULL;
}

gdl_hstruct_static_parameter_registry *
gdl_hstruct_static_parameter_registry_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_hstruct_static_parameter_registry * r;
		
		r = GDL_MALLOC (gdl_hstruct_static_parameter_registry, 1);
		
		r->points = gdl_hashtable_alloc (gdl_hstruct_point_array_interface, 0);
		
		status = gdl_hashtable_fread (stream, r->points);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return r;
	}
	
	return NULL;	
}

int 
gdl_hstruct_static_parameter_registry_fwrite (FILE * stream, const gdl_hstruct_static_parameter_registry * r)
{
	if (stream && r)
	{
		int status;
		
		status = gdl_hashtable_fwrite (stream, r->points);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

static int
gdl_hstruct_parameters_setup_points (gdl_hstruct_parameters * r, const gdl_hstruct_static_parameter_registry * reg, const gdl_hstruct_point_type * T)
{
	size_t i, n;
	void * p, ** np;
	
	n = gdl_hstruct_static_parameter_registry_size (reg, T);
	
	for (i = 0; i < n; i++)
	{
		p  = gdl_hstruct_static_parameter_registry_get (reg, T, i);
		np = gdl_hstruct_parameters_new_point (r, T, 0);
		(T->copy)(*np, p);
	}
	
	return GDL_SUCCESS;
}

gdl_hstruct_parameters *
gdl_hstruct_parameters_setup (const gdl_hstruct_static_parameter_registry * reg)
{
	size_t i, n, k;
	gdl_hstruct_fpoint * q;
	gdl_hstruct_parameters * r;
	
	n = gdl_hstruct_static_parameter_registry_size (reg, gdl_hstruct_point_f);
	for (k = i = 0; i < n; i++)
	{
		q = (gdl_hstruct_fpoint *) gdl_hstruct_static_parameter_registry_get (reg, gdl_hstruct_point_f, i);
		if (k < gdl_hstruct_fpoint_size (q))
		{
			k = gdl_hstruct_fpoint_size (q);
		}
	}
	
	r = gdl_hstruct_parameters_alloc (k);
	
	gdl_hstruct_parameters_setup_points (r, reg, gdl_hstruct_point_rho);
	gdl_hstruct_parameters_setup_points (r, reg, gdl_hstruct_point_hot);
	gdl_hstruct_parameters_setup_points (r, reg, gdl_hstruct_point_mu);
	gdl_hstruct_parameters_setup_points (r, reg, gdl_hstruct_point_f);
	
	r->setup = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	return r;
}
