/*  
 *  hstruct/parray.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_hstruct_point.h>


gdl_hstruct_point_array *
gdl_hstruct_point_array_alloc (const gdl_hstruct_point_type * T, size_t size)
{
	gdl_hstruct_point_array * a;
	
	a = GDL_CALLOC (gdl_hstruct_point_array, 1);
	
	a->type   = T;
	a->size   = size;
	
	if (size)
	{
		a->points = GDL_CALLOC (void *, size); 
	}
	
	return a;
}

void
gdl_hstruct_point_array_free (gdl_hstruct_point_array * a)
{
	if (a)
	{
		size_t i;
		
		for (i = 0; i < a->size; i++)
		{
			(a->type->free)(a->points[i]);
		}
		GDL_FREE (a->points);
		GDL_FREE (a);
	}
}

size_t
gdl_hstruct_point_array_size (gdl_hstruct_point_array * a)
{
	return a->size;	
}

void * 
gdl_hstruct_point_array_get (gdl_hstruct_point_array * a, size_t i)
{
	return a->points[i];
}

void
gdl_hstruct_point_array_set (gdl_hstruct_point_array * a, size_t i, void * point)
{
	a->points[i] = point;
}

gdl_hstruct_point_array *
gdl_hstruct_point_array_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, size;
		gdl_hstruct_point_array * a;
		const gdl_hstruct_point_type * T;
		
		status = fread (&size, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		T = gdl_hstruct_point_type_fread (stream);
		GDL_FREAD_STATUS (T!=0, 1);
		
		a = gdl_hstruct_point_array_alloc (T, size);
		
		for (i = 0; i < a->size; i++)
		{
			a->points[i] = (a->type->fread)(stream);
			GDL_FREAD_STATUS (status, 1);
		}	
		
		return a;
	}
	return NULL;
}

int
gdl_hstruct_point_array_fwrite (FILE * stream, const gdl_hstruct_point_array * a)
{
	if (stream && a)
	{
		int status;
		size_t i, size;
	
		status = fwrite (&(a->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_hstruct_point_type_fwrite (stream, a->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		for (i = 0; i < a->size; i++)
		{
			status = (a->type->fwrite)(stream, a->points[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

static void
_gdl_hstruct_point_array_free (void * va)
{
	if (va)
	{
		gdl_hstruct_point_array * a = (gdl_hstruct_point_array *) va;
		gdl_hstruct_point_array_free (a);	
	}
}

static void *
_gdl_hstruct_point_array_fread (FILE * stream)
{
	return gdl_hstruct_point_array_fread (stream);
}

static int
_gdl_hstruct_point_array_fwrite (FILE * stream, const void * va)
{
	gdl_hstruct_point_array * a = (gdl_hstruct_point_array *) va;
	return gdl_hstruct_point_array_fwrite (stream, a);
}

static const gdl_data_interface _gdl_hstruct_point_array_interface =
{
	&_gdl_hstruct_point_array_free,
	NULL,
	NULL,
   &_gdl_hstruct_point_array_fread,
   &_gdl_hstruct_point_array_fwrite
};

const gdl_data_interface * gdl_hstruct_point_array_interface = &_gdl_hstruct_point_array_interface;
