/*  
 * 	hstruct/partition.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_sort.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gpartition.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hstruct_hmm.h>
#include <gdl/gdl_hstruct_fuzzy.h>
#include <gdl/gdl_hstruct_point.h>
#include <gdl/gdl_hstruct_config.h>
#include <gdl/gdl_hstruct_parameters.h>
#include <gdl/gdl_hstruct_model_type.h>
#include <gdl/gdl_hstruct_model_chromosome.h>
#include <gdl/gdl_hstruct_model_partition.h>

struct _gdl_hstruct_model_partition
{
	size_t _lig;
	size_t cidx;
	size_t pidx;
	size_t pos;
	size_t nl;
	size_t k;
	const gdl_locus_type_registry * locus_type;
	const gdl_rng                 * rng;
	const gdl_chromosome          * chrom;
	gdl_gview_wrapper             * data;
	gdl_allele_block              * seq;
	gdl_hstruct_hmm               * hmm;
	gdl_hstruct_fuzzy             * fuzzy;
   size_t                        * haplotypes;
	gdl_hstruct_config_collector * collector;
};

gdl_hstruct_model_partition *
gdl_hstruct_model_partition_alloc ( const gdl_hstruct_data * d, 
                                    const gdl_mask * gmask,
                                    size_t cidx,
                                    size_t pidx,
                                    size_t pos)
{
	gdl_hstruct_data * pd;
	gdl_hstruct_model_partition * p;
	
	p = GDL_CALLOC (gdl_hstruct_model_partition, 1);
	
	p->cidx = cidx;
	p->pidx = pidx;
	p->pos  = pos;
	p->nl   = gdl_mask_get_size (gmask, GDL_LOCUS);
	
	p->collector = gdl_hstruct_config_collector_alloc (5);
		
	p->rng        = d->rng;
	p->locus_type = d->locus_type;
	p->chrom      = d->chrom;
	
	p->data = gdl_gview_wrapper_alloc (gdl_gview_wrapper_global, d->gview, gmask);
	p->seq  = gdl_gview_wrapper_allele_block_c (p->data, 0);
	
	return p;
}

void
gdl_hstruct_model_partition_free (gdl_hstruct_model_partition * p)
{
	if (p)
	{
		gdl_gview_wrapper_free (p->data);
		gdl_allele_block_free (p->seq);
		gdl_hstruct_fuzzy_free (p->fuzzy);
		gdl_hstruct_hmm_free (p->hmm);
		gdl_hstruct_config_collector_free (p->collector);
		GDL_FREE (p->haplotypes);
		GDL_FREE (p);
	}
}

static size_t
_gdl_hstruct_model_partition_get_ancestral (const void * haplo, size_t k, size_t l)
{
	return gdl_hstruct_model_partition_get_ancestral ((gdl_hstruct_model_partition *)haplo, k, l);
}

int
gdl_hstruct_model_partition_init (gdl_hstruct_model_partition * p, size_t k)
{
	p->k = k;
	p->haplotypes = GDL_CALLOC (size_t, k*p->nl);
	p->fuzzy = gdl_hstruct_fuzzy_alloc (p->data, p->seq, p->rng, k);
	gdl_hstruct_fuzzy_init (p->fuzzy);
	p->hmm   = gdl_hstruct_hmm_alloc (p->data, p->seq, p->chrom, p, &_gdl_hstruct_model_partition_get_ancestral);
	return GDL_SUCCESS;
}

gdl_gview_wrapper *
gdl_hstruct_model_partition_data (const gdl_hstruct_model_partition * p)
{
	return p->data;	
}

const gdl_chromosome *
gdl_hstruct_model_partition_chromosome (const gdl_hstruct_model_partition * p)
{
	return p->chrom;	
}

size_t
gdl_hstruct_model_partition_get_cidx (const gdl_hstruct_model_partition * p)
{
	return p->cidx;	
}

size_t
gdl_hstruct_model_partition_get_pidx (const gdl_hstruct_model_partition * p)
{
	return p->pidx;	
}

size_t
gdl_hstruct_model_partition_size (const gdl_hstruct_model_partition * p)
{
	return p->nl;	
}

size_t
gdl_hstruct_model_partition_ancestral_size (const gdl_hstruct_model_partition * p)
{
	return p->k;	
}

size_t
gdl_hstruct_model_partition_ligation_site (const gdl_hstruct_model_partition * p)
{
	return p->_lig;
}

void
gdl_hstruct_model_partition_reset_ligation_site (gdl_hstruct_model_partition * p)
{
	p->_lig=0;
}

void
gdl_hstruct_model_partition_set_ancestral (gdl_hstruct_model_partition * p, size_t k, size_t l, size_t a)
{
	p->haplotypes[k*p->nl + l] = a;
}

size_t
gdl_hstruct_model_partition_get_ancestral (const gdl_hstruct_model_partition * p, size_t k, size_t l)
{
	return p->haplotypes[k*p->nl + l];
}

void
gdl_hstruct_model_partition_swap_ancestral (gdl_hstruct_model_partition * p, size_t k1, size_t k2, size_t l)
{
	double t;
	
	t = p->haplotypes[k1*p->nl + l];
	p->haplotypes[k1*p->nl + l] = p->haplotypes[k2*p->nl + l];
	p->haplotypes[k2*p->nl + l] = t;
}

void
gdl_hstruct_model_partition_permut_ancestral (gdl_hstruct_model_partition * p, size_t from, size_t to, size_t * permut, size_t k)
{
	size_t i, j, l, * idx, * swap;
	gdl_hashtable * buf;
	gdl_hstruct_fpoint * f;
	const gdl_hstruct_config * c;
	
	// First get the swap cycle
	idx  = GDL_MALLOC (size_t, k);
	swap = GDL_MALLOC (size_t, k);
	for (i = 0; i < k; i++) idx[i] = i;
	for (i = 0; i < k; i++)
	{
		if (permut[i] != i)
		{
			swap[i] = idx[permut[i]];
			j = idx[permut[i]];
			idx[permut[i]] = i;
			idx[idx[i]] = j;
		}
		else
		{
			swap[i] = i;
		}
	}
	GDL_FREE (idx);
	
	// now loop on the locus
	buf = gdl_hashtable_alloc (gdl_hash_default, 0);
	for (l = from; l <= to; l++)
	{
		c = gdl_hstruct_model_partition_get_config (p, l);
		f = gdl_hstruct_config_get_point (c, gdl_hstruct_point_f);
		if (f)
		{
			if (gdl_hashtable_vlookup (buf, f))
			{
				f = NULL;
			}
			else
			{
				gdl_hashtable_vadd (buf, f, f, 0);
			}
		}
		for (i = 0; i < GDL_MIN (k, c->k); i++)
		{
			if (swap[i] < c->k && swap[i] != i)
			{
				gdl_hstruct_model_partition_swap_ancestral (p, i, swap[i], l);	
				if (f)
				{
					gdl_hstruct_fpoint_swap (f, i, swap[i]);	
				}
			}
		}		
	}
	gdl_hashtable_free (buf);
	GDL_FREE (swap);
}

gdl_gdistance *
gdl_hstruct_model_partition_get_distance (const gdl_hstruct_model_partition * p, size_t l)
{
	return gdl_chromosome_get_distance (p->chrom, p->pos + l, p->pos + l + 1);
}

gdl_locus *
gdl_hstruct_model_partition_get_locus (const gdl_hstruct_model_partition * p, size_t l)
{
	return gdl_chromosome_get (p->chrom, p->pos + l);
}

const gdl_locus_type *
gdl_hstruct_model_partition_get_locus_type (const gdl_hstruct_model_partition * p, size_t l)
{
	if (p->locus_type)
	{
		const gdl_locus * locus = gdl_hstruct_model_partition_get_locus (p, l);
		return gdl_locus_type_registry_search (p->locus_type, locus);
	}
	else
	{
		return NULL;	
	}
}

gdl_hstruct_config *
gdl_hstruct_model_partition_get_config (const gdl_hstruct_model_partition * p, size_t l)
{
	return gdl_hstruct_config_get (gdl_hstruct_model_partition_get_locus (p, l));
}

size_t
gdl_hstruct_model_partition_get_kmax (const gdl_hstruct_model_partition * p)
{
	size_t l, kmax;
	gdl_hstruct_config * c;
	
	c = gdl_hstruct_model_partition_get_config (p, 0);
	kmax = c->k;
	
	for (l = 1; l < p->nl; l++)
	{
		c = gdl_hstruct_model_partition_get_config (p, l);
		if (c->k > kmax)
		{
			kmax = c->k;	
		}
	}
	
	return kmax;	
}

int
gdl_hstruct_model_partition_fuzzy2hmm (gdl_hstruct_model_partition * p)
{
	size_t i, l, k, a, n, * idx;
	double q, * tmp;
	gdl_locus * locus;
	gdl_hstruct_config * conf;
	gdl_hstruct_fpoint * fpoint;
	
	n = gdl_gview_wrapper_accession_size (p->data);
	
	if (p->k > 1)
	{
		tmp = GDL_CALLOC (double, p->k);
		idx = GDL_MALLOC (size_t, p->k);
		for (i = 0; i < n; i++)
		{
			for (k = 0; k < p->k; k++)
			{
				tmp[k] += gdl_hstruct_fuzzy_get_q (p->fuzzy, k, i);
			}
		}
		gdl_sort_index (idx, tmp, 1, p->k);
		GDL_FREE (tmp);
	}
	
	
	for (l = 0; l < p->nl; l++)
	{
		for (k = 0; k < p->k; k++)
		{
			a = gdl_hstruct_fuzzy_get_f_max (p->fuzzy, (p->k > 1) ? idx[k] : k, l);
			gdl_hstruct_model_partition_set_ancestral (p, k, l, a);
		}
		conf   = gdl_hstruct_model_partition_get_config (p, l);
		fpoint = gdl_hstruct_config_get_point (conf, gdl_hstruct_point_f);
		if (conf->k > 1)
		{
			for (k = 0; k < GDL_MIN (p->k, conf->k); k++)
			{
				for (i = 0; i < n; i++)
				{
					q = gdl_hstruct_fuzzy_get_q (p->fuzzy, (p->k > 1) ? idx[k] : k, i);
					gdl_hstruct_fpoint_collect (fpoint, k, q);
				}	
			}
		}
		else if (fpoint)
		{
			gdl_hstruct_fpoint_collect (fpoint, 0, 1);
		}
	}
	
	if (p->k > 1)
	{
		GDL_FREE (idx);
	}
	
	return GDL_SUCCESS;
}

gdl_hstruct_hmm *
gdl_hstruct_model_partition_get_hmm (const gdl_hstruct_model_partition * p)
{
	return p->hmm;	
}

gdl_hstruct_fuzzy *
gdl_hstruct_model_partition_get_fuzzy (const gdl_hstruct_model_partition * p)
{
	return p->fuzzy;	
}

int
gdl_hstruct_model_partition_attach_config (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config * c, size_t l)
{
	gdl_locus * locus;
	gdl_hstruct_config * old;
	
	locus = gdl_hstruct_model_partition_get_locus (p, l);
	
	old = gdl_hstruct_parameters_attach_config (pr, p->cidx, p->pidx, c, locus);
	
	gdl_hstruct_config_collector_detach (p->collector, old);
	
	return gdl_hstruct_config_collector_attach (p->collector, c);
}

gdl_hstruct_config_buffer *
gdl_hstruct_model_partition_new_buffer (gdl_hstruct_model_partition * pm)
{
	return gdl_hstruct_config_buffer_alloc (pm->collector);
}

int
gdl_hstruct_model_partition_update (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_buffer * t, size_t l)
{
	gdl_locus * locus;
	gdl_hstruct_config * old, * new;
	
	locus = gdl_hstruct_model_partition_get_locus (p, l);
	
	old = gdl_hstruct_config_get (locus);
	
	new = gdl_hstruct_config_buffer_new (t, old);
	
	if (!gdl_hstruct_config_collector_size (p->collector, new))
	{
		gdl_hstruct_parameters_flush_buffer (pr, t, new);
	}
	
	gdl_hstruct_model_partition_attach_config (p, pr, new, l);
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_model_partition_restore (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_buffer * t, size_t l)
{
	gdl_locus * locus;
	gdl_hstruct_config * old, * new;
	
	locus = gdl_hstruct_model_partition_get_locus (p, l);
	
	new = gdl_hstruct_config_get (locus);
	
	old = gdl_hstruct_config_buffer_old (t, new);
	
	gdl_hstruct_model_partition_attach_config (p, pr, old, l);
	
	if (!gdl_hstruct_config_collector_size (p->collector, new))
	{
		gdl_hstruct_parameters_unflush_buffer (pr, t, new);
	}
	
	return GDL_SUCCESS;
}

double
gdl_hstruct_model_partition_loglikelihood (const gdl_hstruct_model_partition * p)
{
	return gdl_hstruct_hmm_get_loglikelihood (p->hmm);
}

static gdl_mask *
_ligation_init (gdl_hstruct_model_partition * p1,
                const gdl_hstruct_model_partition * p2,
                void * extra)
{
	size_t i, j, k, nl, * haplo;
	gdl_locus * locus;
	gdl_hstruct_config * conf;
	gdl_mask * gmask;
	gdl_accession_mask * ma1, * ma2;
	
	gmask = gdl_mask_ligation (gdl_gview_wrapper_gmask (p1->data),
	                           gdl_gview_wrapper_gmask (p2->data),
	                           GDL_LOCUS,
	                           GDL_ACCESSION,
	                           &ma1,
	                           &ma2);
	// change p1...
	gdl_gview_wrapper_free (p1->data);
	gdl_allele_block_free (p1->seq);
	
	p1->data = gdl_gview_wrapper_alloc (gdl_gview_wrapper_global, gdl_gview_wrapper_gview (p2->data), gmask);
	p1->seq  = gdl_gview_wrapper_allele_block_c (p1->data, 0);
	
	k     = GDL_MAX (p1->k, p2->k);
	nl    = p1->nl + p2->nl;
	haplo = GDL_MALLOC (size_t, k*nl);
	
	for (i = 0; i < p1->nl; i++)
	{
		locus = gdl_hstruct_model_partition_get_locus (p1, i);
		conf  = gdl_hstruct_config_get (locus);
		for (j = 0; j < conf->k; j++)
		{
			haplo[j*nl+i] = gdl_hstruct_model_partition_get_ancestral (p1, j, i);
		}
	}
	for (i = 0; i < p2->nl; i++)
	{
		locus = gdl_hstruct_model_partition_get_locus (p2, i);
		conf  = gdl_hstruct_config_get (locus);
		for (j = 0; j < conf->k; j++)
		{
			haplo[j*nl+i+p1->nl] = gdl_hstruct_model_partition_get_ancestral (p2, j, i);
		}
	}
	
	p1->_lig = p1->nl-1;
	p1->nl   = nl;
	p1->k    = k;
	
	GDL_FREE (p1->haplotypes);
	
	p1->haplotypes = haplo;
	
	gdl_entity_mask_free (ma1);
	gdl_entity_mask_free (ma2);
	
	return gmask;
}

static void
_ligation_param (gdl_hstruct_parameters * pr, gdl_hstruct_model_partition * p1, const gdl_hstruct_model_partition * p2, const gdl_gligation_type * T)
{
	gdl_hstruct_config * conf;
	gdl_hstruct_config_itr * itr;
	
	if (p1->pidx)
	{
		size_t sub;
		if (T == gdl_gligation_progressive)
		{
			sub = 1;
		}
		else if (T == gdl_gligation_hierarchical)
		{
			sub = p1->pidx / 2;
		}
		itr = gdl_hstruct_config_collector_iterator (p1->collector);
		do
		{
			conf = gdl_hstruct_config_collector_iterator_config (itr);
			gdl_hstruct_parameters_update_config_domain (pr, conf, p1->cidx, p1->pidx, p1->cidx, p1->pidx-sub);
		}
		while (gdl_hstruct_config_collector_iterator_next (itr));
		gdl_hstruct_config_collector_iterator_free (itr);
		
		p1->pidx -= sub;
	}
	itr = gdl_hstruct_config_collector_iterator (p2->collector);
	do
	{
		conf = gdl_hstruct_config_collector_iterator_config (itr);
		gdl_hstruct_parameters_update_config_domain (pr, conf, p2->cidx, p2->pidx, p1->cidx, p1->pidx);
		gdl_hstruct_config_collector_attach (p1->collector, conf);
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	gdl_hstruct_config_collector_iterator_free (itr);
}

int
_partition_update (gdl_gpartition_state * cur, const gdl_gligation_type * T, void * extra)
{
	size_t sub;
	gdl_hstruct_config_itr * itr;
	gdl_hstruct_config * conf;
	gdl_hstruct_model_partition * p1;
	gdl_hstruct_parameters * pr = (gdl_hstruct_parameters *) extra;
	
	p1 = (gdl_hstruct_model_partition *) cur->data;
	
	if (p1->pidx)
	{
		if (T == gdl_gligation_progressive)
		{
			sub = 1;
		}
		else if (T == gdl_gligation_hierarchical)
		{
			sub = p1->pidx / 2;	
		}
		
		itr = gdl_hstruct_config_collector_iterator (p1->collector);
		do
		{
			conf = gdl_hstruct_config_collector_iterator_config (itr);
			gdl_hstruct_parameters_update_config_domain (pr, conf, p1->cidx, p1->pidx, p1->cidx, p1->pidx-sub);
		}
		while (gdl_hstruct_config_collector_iterator_next (itr));
		gdl_hstruct_config_collector_iterator_free (itr);
		p1->pidx -= sub;
	}
}

gdl_mask *
gdl_hstruct_model_partition_ligation (gdl_hstruct_parameters * pr,
                                      gdl_hstruct_model_partition * p1,
                                      const gdl_hstruct_model_partition * p2,
                                      const gdl_gligation_type * T,
                                      void * extra)
{
	gdl_mask * g;
		
	g = _ligation_init (p1, p2, extra);
	
	_ligation_param (pr, p1, p2, T);
	
	return g;
}

int
gdl_hstruct_model_partition_backup_points (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr)
{
	return gdl_hstruct_parameters_backup_points (pr, p->collector);
}

int
gdl_hstruct_model_partition_restore_points (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr)
{
	return gdl_hstruct_parameters_restore_points (pr, p->collector);
}

struct _gdl_hstruct_config_block_itr
{
	gdl_list     * blocks;
	gdl_list_itr * itr;
};

gdl_hstruct_config_block_itr *
gdl_hstruct_model_partition_config_block_iterator (gdl_hstruct_model_partition * p, const gdl_hstruct_config_template * template)
{
	size_t i, j;
	gdl_hstruct_config * prev, * next;
	gdl_hstruct_config_block * block;
	gdl_hstruct_config_block_itr * bitr;
	gdl_hstruct_config_collector * collector;
	
	bitr = GDL_MALLOC (gdl_hstruct_config_block_itr, 1);
	
	bitr->blocks = gdl_list_alloc (gdl_list_default);
	
	prev      = gdl_hstruct_model_partition_get_config (p, 0);
	collector = gdl_hstruct_config_collector_alloc (1);
	gdl_hstruct_config_collector_attach (collector, prev);
	for (j = 0, i = 1; i < p->nl; i++)
	{
		next = gdl_hstruct_model_partition_get_config (p, i);
		if (prev != next)
		{
			if (gdl_hstruct_config_template_compare (template, prev, next))
			{
				block = gdl_hstruct_config_block_alloc (collector, j, i-1);
				gdl_list_push_back (bitr->blocks, block, 0);
				collector = gdl_hstruct_config_collector_alloc (1);
				j = i;
			}
			gdl_hstruct_config_collector_attach (collector, next);
		}
		prev = next;
	}
	block = gdl_hstruct_config_block_alloc (collector, j, i-1);
	gdl_list_push_back (bitr->blocks, block, 0);
	
	bitr->itr = gdl_list_iterator_front (bitr->blocks);
	
	return bitr;
}

gdl_boolean
gdl_hstruct_config_block_iterator_next (gdl_hstruct_config_block_itr * itr, gdl_boolean free)
{
	if (free)
	{
		gdl_hstruct_config_block * b = (gdl_hstruct_config_block *) gdl_list_iterator_value (itr->itr);
		gdl_hstruct_config_block_free (b);
	}
	return (gdl_list_iterator_next (itr->itr)) ? gdl_true : gdl_false;
}

gdl_hstruct_config_block *
gdl_hstruct_config_block_iterator_get (gdl_hstruct_config_block_itr * itr)
{
	return (gdl_hstruct_config_block *) gdl_list_iterator_value (itr->itr);
}

void
gdl_hstruct_config_block_iterator_free (gdl_hstruct_config_block_itr * itr)
{
	gdl_list_iterator_free (itr->itr);	
	gdl_list_free (itr->blocks);
	GDL_FREE (itr);
}

double
gdl_hstruct_model_partition_ancestral_freq (const gdl_hstruct_model_partition * p, size_t k, size_t from, size_t to)
{
	return gdl_hstruct_hmm_get_ancestral_freq (p->hmm, k, from, to);
}

void
gdl_hstruct_model_partition_sort_ancestral (gdl_hstruct_model_partition * p)
{
	
}

gdl_hstruct_point_handler *
gdl_hstruct_model_partition_collapse_ancestral_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * block, size_t k)
{
	size_t i, j, from, to, kk;
	gdl_hstruct_config * config;
	gdl_hstruct_config_collector * collector;
	gdl_hstruct_config_itr * itr;
	gdl_hstruct_config_buffer * buffer;
	gdl_hstruct_point_handler * handler;
	gdl_hstruct_fpoint * f, * nf;
	
	kk = gdl_hstruct_config_block_ancestral_size (block);
	kk--;
	
	from      = gdl_hstruct_config_block_from (block);
	to        = gdl_hstruct_config_block_to (block);
	collector = gdl_hstruct_config_block_config_collector (block);
	buffer    = gdl_hstruct_config_block_buffer (block);
	handler   = gdl_hstruct_parameters_new_handler (pr, gdl_hstruct_point_f, &kk);
	
	if (kk >  1)
	{
		// Init the new vector of frequency
		nf = (gdl_hstruct_fpoint *) gdl_hstruct_point_handler_point (handler);
		itr = gdl_hstruct_config_collector_iterator (collector);
		do
		{
			config = gdl_hstruct_config_collector_iterator_config (itr);
			f  = (gdl_hstruct_fpoint *) gdl_hstruct_config_get_point (config, gdl_hstruct_point_f);
			for (j = i = 0; i <= kk; i++)
			{
				if (i != k)
				{
					gdl_hstruct_fpoint_collect (nf, j, gdl_hstruct_fpoint_value (f, i));
					j++;
				}
			}
		}
		while (gdl_hstruct_config_collector_iterator_next (itr));
		gdl_hstruct_config_collector_iterator_free (itr);
		gdl_hstruct_fpoint_update (nf, NULL, NULL);
	}
	
	gdl_hstruct_config_buffer_set (buffer, gdl_hstruct_point_f, gdl_hstruct_point_handler_point_ptr (handler));
	
	for (i = from; i <= to; i++)
	{
		gdl_hstruct_model_partition_update (p, pr, buffer, i);
		for (j = k; j < kk; j++)
		{
			gdl_hstruct_model_partition_swap_ancestral (p, j, j+1, i);
		}
	}
	
	return handler;
}

static void
_gdl_hstruct_model_partition_collapse_ancestral_block2 (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * hot, gdl_hstruct_point_handler * handler, size_t k)
{
	size_t i, j, kk, from, to;
	gdl_hstruct_config        * config;
	gdl_hstruct_config_buffer * buffer;
	
	// hotspot
	kk      = gdl_hstruct_config_block_ancestral_size (hot);
	kk--;
	
	from    = gdl_hstruct_config_block_from (hot);
	to      = gdl_hstruct_config_block_to (hot);
	buffer  = gdl_hstruct_config_block_buffer (hot);
	
	gdl_hstruct_config_buffer_set (buffer, gdl_hstruct_point_f, gdl_hstruct_point_handler_point_ptr (handler));
	
	for (i = from; i <= to; i++)
	{
		gdl_hstruct_model_partition_update (p, pr, buffer, i);
		
		for (j = k; j < kk; j++)
		{
			gdl_hstruct_model_partition_swap_ancestral (p, j, j+1, i);
		}
	}
}

gdl_hstruct_point_handler *
gdl_hstruct_model_partition_collapse_ancestral_block2 (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * hot1, gdl_hstruct_config_block * block, gdl_hstruct_config_block * hot2, size_t k)
{
	size_t kk, kh1=0, kh2=0;
	gdl_hstruct_point_handler * handler;
	
	if (hot1) kh1 = gdl_hstruct_config_block_ancestral_size (hot1);
   if (hot2) kh2 = gdl_hstruct_config_block_ancestral_size (hot2);
   
   kk = gdl_hstruct_config_block_ancestral_size (block);
   //printf ("COLLAPSE ANCESTRAL BLOCK [%d, %d, %d]\n", kh1, kk, kh2);
	kk--;
		
	handler = gdl_hstruct_model_partition_collapse_ancestral_block (p, pr, block, k);
	
	if (hot1 && kk <= kh1)
	{
		_gdl_hstruct_model_partition_collapse_ancestral_block2 (p, pr, hot1, handler, k);
	}
	
	if (hot2 && kk <= kh2)
	{
		_gdl_hstruct_model_partition_collapse_ancestral_block2 (p, pr, hot2, handler, k);
	}
	
	return handler;
}

void
gdl_hstruct_model_partition_restore_ancestral_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * block, size_t k)
{
	size_t i, j, from, to, kk;
	gdl_hstruct_config_buffer * buffer;
	gdl_hstruct_point_handler * handler;
	
	kk = gdl_hstruct_config_block_ancestral_size (block);
	
	from = gdl_hstruct_config_block_from (block);
	to   = gdl_hstruct_config_block_to (block);
	
	buffer = gdl_hstruct_config_block_buffer (block);
	
	for (i = from; i <= to; i++)
	{
		gdl_hstruct_model_partition_restore (p, pr, buffer, i);
		
		for (j = kk-1; j > k; j--)
		{
			gdl_hstruct_model_partition_swap_ancestral (p, j-1, j, i);
		}		
	}	
}

void
_gdl_hstruct_model_partition_restore_ancestral_block2 (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * hot, size_t k)
{
	size_t i, j, from, to, kk;
	gdl_hstruct_config * config;
	gdl_hstruct_config_buffer * buffer;
	
	kk = gdl_hstruct_config_block_ancestral_size (hot);

	from   = gdl_hstruct_config_block_from (hot);
	to     = gdl_hstruct_config_block_to (hot);
	buffer = gdl_hstruct_config_block_buffer (hot);
	
	for (i = from; i <= to; i++)
	{
		gdl_hstruct_model_partition_restore (p, pr, buffer, i);
		for (j = kk-1; j > k; j--)
		{
			gdl_hstruct_model_partition_swap_ancestral (p, j-1, j, i);
		}
	}
}

void
gdl_hstruct_model_partition_restore_ancestral_block2 (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * hot1, gdl_hstruct_config_block * block, gdl_hstruct_config_block * hot2, size_t k)
{
	size_t i, j, from, to, kk, kh1 = 0, kh2 = 0;
	gdl_hstruct_config_buffer * buffer;
	gdl_hstruct_point_handler * handler;
	
	if (hot1) kh1 = gdl_hstruct_config_block_ancestral_size (hot1);
   if (hot2) kh2 = gdl_hstruct_config_block_ancestral_size (hot2);
	
	kk = gdl_hstruct_config_block_ancestral_size (block);
	//printf ("RESTORE ANCESTRAL BLOCK [%d, %d, %d]\n", kh1, kk, kh2);
	kk--;
	
	gdl_hstruct_model_partition_restore_ancestral_block (p, pr, block, k);
	
	if (hot1 && kk <= kh1)
	{
		_gdl_hstruct_model_partition_restore_ancestral_block2 (p, pr, hot1, k);	
	}
	if (hot2 && kk <= kh2)
	{
		_gdl_hstruct_model_partition_restore_ancestral_block2 (p, pr, hot2, k);	
	}	
}

gdl_hstruct_point_handler *
gdl_hstruct_model_partition_remove_hotspot_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * block)
{
	size_t i, j, from, to, kk;
	gdl_hstruct_config        * config;
	gdl_hstruct_config_collector        * collector;
	gdl_hstruct_config_itr * itr;
	gdl_hstruct_config_buffer * buffer;
	gdl_hstruct_point_handler * handler;
	void ** hot;
	
	from    = gdl_hstruct_config_block_from (block);
	to      = gdl_hstruct_config_block_to (block);
	buffer  = gdl_hstruct_config_block_buffer (block);
	collector = gdl_hstruct_config_block_config_collector (block);
	
	itr = gdl_hstruct_config_collector_iterator (collector);
	do
	{
		config  = gdl_hstruct_config_collector_iterator_config (itr);
		hot     = gdl_hstruct_config_get_point_ptr (config, gdl_hstruct_point_hot);
		if (hot)
		{
			handler = gdl_hstruct_point_handler_alloc (gdl_hstruct_point_hot, hot);
			break;
		}
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	gdl_hstruct_config_collector_iterator_free (itr);
	
	gdl_hstruct_config_buffer_set (buffer, gdl_hstruct_point_hot, NULL);
	
	for (i = from; i <= to; i++)
	{
		gdl_hstruct_model_partition_update (p, pr, buffer, i);
	}
	
	return handler;
}

void
gdl_hstruct_model_partition_restore_hotspot_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr, gdl_hstruct_config_block * block)
{
	size_t i, j, from, to, kk;
	gdl_hstruct_config_buffer * buffer;
	
	from = gdl_hstruct_config_block_from (block);
	to   = gdl_hstruct_config_block_to (block);
	
	buffer = gdl_hstruct_config_block_buffer (block);
	
	for (i = from; i <= to; i++)
	{
		gdl_hstruct_model_partition_restore (p, pr, buffer, i);		
	}	
}

static void
gdl_hstruct_model_partition_collapse_config_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr)
{
	gdl_hstruct_config_collector * collector;
	gdl_hstruct_config_itr * citr;
	gdl_hstruct_config * c1, * c2;
	gdl_hstruct_config_block * b1, * b2;
	gdl_hstruct_config_block_itr * itr;
	
	itr = gdl_hstruct_model_partition_config_block_iterator (p, NULL);
	
	b1 = gdl_hstruct_config_block_iterator_get (itr);
	
	collector = gdl_hstruct_config_block_config_collector (b1);
	citr = gdl_hstruct_config_collector_iterator (collector);
	c1   = gdl_hstruct_config_collector_iterator_config (citr);
	gdl_hstruct_config_collector_iterator_free (citr);
	
	while (gdl_hstruct_config_block_iterator_next (itr, gdl_true))
	{
		b2 = gdl_hstruct_config_block_iterator_get (itr);
		
		collector = gdl_hstruct_config_block_config_collector (b2);
		citr = gdl_hstruct_config_collector_iterator (collector);
		c2  = gdl_hstruct_config_collector_iterator_config (citr);
		gdl_hstruct_config_collector_iterator_free (citr);
		
		if (gdl_hstruct_config_compare (c1, c2))
		{
			size_t i, from, to;
			// set config 1 on block 2
			from   = gdl_hstruct_config_block_from (b2);
			to     = gdl_hstruct_config_block_to (b2);
			for (i = from; i <= to; i++)
			{
				gdl_hstruct_model_partition_attach_config (p, pr, c1, i);
			}
		}
		else
		{
			c1 = c2;	
		}
	}	
	gdl_hstruct_config_block_iterator_free (itr);
}

static void
gdl_hstruct_model_partition_check_single_ancestral_config_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr)
{
	size_t i, from, to, k1;
	gdl_hstruct_config_buffer * buffer;
	gdl_hstruct_config_block * b1;
	gdl_hstruct_config_block_itr * itr;
	
	itr = gdl_hstruct_model_partition_config_block_iterator (p, NULL);
	
	do
	{
		b1  = gdl_hstruct_config_block_iterator_get (itr);
		k1 = gdl_hstruct_config_block_ancestral_size (b1);
		if (k1 == 1)
		{
			from   = gdl_hstruct_config_block_from (b1);
			to     = gdl_hstruct_config_block_to (b1);
			buffer = gdl_hstruct_config_block_buffer (b1);
			gdl_hstruct_config_buffer_set (buffer, gdl_hstruct_point_rho, NULL);
			gdl_hstruct_config_buffer_set (buffer, gdl_hstruct_point_hot, NULL); // in case ...
			for (i = from; i <= to; i++)
			{
				gdl_hstruct_model_partition_update (p, pr, buffer, i);
			}
			gdl_hstruct_config_block_update (pr, b1);
		}
	}
	while (gdl_hstruct_config_block_iterator_next (itr, gdl_true));
	
	gdl_hstruct_config_block_iterator_free (itr);
}

void
gdl_hstruct_model_partition_check_config_block (gdl_hstruct_model_partition * p, gdl_hstruct_parameters * pr)
{
	
	gdl_hstruct_model_partition_collapse_config_block (p, pr);
	
	gdl_hstruct_model_partition_check_single_ancestral_config_block (p, pr);

}

gdl_vector *
gdl_hstruct_model_partition_cumul_transition_proba (gdl_hstruct_model_partition * p)
{
	size_t i, j;
	double rec, tot;
	gdl_matrix * tr;
	gdl_vector * proba;
	
	proba = gdl_vector_alloc (p->nl);
	
	gdl_hstruct_hmm_store_fb_probs (p->hmm);
	
	for (i = 0; i < p->nl-1; i++)
	{
		tr = gdl_hstruct_hmm_get_transition_matrix (p->hmm, i);
		for (rec = 0, j = 0; j < GDL_MIN (tr->size1, tr->size2); j++)
		{
			rec += gdl_matrix_get (tr, j, j);
		}
		tot = gdl_matrix_sum_all (tr);
		gdl_vector_set (proba, i, 1.0-rec/tot);
		gdl_matrix_free (tr);
	}
	
	gdl_hstruct_hmm_clean_fb_probs (p->hmm);
	
	return proba;
}


gdl_gview_wrapper *
gdl_hstruct_model_partition_get_wrapper (const gdl_hstruct_model_partition * p)
{
	return p->data;	
}
//gdl_gview_wrapper *
//gdl_hstruct_model_partition_get_wrapper (const gdl_hstruct_model_partition * p)
//{
//	size_t i, j, k, l, ll, na, np, nh;
//	gdl_gvalues * x, * y;
//	gdl_gvalue * xv, * yv;
//	const gdl_mask  * mask;
//	const gdl_gview * gview;
//	gdl_gview_wrapper   * wrapper;
//	const gdl_gview_collector * missing;
//	
//	mask  = gdl_hstruct_data_get_gmask (p->pdata);
//	gview = gdl_hstruct_data_get_gview (p->pdata);
//	
//	wrapper = gdl_gview_wrapper_alloc (gdl_gview_wrapper_global, gview, mask);
//	
//	missing = gdl_hstruct_hmm_missing_collector (p->hmm);
//	
//	na = gdl_gview_wrapper_missing_accession_size (wrapper);
//	np = gdl_gview_wrapper_ploidy (wrapper);
//	
//	for (i = 0; i < na; i++)
//	{
//		for (j = 0; j < np; j++)
//		{
//			nh   = gdl_gview_wrapper_missing_hlocus_size (wrapper, i, j);
//			for (k = 0; k < nh; k++)
//			{
//				x = gdl_gview_wrapper_missing_hget (wrapper, i, j, k);
//				y = gdl_gview_collector_hget (missing, i, j, k);
//				for (l = ll = 0; l < x->size; l++)
//				{
//					xv = x->values[l];
//					yv = y->values[ll];
//					if (xv->idx == yv->idx)
//					{
//						xv->value = yv->value;
//						ll++;
//					}
//					else
//					{
//						xv->value = 0.0;	
//					}
//				}
//			}
//		}
//	}
//	
//	return wrapper;
//}

#include <gdl/gdl_hstruct_result.h>

int
gdl_hstruct_model_partition_setup (gdl_hstruct_model_partition * part, const gdl_hstruct_partition_result * pr, gdl_hstruct_parameters * params)
{
	size_t l, k, nl;
	const gdl_mask * mask;
	const gdl_hstruct_static_config * sc;
	gdl_locus * locus;
	gdl_hstruct_config * c;
	
	// set missing(part->data) == !!!
	
	gdl_hstruct_model_partition_init (part, gdl_hstruct_partition_result_ancestral_size (pr));

	// Set the locus configs...
	nl = gdl_hstruct_model_partition_size (part);
	
	for (l = 0; l < nl; l++)
	{
		locus = gdl_hstruct_model_partition_get_locus (part, l);
		
		sc    = gdl_hstruct_partition_result_config (pr, l);
		c     = gdl_hstruct_parameters_setup_config (params, sc);
		
		locus->extra = NULL;
		
		gdl_hstruct_model_partition_attach_config (part, params, c, l);
		
		for (k = 0; k < sc->k; k++)
		{
			gdl_hstruct_model_partition_set_ancestral (part, k, l, gdl_entity_get_idx (sc->ancestral[k]));	
		}
	} 
}

