/*  
 * 	lasso/model.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:08 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2008  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_mode.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_permutation.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_statistics_double.h>

#include <gdl/gdl_lasso.h>

gdl_lasso_model *
gdl_lasso_model_alloc (const size_t M)
{
	size_t i;
	gdl_lasso_model * s;
	
	s = GDL_CALLOC (gdl_lasso_model, 1);
	
	s->M = M;
	s->states    = GDL_CALLOC (unsigned char, M);
	s->actives   = GDL_CALLOC (size_t, M); 
	s->inactives = GDL_CALLOC (size_t, M);
	s->drops     = GDL_CALLOC (size_t, M);
	s->ignores   = GDL_CALLOC (size_t, M);
	s->where     = GDL_CALLOC (size_t, M);
	
	gdl_lasso_model_reset (s);
	
	return s;
} 

void
gdl_lasso_model_reset (gdl_lasso_model * s)
{
	size_t i;
	for(i = 0; i < s->M; i++)
		s->states[i] = '?';
	s->nactive   = 0;
	s->ninactive = 0;
	s->nignore   = 0;
	s->ndrop     = 0;
}

void
gdl_lasso_model_free (gdl_lasso_model * m)
{
	if (m)
	{
		GDL_FREE (m->states);
		GDL_FREE (m->actives);
		GDL_FREE (m->inactives);
		GDL_FREE (m->ignores);
		GDL_FREE (m->drops);
		GDL_FREE (m->where);
		GDL_FREE (m);
	}	
}

void
gdl_lasso_shift_predictor (gdl_lasso_model * s, const size_t p, const unsigned char which)
{
	size_t i, *n, * array = 0;
	switch(which)
	{
	 	case 'a':
	 		array = s->actives;
	 		n = &(s->nactive);
	 		break;
	 	case 'i':
	 		array = s->inactives;
	 		n = &(s->ninactive);
	 		break;
	 	case 'r':
	 		array = s->ignores;
	 		n = &(s->nignore);
	 		break;
	}
	if (array)
	{
		//printf ("SHIFT %d in %c ==> where=%d state=%c\n", p, which, s->where[p], s->states[p]);
		for(i = s->where[p]; i < (*n)-1; i++)
		{
			array[i]=array[i+1];
			(s->where[array[i]])--;
		}
		(*n)--;
	}
}

void
gdl_lasso_pop_predictor (gdl_lasso_model * s,
                         const size_t p,
                         const unsigned char which)
{
	size_t i, *n, * array = 0;
	switch(which)
	{
	 	case 'a':
	 		array = s->actives;
	 		n = &(s->nactive);
	 		break;
	 	case 'i':
	 		array = s->inactives;
	 		n = &(s->ninactive);
	 		break;
	 	case 'r':
	 		array = s->ignores;
	 		n = &(s->nignore);
	 		break;
	}
	if (array)
	{
		array[(*n)]  = p;
		s->states[p] = which;
		s->where[p]  = *n;
 		(*n)++;
	}
}

size_t
gdl_lasso_active_predictor (gdl_lasso_model * s, const size_t p)
{
	size_t i,j,add=0;
	
	switch(s->states[p])
	{
		case 'i':
			gdl_lasso_shift_predictor (s, p, 'i');
			add=1;
			break;
		case 'r':
			gdl_lasso_shift_predictor (s, p, 'r');
			add=1;
			break;
		case '?':
			add=1;
		case 'a':
		default:
			break;		
	}
	if (add)
	{
		gdl_lasso_pop_predictor (s, p, 'a');
	}
}

size_t
gdl_lasso_inactive_predictor (gdl_lasso_model * s, const size_t p)
{
	size_t i,j,add=0;
	
	switch(s->states[p])
	{
		case 'a':
			gdl_lasso_shift_predictor (s, p, 'a');
			add=1;
			break;
		case 'r':
			gdl_lasso_shift_predictor (s, p, 'r');
			add=1;
			break;
		case '?':
			add=1;
		case 'i':
		default:
			break;		
	}
	if (add)
	{
		gdl_lasso_pop_predictor (s, p, 'i');
	}
}

void
gdl_lasso_inactive_all (gdl_lasso_model * s)
{
	size_t i;
	for(i = 0; i < s->M; i++)
	{
		if (s->states[i] != 'i')
			gdl_lasso_inactive_predictor (s, i);
	}
}

size_t
gdl_lasso_drop_cache_predictor (gdl_lasso_model * s, const size_t p)
{
	if (s->states[p] == 'a')
	{
		s->drops[s->ndrop] = s->where[p];
		(s->ndrop)++;
	}
}

size_t
gdl_lasso_drop_apply_predictor (gdl_lasso_model * s)
{
	size_t i, j;
	for(i = 0; i < s->ndrop; i++)
	{
		j = s->actives[s->drops[i]];
		if (s->states[j] == 'a')
		{
			gdl_lasso_inactive_predictor (s, j);
		}
	}
	s->ndrop=0;
}

size_t
gdl_lasso_ignore_predictor (gdl_lasso_model * s, const size_t p)
{
	size_t i,j,add=0;
	
	switch(s->states[p])
	{
		case 'a':
			gdl_lasso_shift_predictor (s, p, 'a');
			add=1;
			break;	
		case 'i':
			gdl_lasso_shift_predictor (s, p, 'i');
			add=1;
			break;
		case '?':
			add=1;
		default:
			break;		
	}
	if (add)
	{
		gdl_lasso_pop_predictor (s, p, 'r');
	}
}
