/*  
 * 	linalg/gdl_linalg.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:08 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#ifndef __GDL_LINALG_H__
#define __GDL_LINALG_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_mode.h>
#include <gdl/gdl_permutation.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_linalg_svd.h>

__BEGIN_DECLS

typedef enum
  {
    GDL_LINALG_MOD_NONE = 0,
    GDL_LINALG_MOD_TRANSPOSE = 1,
    GDL_LINALG_MOD_CONJUGATE = 2
  }
gdl_linalg_matrix_mod_t;


/* Note: You can now use the gdl_blas_dgemm function instead of matmult */

/* Simple implementation of matrix multiply.
 * Calculates C = A.B
 *
 * exceptions: GDL_EBADLEN
 */
int gdl_linalg_matmult (const gdl_matrix * A,
                        const gdl_matrix * B,
                        gdl_matrix * C);


/* Simple implementation of matrix multiply.
 * Allows transposition of either matrix, so it
 * can compute A.B or Trans(A).B or A.Trans(B) or Trans(A).Trans(B)
 *
 * exceptions: GDL_EBADLEN
 */
int gdl_linalg_matmult_mod (const gdl_matrix * A,
                            gdl_linalg_matrix_mod_t modA,
                            const gdl_matrix * B,
                            gdl_linalg_matrix_mod_t modB,
                            gdl_matrix * C);

/* Calculate the matrix exponential by the scaling and
 * squaring method described in Moler + Van Loan,
 * SIAM Rev 20, 801 (1978). The mode argument allows
 * choosing an optimal strategy, from the table
 * given in the paper, for a given precision.
 *
 * exceptions: GDL_ENOTSQR, GDL_EBADLEN
 */
int gdl_linalg_exponential_ss(
  const gdl_matrix * A,
  gdl_matrix * eA,
  gdl_mode_t mode
  );


/* Householder Transformations */

double gdl_linalg_householder_transform (gdl_vector * v);

int gdl_linalg_householder_hm (double tau, 
                               const gdl_vector * v, 
                               gdl_matrix * A);

int gdl_linalg_householder_mh (double tau, 
                               const gdl_vector * v, 
                               gdl_matrix * A);

int gdl_linalg_householder_hv (double tau, 
                               const gdl_vector * v, 
                               gdl_vector * w);

int gdl_linalg_householder_hm1 (double tau, 
                                gdl_matrix * A);


/* LU Decomposition, Gaussian elimination with partial pivoting
 */

int gdl_linalg_LU_decomp (gdl_matrix * A, gdl_permutation * p, int *signum);

int gdl_linalg_LU_solve (const gdl_matrix * LU,
                         const gdl_permutation * p,
                         const gdl_vector * b,
                         gdl_vector * x);

int gdl_linalg_LU_svx (const gdl_matrix * LU,
                       const gdl_permutation * p,
                       gdl_vector * x);

int gdl_linalg_LU_refine (const gdl_matrix * A,
                          const gdl_matrix * LU,
                          const gdl_permutation * p,
                          const gdl_vector * b,
                          gdl_vector * x,
                          gdl_vector * residual);

int gdl_linalg_LU_invert (const gdl_matrix * LU,
                          const gdl_permutation * p,
                          gdl_matrix * inverse);

double gdl_linalg_LU_det (gdl_matrix * LU, int signum);
double gdl_linalg_LU_lndet (gdl_matrix * LU);
int gdl_linalg_LU_sgndet (gdl_matrix * lu, int signum);

/* Complex LU Decomposition */


/* QR decomposition */

int gdl_linalg_QR_decomp (gdl_matrix * A,
                          gdl_vector * tau);

int gdl_linalg_QR_solve (const gdl_matrix * QR,
                         const gdl_vector * tau,
                         const gdl_vector * b,
                         gdl_vector * x);

int gdl_linalg_QR_svx (const gdl_matrix * QR,
                       const gdl_vector * tau,
                       gdl_vector * x);

int gdl_linalg_QR_lssolve (const gdl_matrix * QR, 
                           const gdl_vector * tau, 
                           const gdl_vector * b, 
                           gdl_vector * x, 
                           gdl_vector * residual);


int gdl_linalg_QR_QRsolve (gdl_matrix * Q,
                           gdl_matrix * R,
                           const gdl_vector * b,
                           gdl_vector * x);

int gdl_linalg_QR_Rsolve (const gdl_matrix * QR,
                          const gdl_vector * b,
                          gdl_vector * x);

int gdl_linalg_QR_Rsvx (const gdl_matrix * QR,
                        gdl_vector * x);

int gdl_linalg_QR_update (gdl_matrix * Q,
                          gdl_matrix * R,
                          gdl_vector * w,
                          const gdl_vector * v);

int gdl_linalg_QR_QTvec (const gdl_matrix * QR,
                         const gdl_vector * tau,
                         gdl_vector * v);

int gdl_linalg_QR_Qvec (const gdl_matrix * QR,
                        const gdl_vector * tau,
                        gdl_vector * v);

int gdl_linalg_QR_unpack (const gdl_matrix * QR,
                          const gdl_vector * tau,
                          gdl_matrix * Q,
                          gdl_matrix * R);

int gdl_linalg_R_solve (const gdl_matrix * R,
                        const gdl_vector * b,
                        gdl_vector * x);

int gdl_linalg_R_svx (const gdl_matrix * R,
                      gdl_vector * x);


/* Q R P^T decomposition */

int gdl_linalg_QRPT_decomp (gdl_matrix * A,
                            gdl_vector * tau,
                            gdl_permutation * p,
                            int *signum,
                            gdl_vector * norm);

int gdl_linalg_QRPT_decomp2 (const gdl_matrix * A, 
                             gdl_matrix * q, gdl_matrix * r, 
                             gdl_vector * tau, 
                             gdl_permutation * p, 
                             int *signum,
                             gdl_vector * norm);

int gdl_linalg_QRPT_solve (const gdl_matrix * QR,
                           const gdl_vector * tau,
                           const gdl_permutation * p,
                           const gdl_vector * b,
                           gdl_vector * x);


int gdl_linalg_QRPT_svx (const gdl_matrix * QR,
                         const gdl_vector * tau,
                         const gdl_permutation * p,
                         gdl_vector * x);

int gdl_linalg_QRPT_QRsolve (const gdl_matrix * Q,
                             const gdl_matrix * R,
                             const gdl_permutation * p,
                             const gdl_vector * b,
                             gdl_vector * x);

int gdl_linalg_QRPT_Rsolve (const gdl_matrix * QR,
                             const gdl_permutation * p,
                             const gdl_vector * b,
                             gdl_vector * x);

int gdl_linalg_QRPT_Rsvx (const gdl_matrix * QR,
                           const gdl_permutation * p,
                           gdl_vector * x);

int gdl_linalg_QRPT_update (gdl_matrix * Q,
                            gdl_matrix * R,
                            const gdl_permutation * p,
                            gdl_vector * u,
                            const gdl_vector * v);

/* LQ decomposition */

int gdl_linalg_LQ_decomp (gdl_matrix * A, gdl_vector * tau);

int gdl_linalg_LQ_solve_T (const gdl_matrix * LQ, const gdl_vector * tau, 
			 const gdl_vector * b, gdl_vector * x);

int gdl_linalg_LQ_svx_T (const gdl_matrix * LQ, const gdl_vector * tau, 
                         gdl_vector * x);

int gdl_linalg_LQ_lssolve_T (const gdl_matrix * LQ, const gdl_vector * tau, 
			   const gdl_vector * b, gdl_vector * x, 
			   gdl_vector * residual);

int gdl_linalg_LQ_Lsolve_T (const gdl_matrix * LQ, const gdl_vector * b, 
			  gdl_vector * x);

int gdl_linalg_LQ_Lsvx_T (const gdl_matrix * LQ, gdl_vector * x);

int gdl_linalg_L_solve_T (const gdl_matrix * L, const gdl_vector * b, 
			gdl_vector * x);

int gdl_linalg_LQ_vecQ (const gdl_matrix * LQ, const gdl_vector * tau, 
			gdl_vector * v);

int gdl_linalg_LQ_vecQT (const gdl_matrix * LQ, const gdl_vector * tau, 
			 gdl_vector * v);

int gdl_linalg_LQ_unpack (const gdl_matrix * LQ, const gdl_vector * tau, 
			  gdl_matrix * Q, gdl_matrix * L);

int gdl_linalg_LQ_update (gdl_matrix * Q, gdl_matrix * R,
			  const gdl_vector * v, gdl_vector * w);
int gdl_linalg_LQ_LQsolve (gdl_matrix * Q, gdl_matrix * L, 
			   const gdl_vector * b, gdl_vector * x);

/* P^T L Q decomposition */

int gdl_linalg_PTLQ_decomp (gdl_matrix * A, gdl_vector * tau, 
			    gdl_permutation * p, int *signum, 
			    gdl_vector * norm);

int gdl_linalg_PTLQ_decomp2 (const gdl_matrix * A, gdl_matrix * q, 
			     gdl_matrix * r, gdl_vector * tau, 
			     gdl_permutation * p, int *signum, 
			     gdl_vector * norm);

int gdl_linalg_PTLQ_solve_T (const gdl_matrix * QR,
			   const gdl_vector * tau,
			   const gdl_permutation * p,
			   const gdl_vector * b,
			   gdl_vector * x);

int gdl_linalg_PTLQ_svx_T (const gdl_matrix * LQ,
                           const gdl_vector * tau,
                           const gdl_permutation * p,
                           gdl_vector * x);

int gdl_linalg_PTLQ_LQsolve_T (const gdl_matrix * Q, const gdl_matrix * L,
			     const gdl_permutation * p,
			     const gdl_vector * b,
			     gdl_vector * x);

int gdl_linalg_PTLQ_Lsolve_T (const gdl_matrix * LQ,
			    const gdl_permutation * p,
			    const gdl_vector * b,
			    gdl_vector * x);

int gdl_linalg_PTLQ_Lsvx_T (const gdl_matrix * LQ,
			  const gdl_permutation * p,
			  gdl_vector * x);

int gdl_linalg_PTLQ_update (gdl_matrix * Q, gdl_matrix * L,
			    const gdl_permutation * p,
			    const gdl_vector * v, gdl_vector * w);

/* Cholesky Decomposition */

int gdl_linalg_cholesky_decomp (gdl_matrix * A);

int gdl_linalg_cholesky_solve (const gdl_matrix * cholesky,
                               const gdl_vector * b,
                               gdl_vector * x);

int gdl_linalg_cholesky_svx (const gdl_matrix * cholesky,
                             gdl_vector * x);

/* Symmetric to symmetric tridiagonal decomposition */

int gdl_linalg_symmtd_decomp (gdl_matrix * A, 
                              gdl_vector * tau);

int gdl_linalg_symmtd_unpack (const gdl_matrix * A, 
                              const gdl_vector * tau,
                              gdl_matrix * Q, 
                              gdl_vector * diag, 
                              gdl_vector * subdiag);

int gdl_linalg_symmtd_unpack_T (const gdl_matrix * A,
                                gdl_vector * diag, 
                                gdl_vector * subdiag);


/* Linear Solve Using Householder Transformations

 * exceptions: 
 */

int gdl_linalg_HH_solve (gdl_matrix * A, const gdl_vector * b, gdl_vector * x);
int gdl_linalg_HH_svx (gdl_matrix * A, gdl_vector * x);

/* Linear solve for a symmetric tridiagonal system.

 * The input vectors represent the NxN matrix as follows:
 *
 *     diag[0]  offdiag[0]             0    ...
 *  offdiag[0]     diag[1]    offdiag[1]    ...
 *           0  offdiag[1]       diag[2]    ...
 *           0           0    offdiag[2]    ...
 *         ...         ...           ...    ...
 */
int gdl_linalg_solve_symm_tridiag (const gdl_vector * diag,
                                   const gdl_vector * offdiag,
                                   const gdl_vector * b,
                                   gdl_vector * x);

/* Linear solve for a nonsymmetric tridiagonal system.

 * The input vectors represent the NxN matrix as follows:
 *
 *       diag[0]  abovediag[0]              0    ...
 *  belowdiag[0]       diag[1]   abovediag[1]    ...
 *             0  belowdiag[1]        diag[2]    ...
 *             0             0   belowdiag[2]    ...
 *           ...           ...            ...    ...
 */
int gdl_linalg_solve_tridiag (const gdl_vector * diag,
                                   const gdl_vector * abovediag,
                                   const gdl_vector * belowdiag,
                                   const gdl_vector * b,
                                   gdl_vector * x);


/* Linear solve for a symmetric cyclic tridiagonal system.

 * The input vectors represent the NxN matrix as follows:
 *
 *      diag[0]  offdiag[0]             0   .....  offdiag[N-1]
 *   offdiag[0]     diag[1]    offdiag[1]   .....
 *            0  offdiag[1]       diag[2]   .....
 *            0           0    offdiag[2]   .....
 *          ...         ...
 * offdiag[N-1]         ...
 */
int gdl_linalg_solve_symm_cyc_tridiag (const gdl_vector * diag,
                                       const gdl_vector * offdiag,
                                       const gdl_vector * b,
                                       gdl_vector * x);

/* Linear solve for a nonsymmetric cyclic tridiagonal system.

 * The input vectors represent the NxN matrix as follows:
 *
 *        diag[0]  abovediag[0]             0   .....  belowdiag[N-1]
 *   belowdiag[0]       diag[1]  abovediag[1]   .....
 *              0  belowdiag[1]       diag[2]
 *              0             0  belowdiag[2]   .....
 *            ...           ...
 * abovediag[N-1]           ...
 */
int gdl_linalg_solve_cyc_tridiag (const gdl_vector * diag,
                                  const gdl_vector * abovediag,
                                  const gdl_vector * belowdiag,
                                  const gdl_vector * b,
                                  gdl_vector * x);


/* Bidiagonal decomposition */

int gdl_linalg_bidiag_decomp (gdl_matrix * A, 
                              gdl_vector * tau_U, 
                              gdl_vector * tau_V);

int gdl_linalg_bidiag_unpack (const gdl_matrix * A, 
                              const gdl_vector * tau_U, 
                              gdl_matrix * U, 
                              const gdl_vector * tau_V,
                              gdl_matrix * V,
                              gdl_vector * diag, 
                              gdl_vector * superdiag);

int gdl_linalg_bidiag_unpack2 (gdl_matrix * A, 
                               gdl_vector * tau_U, 
                               gdl_vector * tau_V,
                               gdl_matrix * V);

int gdl_linalg_bidiag_unpack_B (const gdl_matrix * A, 
                                gdl_vector * diag, 
                                gdl_vector * superdiag);

/* Varimax */
int gdl_linalg_varimax (gdl_matrix * A, gdl_matrix * T, gdl_matrix * Z, double epsilon, size_t max_iter);

/* Balancing */
int gdl_linalg_balance_columns (gdl_matrix * A, gdl_vector * D);


__END_DECLS

#endif /* GDL_LINALG */
