/* linalg/multiply.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Author:  G. Jungman */

#include <stdlib.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_linalg.h>

#define SWAP_SIZE_T(a, b)  do { size_t swap_tmp = a; a = b; b = swap_tmp; } while(0)

int
gdl_linalg_matmult (const gdl_matrix * A, const gdl_matrix * B, gdl_matrix * C)
{
  if (A->size2 != B->size1 || A->size1 != C->size1 || B->size2 != C->size2)
    {
      GDL_ERROR ("matrix sizes are not conformant", GDL_EBADLEN);
    }
  else
    {
      double a, b;
      double temp;
      size_t i, j, k;

      for (i = 0; i < C->size1; i++)
        {
          for (j = 0; j < C->size2; j++)
            {
              a = gdl_matrix_get (A, i, 0);
              b = gdl_matrix_get (B, 0, j);
              temp = a * b;
              for (k = 1; k < A->size2; k++)
                {
                  a = gdl_matrix_get (A, i, k);
                  b = gdl_matrix_get (B, k, j);
                  temp += a * b;
                }
              gdl_matrix_set (C, i, j, temp);
            }
        }

      return GDL_SUCCESS;
    }
}


int
gdl_linalg_matmult_mod (const gdl_matrix * A, gdl_linalg_matrix_mod_t modA,
                    const gdl_matrix * B, gdl_linalg_matrix_mod_t modB,
                    gdl_matrix * C)
{
  if (modA == GDL_LINALG_MOD_NONE && modB == GDL_LINALG_MOD_NONE)
    {
      return gdl_linalg_matmult (A, B, C);
    }
  else
    {
      size_t dim1_A = A->size1;
      size_t dim2_A = A->size2;
      size_t dim1_B = B->size1;
      size_t dim2_B = B->size2;
      size_t dim1_C = C->size1;
      size_t dim2_C = C->size2;

      if (modA & GDL_LINALG_MOD_TRANSPOSE)
        SWAP_SIZE_T (dim1_A, dim2_A);
      if (modB & GDL_LINALG_MOD_TRANSPOSE)
        SWAP_SIZE_T (dim1_B, dim2_B);

      if (dim2_A != dim1_B || dim1_A != dim1_C || dim2_B != dim2_C)
        {
          GDL_ERROR ("matrix sizes are not conformant", GDL_EBADLEN);
        }
      else
        {
          double a, b;
          double temp;
          size_t i, j, k;
          size_t a1, a2, b1, b2;

          for (i = 0; i < dim1_C; i++)
            {
              for (j = 0; j < dim2_C; j++)
                {
                  a1 = i;
                  a2 = 0;
                  b1 = 0;
                  b2 = j;
                  if (modA & GDL_LINALG_MOD_TRANSPOSE)
                    SWAP_SIZE_T (a1, a2);
                  if (modB & GDL_LINALG_MOD_TRANSPOSE)
                    SWAP_SIZE_T (b1, b2);

                  a = gdl_matrix_get (A, a1, a2);
                  b = gdl_matrix_get (B, b1, b2);
                  temp = a * b;

                  for (k = 1; k < dim2_A; k++)
                    {
                      a1 = i;
                      a2 = k;
                      b1 = k;
                      b2 = j;
                      if (modA & GDL_LINALG_MOD_TRANSPOSE)
                        SWAP_SIZE_T (a1, a2);
                      if (modB & GDL_LINALG_MOD_TRANSPOSE)
                        SWAP_SIZE_T (b1, b2);
                      a = gdl_matrix_get (A, a1, a2);
                      b = gdl_matrix_get (B, b1, b2);
                      temp += a * b;
                    }

                  gdl_matrix_set (C, i, j, temp);
                }
            }

          return GDL_SUCCESS;
        }
    }
}
