/* linalg/qr.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Author:  G. Jungman */

#include <stdlib.h>
#include <string.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>

#include <gdl/gdl_linalg.h>

#define REAL double

#include "givens.c"
#include "apply_givens.c"

/* Factorise a general M x N matrix A into
 *  
 *   A = Q R
 *
 * where Q is orthogonal (M x M) and R is upper triangular (M x N).
 *
 * Q is stored as a packed set of Householder transformations in the
 * strict lower triangular part of the input matrix.
 *
 * R is stored in the diagonal and upper triangle of the input matrix.
 *
 * The full matrix for Q can be obtained as the product
 *
 *       Q = Q_k .. Q_2 Q_1
 *
 * where k = MIN(M,N) and
 *
 *       Q_i = (I - tau_i * v_i * v_i')
 *
 * and where v_i is a Householder vector
 *
 *       v_i = [1, m(i+1,i), m(i+2,i), ... , m(M,i)]
 *
 * This storage scheme is the same as in LAPACK.  */

int
gdl_linalg_QR_decomp (gdl_matrix * A, gdl_vector * tau)
{
  const size_t M = A->size1;
  const size_t N = A->size2;

  if (tau->size != GDL_MIN (M, N))
    {
      GDL_ERROR ("size of tau must be MIN(M,N)", GDL_EBADLEN);
    }
  else
    {
      size_t i;

      for (i = 0; i < GDL_MIN (M, N); i++)
        {
          /* Compute the Householder transformation to reduce the j-th
             column of the matrix to a multiple of the j-th unit vector */

          gdl_vector_view c_full = gdl_matrix_column (A, i);
          gdl_vector_view c = gdl_vector_subvector (&(c_full.vector), i, M-i);

          double tau_i = gdl_linalg_householder_transform (&(c.vector));

          gdl_vector_set (tau, i, tau_i);

          /* Apply the transformation to the remaining columns and
             update the norms */

          if (i + 1 < N)
            {
              gdl_matrix_view m = gdl_matrix_submatrix (A, i, i + 1, M - i, N - (i + 1));
              gdl_linalg_householder_hm (tau_i, &(c.vector), &(m.matrix));
            }
        }

      return GDL_SUCCESS;
    }
}

/* Solves the system A x = b using the QR factorisation,

 *  R x = Q^T b
 *
 * to obtain x. Based on SLATEC code. 
 */

int
gdl_linalg_QR_solve (const gdl_matrix * QR, const gdl_vector * tau, const gdl_vector * b, gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (QR->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      /* Copy x <- b */

      gdl_vector_memcpy (x, b);

      /* Solve for x */

      gdl_linalg_QR_svx (QR, tau, x);

      return GDL_SUCCESS;
    }
}

/* Solves the system A x = b in place using the QR factorisation,

 *  R x = Q^T b
 *
 * to obtain x. Based on SLATEC code. 
 */

int
gdl_linalg_QR_svx (const gdl_matrix * QR, const gdl_vector * tau, gdl_vector * x)
{

  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != x->size)
    {
      GDL_ERROR ("matrix size must match x/rhs size", GDL_EBADLEN);
    }
  else
    {
      /* compute rhs = Q^T b */

      gdl_linalg_QR_QTvec (QR, tau, x);

      /* Solve R x = rhs, storing x in-place */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, QR, x);

      return GDL_SUCCESS;
    }
}


/* Find the least squares solution to the overdetermined system 
 *
 *   A x = b 
 *  
 * for M >= N using the QR factorization A = Q R. 
 */

int
gdl_linalg_QR_lssolve (const gdl_matrix * QR, const gdl_vector * tau, const gdl_vector * b, gdl_vector * x, gdl_vector * residual)
{
  const size_t M = QR->size1;
  const size_t N = QR->size2;

  if (M < N)
    {
      GDL_ERROR ("QR matrix must have M>=N", GDL_EBADLEN);
    }
  else if (M != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (N != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else if (M != residual->size)
    {
      GDL_ERROR ("matrix size must match residual size", GDL_EBADLEN);
    }
  else
    {
      gdl_matrix_const_view R = gdl_matrix_const_submatrix (QR, 0, 0, N, N);
      gdl_vector_view c = gdl_vector_subvector(residual, 0, N);

      gdl_vector_memcpy(residual, b);

      /* compute rhs = Q^T b */

      gdl_linalg_QR_QTvec (QR, tau, residual);

      /* Solve R x = rhs */

      gdl_vector_memcpy(x, &(c.vector));

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, &(R.matrix), x);

      /* Compute residual = b - A x = Q (Q^T b - R x) */
      
      gdl_vector_set_zero(&(c.vector));

      gdl_linalg_QR_Qvec(QR, tau, residual);

      return GDL_SUCCESS;
    }
}


int
gdl_linalg_QR_Rsolve (const gdl_matrix * QR, const gdl_vector * b, gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (QR->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match x size", GDL_EBADLEN);
    }
  else
    {
      /* Copy x <- b */

      gdl_vector_memcpy (x, b);

      /* Solve R x = b, storing x in-place */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, QR, x);

      return GDL_SUCCESS;
    }
}


int
gdl_linalg_QR_Rsvx (const gdl_matrix * QR, gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != x->size)
    {
      GDL_ERROR ("matrix size must match rhs size", GDL_EBADLEN);
    }
  else
    {
      /* Solve R x = b, storing x in-place */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, QR, x);

      return GDL_SUCCESS;
    }
}

int
gdl_linalg_R_solve (const gdl_matrix * R, const gdl_vector * b, gdl_vector * x)
{
  if (R->size1 != R->size2)
    {
      GDL_ERROR ("R matrix must be square", GDL_ENOTSQR);
    }
  else if (R->size1 != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (R->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      /* Copy x <- b */

      gdl_vector_memcpy (x, b);

      /* Solve R x = b, storing x inplace in b */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, R, x);

      return GDL_SUCCESS;
    }
}

int
gdl_linalg_R_svx (const gdl_matrix * R, gdl_vector * x)
{
  if (R->size1 != R->size2)
    {
      GDL_ERROR ("R matrix must be square", GDL_ENOTSQR);
    }
  else if (R->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      /* Solve R x = b, storing x inplace in b */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, R, x);

      return GDL_SUCCESS;
    }
}



/* Form the product Q^T v  from a QR factorized matrix 
 */

int
gdl_linalg_QR_QTvec (const gdl_matrix * QR, const gdl_vector * tau, gdl_vector * v)
{
  const size_t M = QR->size1;
  const size_t N = QR->size2;

  if (tau->size != GDL_MIN (M, N))
    {
      GDL_ERROR ("size of tau must be MIN(M,N)", GDL_EBADLEN);
    }
  else if (v->size != M)
    {
      GDL_ERROR ("vector size must be N", GDL_EBADLEN);
    }
  else
    {
      size_t i;

      /* compute Q^T v */

      for (i = 0; i < GDL_MIN (M, N); i++)
        {
          gdl_vector_const_view c = gdl_matrix_const_column (QR, i);
          gdl_vector_const_view h = gdl_vector_const_subvector (&(c.vector), i, M - i);
          gdl_vector_view w = gdl_vector_subvector (v, i, M - i);
          double ti = gdl_vector_get (tau, i);
          gdl_linalg_householder_hv (ti, &(h.vector), &(w.vector));
        }
      return GDL_SUCCESS;
    }
}


int
gdl_linalg_QR_Qvec (const gdl_matrix * QR, const gdl_vector * tau, gdl_vector * v)
{
  const size_t M = QR->size1;
  const size_t N = QR->size2;

  if (tau->size != GDL_MIN (M, N))
    {
      GDL_ERROR ("size of tau must be MIN(M,N)", GDL_EBADLEN);
    }
  else if (v->size != M)
    {
      GDL_ERROR ("vector size must be N", GDL_EBADLEN);
    }
  else
    {
      size_t i;

      /* compute Q^T v */

      for (i = GDL_MIN (M, N); i > 0 && i--;)
        {
          gdl_vector_const_view c = gdl_matrix_const_column (QR, i);
          gdl_vector_const_view h = gdl_vector_const_subvector (&(c.vector), 
                                                                i, M - i);
          gdl_vector_view w = gdl_vector_subvector (v, i, M - i);
          double ti = gdl_vector_get (tau, i);
          gdl_linalg_householder_hv (ti, &h.vector, &w.vector);
        }
      return GDL_SUCCESS;
    }
}


/*  Form the orthogonal matrix Q from the packed QR matrix */

int
gdl_linalg_QR_unpack (const gdl_matrix * QR, const gdl_vector * tau, gdl_matrix * Q, gdl_matrix * R)
{
  const size_t M = QR->size1;
  const size_t N = QR->size2;

  if (Q->size1 != M || Q->size2 != M)
    {
      GDL_ERROR ("Q matrix must be M x M", GDL_ENOTSQR);
    }
  else if (R->size1 != M || R->size2 != N)
    {
      GDL_ERROR ("R matrix must be M x N", GDL_ENOTSQR);
    }
  else if (tau->size != GDL_MIN (M, N))
    {
      GDL_ERROR ("size of tau must be MIN(M,N)", GDL_EBADLEN);
    }
  else
    {
      size_t i, j;

      /* Initialize Q to the identity */

      gdl_matrix_set_identity (Q);

      for (i = GDL_MIN (M, N); i > 0 && i--;)
        {
          gdl_vector_const_view c = gdl_matrix_const_column (QR, i);
          gdl_vector_const_view h = gdl_vector_const_subvector (&c.vector,
                                                                i, M - i);
          gdl_matrix_view m = gdl_matrix_submatrix (Q, i, i, M - i, M - i);
          double ti = gdl_vector_get (tau, i);
          gdl_linalg_householder_hm (ti, &h.vector, &m.matrix);
        }

      /*  Form the right triangular matrix R from a packed QR matrix */

      for (i = 0; i < M; i++)
        {
          for (j = 0; j < i && j < N; j++)
            gdl_matrix_set (R, i, j, 0.0);

          for (j = i; j < N; j++)
            gdl_matrix_set (R, i, j, gdl_matrix_get (QR, i, j));
        }

      return GDL_SUCCESS;
    }
}


/* Update a QR factorisation for A= Q R ,  A' = A + u v^T,

 * Q' R' = QR + u v^T
 *       = Q (R + Q^T u v^T)
 *       = Q (R + w v^T)
 *
 * where w = Q^T u.
 *
 * Algorithm from Golub and Van Loan, "Matrix Computations", Section
 * 12.5 (Updating Matrix Factorizations, Rank-One Changes)  
 */

int
gdl_linalg_QR_update (gdl_matrix * Q, gdl_matrix * R,
                      gdl_vector * w, const gdl_vector * v)
{
  const size_t M = R->size1;
  const size_t N = R->size2;

  if (Q->size1 != M || Q->size2 != M)
    {
      GDL_ERROR ("Q matrix must be M x M if R is M x N", GDL_ENOTSQR);
    }
  else if (w->size != M)
    {
      GDL_ERROR ("w must be length M if R is M x N", GDL_EBADLEN);
    }
  else if (v->size != N)
    {
      GDL_ERROR ("v must be length N if R is M x N", GDL_EBADLEN);
    }
  else
    {
      size_t j, k;
      double w0;

      /* Apply Given's rotations to reduce w to (|w|, 0, 0, ... , 0)

         J_1^T .... J_(n-1)^T w = +/- |w| e_1

         simultaneously applied to R,  H = J_1^T ... J^T_(n-1) R
         so that H is upper Hessenberg.  (12.5.2) */

      for (k = M - 1; k > 0; k--)
        {
          double c, s;
          double wk = gdl_vector_get (w, k);
          double wkm1 = gdl_vector_get (w, k - 1);

          create_givens (wkm1, wk, &c, &s);
          apply_givens_vec (w, k - 1, k, c, s);
          apply_givens_qr (M, N, Q, R, k - 1, k, c, s);
        }

      w0 = gdl_vector_get (w, 0);

      /* Add in w v^T  (Equation 12.5.3) */

      for (j = 0; j < N; j++)
        {
          double r0j = gdl_matrix_get (R, 0, j);
          double vj = gdl_vector_get (v, j);
          gdl_matrix_set (R, 0, j, r0j + w0 * vj);
        }

      /* Apply Givens transformations R' = G_(n-1)^T ... G_1^T H
         Equation 12.5.4 */

      for (k = 1; k < GDL_MIN(M,N+1); k++)
        {
          double c, s;
          double diag = gdl_matrix_get (R, k - 1, k - 1);
          double offdiag = gdl_matrix_get (R, k, k - 1);

          create_givens (diag, offdiag, &c, &s);
          apply_givens_qr (M, N, Q, R, k - 1, k, c, s);

          gdl_matrix_set (R, k, k - 1, 0.0);    /* exact zero of G^T */
        }

      return GDL_SUCCESS;
    }
}

int
gdl_linalg_QR_QRsolve (gdl_matrix * Q, gdl_matrix * R, const gdl_vector * b, gdl_vector * x)
{
  const size_t M = R->size1;
  const size_t N = R->size2;

  if (M != N)
    {
      return GDL_ENOTSQR;
    }
  else if (Q->size1 != M || b->size != M || x->size != M)
    {
      return GDL_EBADLEN;
    }
  else
    {
      /* compute sol = Q^T b */

      gdl_blas_dgemv (CblasTrans, 1.0, Q, b, 0.0, x);

      /* Solve R x = sol, storing x in-place */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, R, x);

      return GDL_SUCCESS;
    }
}
