/*  
 * 	linalg/svd.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:08 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h> 

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_linalg.h>
 
struct _gdl_linalg_svd_workspace
{
	const gdl_linalg_svd_workspace_type * type;
	size_t size1;
	size_t size2;
	size_t M;
	size_t N;
	size_t ns;
	void * state;
	gdl_matrix * U;
	gdl_matrix * V;
	gdl_vector * S;
};

gdl_linalg_svd_workspace *
gdl_linalg_svd_workspace_alloc (const gdl_linalg_svd_workspace_type * T, size_t size1, size_t size2)
{
	gdl_linalg_svd_workspace * w;
	
	w = GDL_CALLOC (gdl_linalg_svd_workspace, 1);
	
	w->type = T;
	
	w->size1 = size1;
	w->size2 = size2;
	
	w->state = gdl_malloc (T->size);
	
	(T->alloc)(w->state, size1, size2);
	
	w->U = NULL;
	w->V = gdl_matrix_alloc (size2, size2);
	w->S = gdl_vector_alloc (size2);
	
	return w;
}

void
gdl_linalg_svd_workspace_free (gdl_linalg_svd_workspace * svd)
{
	if (svd)
	{
		(svd->type->free)(svd->state);
		GDL_FREE (svd->state);
		if (svd->U)
		{
			gdl_matrix_free (svd->U);
		}
		gdl_matrix_free (svd->V);
		gdl_vector_free (svd->S);
		GDL_FREE (svd);
	}
}

int
gdl_linalg_svd_workspace_perform (gdl_linalg_svd_workspace * svd, gdl_matrix * A, size_t ns)
{
	if (A->size1 > svd->size1)
	{
		GDL_ERROR_VAL ("Number of rows exceeds the workspace size", GDL_EINVAL, GDL_EINVAL);	
	}
	if (A->size2 > svd->size2)
	{
		GDL_ERROR_VAL ("Number of columns exceeds the workspace size", GDL_EINVAL, GDL_EINVAL);	
	}
	svd->M = A->size1;
	svd->N = A->size2;
	svd->ns = ns;
	return (svd->type->perform)(svd->state, A, svd->V, svd->S, ns);
}

int
gdl_linalg_svd_workspace_const_perform (gdl_linalg_svd_workspace * svd, const gdl_matrix * A, size_t ns)
{
	if (A->size1 > svd->size1)
	{
		GDL_ERROR_VAL ("Number of rows exceeds the workspace size", GDL_EINVAL, GDL_EINVAL);	
	}
	if (A->size2 > svd->size2)
	{
		GDL_ERROR_VAL ("Number of columns exceeds the workspace size", GDL_EINVAL, GDL_EINVAL);	
	}
	if (svd->U == 0)
	{
		svd->U = gdl_matrix_alloc (svd->size1, svd->size2);
	}
	svd->M  = A->size1;
	svd->N  = A->size2;
	svd->ns = ns;
	return (svd->type->perform)(svd->state, svd->U, svd->V, svd->S, ns);
}

gdl_matrix *
gdl_linalg_svd_workspace_const_get_left  (const gdl_linalg_svd_workspace * svd)
{
	if (svd->U && svd->ns)
	{
		size_t i, j;
		gdl_matrix * U;
		
		U = gdl_matrix_alloc (svd->M, svd->ns);
		
		for (i = 0; i < svd->M; i++)
		{
			for (j = 0; j < svd->ns; j++)
			{
				gdl_matrix_set (U, i, j, gdl_matrix_get (svd->U, i, j));
			}	
		}
		
		return U;
	}
	return NULL;
}

gdl_matrix *
gdl_linalg_svd_workspace_get_right (const gdl_linalg_svd_workspace * svd)
{
	if (svd->ns)
	{
		size_t i, j;
		gdl_matrix * V;
		
		V = gdl_matrix_alloc (svd->N, svd->ns);
		
		for (i = 0; i < svd->N; i++)
		{
			for (j = 0; j < svd->ns; j++)
			{
				gdl_matrix_set (V, i, j, gdl_matrix_get (svd->V, i, j));
			}	
		}
		
		return V;
	}
	return NULL;
}

gdl_vector *
gdl_linalg_svd_workspace_get_singular (const gdl_linalg_svd_workspace * svd)
{
	if (svd->ns)
	{
		size_t i;
		gdl_vector * S;
		
		S = gdl_vector_alloc (svd->ns);
		
		for (i = 0; i < svd->ns; i++)
		{
			gdl_vector_set (S, i, gdl_vector_get (svd->S, i));
		}
		
		return S;
	}
	return NULL;
}

gdl_matrix *
gdl_linalg_svd_workspace_get_right_ptr (const gdl_linalg_svd_workspace * svd)
{
	return svd->V;	
}

gdl_vector *
gdl_linalg_svd_workspace_get_singular_ptr (const gdl_linalg_svd_workspace * svd)
{
	return svd->S;
}

#include "givens.c"
#include "svdstep.c"

int
gdl_linalg_SV_decomp (gdl_matrix * A, gdl_matrix * V, gdl_vector * S, gdl_vector * work)
{
  size_t a, b, i, j;
  
  const size_t M = A->size1;
  const size_t N = A->size2;
  const size_t K = GDL_MIN (M, N);

  if (M < N)
    {
      GDL_ERROR ("svd of MxN matrix, M<N, is not implemented", GDL_EUNIMPL);
    }
  else if (V->size1 != N)
    {
      GDL_ERROR ("square matrix V must match second dimension of matrix A",
                 GDL_EBADLEN);
    }
  else if (V->size1 != V->size2)
    {
      GDL_ERROR ("matrix V must be square", GDL_ENOTSQR);
    }
  else if (S->size != N)
    {
      GDL_ERROR ("length of vector S must match second dimension of matrix A",
                 GDL_EBADLEN);
    }
  else if (work->size != N)
    {
      GDL_ERROR ("length of workspace must match second dimension of matrix A",
                 GDL_EBADLEN);
    }

  /* Handle the case of N = 1 (SVD of a column vector) */

  if (N == 1)
    {
      gdl_vector_view column = gdl_matrix_column (A, 0);
      double norm = gdl_blas_dnrm2 (&column.vector);

      gdl_vector_set (S, 0, norm); 
      gdl_matrix_set (V, 0, 0, 1.0);
      
      if (norm != 0.0)
        {
          gdl_blas_dscal (1.0/norm, &column.vector);
        }

      return GDL_SUCCESS;
    }
  
  {
    gdl_vector_view f = gdl_vector_subvector (work, 0, K - 1);
    
    /* bidiagonalize matrix A, unpack A into U S V */
    
    gdl_linalg_bidiag_decomp (A, S, &f.vector);
    gdl_linalg_bidiag_unpack2 (A, S, &f.vector, V);
    
    /* apply reduction steps to B=(S,Sd) */
    
    chop_small_elements (S, &f.vector);
    
    /* Progressively reduce the matrix until it is diagonal */
    
    b = N - 1;
    
    while (b > 0)
      {
        double fbm1 = gdl_vector_get (&f.vector, b - 1);

        if (fbm1 == 0.0 || gdl_isnan (fbm1))
          {
            b--;
            continue;
          }
        
        /* Find the largest unreduced block (a,b) starting from b
           and working backwards */
        
        a = b - 1;
        
        while (a > 0)
          {
            double fam1 = gdl_vector_get (&f.vector, a - 1);

            if (fam1 == 0.0 || gdl_isnan (fam1))
              {
                break;
              }
            
            a--;
          }
        
        {
          const size_t n_block = b - a + 1;
          gdl_vector_view S_block = gdl_vector_subvector (S, a, n_block);
          gdl_vector_view f_block = gdl_vector_subvector (&f.vector, a, n_block - 1);
          
          gdl_matrix_view U_block =
            gdl_matrix_submatrix (A, 0, a, A->size1, n_block);
          gdl_matrix_view V_block =
            gdl_matrix_submatrix (V, 0, a, V->size1, n_block);
          
          qrstep (&S_block.vector, &f_block.vector, &U_block.matrix, &V_block.matrix);
          
          /* remove any small off-diagonal elements */
          
          chop_small_elements (&S_block.vector, &f_block.vector);
        }
      }
  }
  /* Make singular values positive by reflections if necessary */
  
  for (j = 0; j < K; j++)
    {
      double Sj = gdl_vector_get (S, j);
      
      if (Sj < 0.0)
        {
          for (i = 0; i < N; i++)
            {
              double Vij = gdl_matrix_get (V, i, j);
              gdl_matrix_set (V, i, j, -Vij);
            }
          
          gdl_vector_set (S, j, -Sj);
        }
    }
  
  /* Sort singular values into decreasing order */
  
  for (i = 0; i < K; i++)
    {
      double S_max = gdl_vector_get (S, i);
      size_t i_max = i;
      
      for (j = i + 1; j < K; j++)
        {
          double Sj = gdl_vector_get (S, j);
          
          if (Sj > S_max)
            {
              S_max = Sj;
              i_max = j;
            }
        }
      
      if (i_max != i)
        {
          /* swap eigenvalues */
          gdl_vector_swap_elements (S, i, i_max);
          
          /* swap eigenvectors */
          gdl_matrix_swap_columns (A, i, i_max);
          gdl_matrix_swap_columns (V, i, i_max);
        }
    }
  
  return GDL_SUCCESS;
}

int
gdl_linalg_SV_decomp_mod (gdl_matrix * A, gdl_matrix * X, gdl_matrix * V, gdl_vector * S, gdl_vector * work)
{
  size_t i, j;

  const size_t M = A->size1;
  const size_t N = A->size2;
  
  if (M < N)
    {
      GDL_ERROR ("svd of MxN matrix, M<N, is not implemented", GDL_EUNIMPL);
    }
  else if (V->size1 != N)
    {
      GDL_ERROR ("square matrix V must match second dimension of matrix A",
                 GDL_EBADLEN);
    }
  else if (V->size1 != V->size2)
    {
      GDL_ERROR ("matrix V must be square", GDL_ENOTSQR);
    }
  else if (X->size1 != N)
    {
      GDL_ERROR ("square matrix X must match second dimension of matrix A",
                 GDL_EBADLEN);
    }
  else if (X->size1 != X->size2)
    {
      GDL_ERROR ("matrix X must be square", GDL_ENOTSQR);
    }
  else if (S->size != N)
    {
      GDL_ERROR ("length of vector S must match second dimension of matrix A",
                 GDL_EBADLEN);
    }
  else if (work->size != N)
    {
      GDL_ERROR ("length of workspace must match second dimension of matrix A",
                 GDL_EBADLEN);
    }

  if (N == 1)
    {
      gdl_vector_view column = gdl_matrix_column (A, 0);
      double norm = gdl_blas_dnrm2 (&column.vector);

      gdl_vector_set (S, 0, norm); 
      gdl_matrix_set (V, 0, 0, 1.0);
      
      if (norm != 0.0)
        {
          gdl_blas_dscal (1.0/norm, &column.vector);
        }

      return GDL_SUCCESS;
    }

  /* Convert A into an upper triangular matrix R */

  for (i = 0; i < N; i++)
    {
      gdl_vector_view c = gdl_matrix_column (A, i);
      gdl_vector_view v = gdl_vector_subvector (&c.vector, i, M - i);
      double tau_i = gdl_linalg_householder_transform (&v.vector);

      /* Apply the transformation to the remaining columns */

      if (i + 1 < N)
        {
          gdl_matrix_view m =
            gdl_matrix_submatrix (A, i, i + 1, M - i, N - (i + 1));
          gdl_linalg_householder_hm (tau_i, &v.vector, &m.matrix);
        }

      gdl_vector_set (S, i, tau_i);
    }

  /* Copy the upper triangular part of A into X */

  for (i = 0; i < N; i++)
    {
      for (j = 0; j < i; j++)
        {
          gdl_matrix_set (X, i, j, 0.0);
        }

      {
        double Aii = gdl_matrix_get (A, i, i);
        gdl_matrix_set (X, i, i, Aii);
      }

      for (j = i + 1; j < N; j++)
        {
          double Aij = gdl_matrix_get (A, i, j);
          gdl_matrix_set (X, i, j, Aij);
        }
    }

  /* Convert A into an orthogonal matrix L */

  for (j = N; j > 0 && j--;)
    {
      /* Householder column transformation to accumulate L */
      double tj = gdl_vector_get (S, j);
      gdl_matrix_view m = gdl_matrix_submatrix (A, j, j, M - j, N - j);
      gdl_linalg_householder_hm1 (tj, &m.matrix);
    }

  /* unpack R into X V S */

  gdl_linalg_SV_decomp (X, V, S, work);

  /* Multiply L by X, to obtain U = L X, stored in U */

  {
    gdl_vector_view sum = gdl_vector_subvector (work, 0, N);

    for (i = 0; i < M; i++)
      {
        gdl_vector_view L_i = gdl_matrix_row (A, i);
        gdl_vector_set_zero (&sum.vector);

        for (j = 0; j < N; j++)
          {
            double Lij = gdl_vector_get (&L_i.vector, j);
            gdl_vector_view X_j = gdl_matrix_row (X, j);
            gdl_blas_daxpy (Lij, &X_j.vector, &sum.vector);
          }

        gdl_vector_memcpy (&L_i.vector, &sum.vector);
      }
  }

  return GDL_SUCCESS;
}

int
gdl_linalg_SV_ginv (gdl_matrix * A, gdl_matrix * V, gdl_vector * S, gdl_vector * work, gdl_matrix * gA)
{
	size_t i, j, k;
	double t;
	
	gdl_linalg_SV_decomp (A, V, S, work);
	
	for(i = 0; i < S->size; i++)
	{
		t = gdl_vector_get (S, i);
		if (t != 0)
		{
			gdl_vector_set (S, i, 1.0/t);
		}
	}
	for (i = 0; i < A->size2; i++)
	{
		for (j = 0; j < A->size1; j++)
		{
			t = 0;
			for (k = 0; k < A->size2; k++)
			{
				t += gdl_matrix_get(V, i, k)*gdl_vector_get(S, k)*gdl_matrix_get(A, j, k);
			}
			gdl_matrix_set (gA, i, j, t);
		}
	}	
}

int gdl_linalg_SV_ginv_mod (gdl_matrix * A, gdl_matrix * X, gdl_matrix * V, gdl_vector * S, gdl_vector * work);

