/*  
 * 	list/util.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:45 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_list.h>

struct _gdl_link {
  size_t       owner;
  struct _gdl_link * next;
  struct _gdl_link * prev;
  void        * data;
};

struct _gdl_list
{
	size_t            size;
	struct _gdl_link  * state;
    const  gdl_data_interface     * type;
    gdl_data_compare  compare;
};

static void
gdl_default_link_free (void * data)
{
	GDL_FREE (data);
}

static int
gdl_default_link_compare (const void * data0, const void * data1)
{
    if (data0 < data1)
        return (-1);
    else if (data0 == data1)
		return (0);
    return (1);
}

static void
gdl_link_free (gdl_list * l, gdl_link * lk)
{
    (lk->prev)->next = lk->next;
    (lk->next)->prev = lk->prev;
    if(lk->owner == 1 && l->type->free != 0)
        (l->type->free)(lk->data);
    GDL_FREE (lk);
}

static gdl_link *
gdl_list_lower_search (gdl_list * l, void * data) 
{
    gdl_link * lk;
    
	if (l == NULL)
        return(NULL);
    
    for(lk = l->state->next;lk != l->state && (l->compare)(lk->data, data) < 0 ;lk = lk->next);
    return lk;
}

static gdl_link * 
gdl_list_higher_search (gdl_list * l, void * data) 
{
    gdl_link * lk;
    
    if (l == NULL)
        return(NULL);
       
    for(lk = l->state->prev;lk != l->state && (l->compare)(lk->data, data) >0 ;lk = lk->prev);
    return lk;    
}

static gdl_link *
gdl_list_link_search (gdl_list * l, void * data) 
{
    gdl_link * lk;
    
    if (l == NULL)
        return(NULL);
    
    lk = gdl_list_lower_search (l, data);
    
    if (lk == l->state)
        return NULL;
    else {
        if ((l->compare)(lk->data, data) ==0)
            return lk;
        return NULL;
    }
}

static gdl_link *
gdl_list_link_rsearch (gdl_list * l, void * data) 
{
    gdl_link * lk;
    
    if (l == NULL)
        return(NULL);
    	
    lk = gdl_list_higher_search (l, data);
    
    if (lk == l->state)
        return NULL;
    else {
        if ((l->compare)(lk->data, data) ==0)
            return lk;
        return NULL;
    }
}

static void
gdl_list_swap (gdl_list * x, gdl_link * f, gdl_link * l)
{
	gdl_link * lkp, * lkn;
	
	if (x == NULL || ( f == NULL || l == NULL ))
		return;
		
	lkp = f->prev;
	lkn = f->next;
	f->next = l->next;
	f->prev = l->prev;
	l->next = lkn;
	l->prev = lkp;
	
	return;
}

static gdl_link *
gdl_list_partition(gdl_list * x, gdl_link * f, gdl_link * l)
{
	gdl_link * up, * down, * temp, * piv;
	
    up   = piv = f;
    down = l;
    
    do
    { 
        while ((x->compare)(up->data, piv->data) <= 0 
               && up != l)
        {
            up = up->next;
        }
        while ((x->compare)(down->data, piv->data) > 0
               && down != f )
        {
            down = down->prev;
        }
        if (up != down) {
        	
            gdl_list_swap (x, up, down);
            
        }
    } while (down != up);
    
    // Move pivot to its final place
    gdl_list_swap (x, piv, down);
    
    return down;
}

static void
gdl_list_quicksort(gdl_list * l, gdl_link * first, gdl_link * last) {
    gdl_link * pivot = NULL;
    if(first != last)
    {
        pivot = gdl_list_partition (l, first, last);
        gdl_list_quicksort (l, first, pivot->prev);
        gdl_list_quicksort (l, pivot->next, last);
    }
}

static gdl_data_interface _gdl_list_type_default =
{
	&gdl_default_link_free,
	NULL,
	&gdl_default_link_compare,
	NULL,
	NULL
};
