/*  
 *  locus/registry.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_locus_type.h>

typedef struct 
{
	gdl_locus_type  * type;
	gdl_hashtable   * loci;
} gdl_locus_type_entry;

static gdl_locus_type_entry *
gdl_locus_type_entry_alloc ()
{
	gdl_locus_type_entry * e;
	
	e = GDL_MALLOC (gdl_locus_type_entry, 1);
	
	e->type = NULL;
	e->loci = gdl_hashtable_alloc (gdl_entity_interface, 0);
	
	return e;
}

static gdl_locus_type_entry *
gdl_locus_type_entry_clone (const gdl_locus_type_entry * e)
{
	if (e)
	{
		gdl_locus_type_entry * c;
		
		c = gdl_locus_type_entry_alloc ();
		
		c->type = gdl_locus_type_clone (e->type);
		c->loci = gdl_hashtable_clone (e->loci);
		
		return c;
	}
	
	return NULL;
}

static int
gdl_locus_type_entry_merge (gdl_locus_type_entry * dest, const gdl_locus_type_entry * src)
{
	if (dest && src)
	{
		if (!gdl_locus_type_compare (dest->type, src->type))
		{
			if (gdl_hashtable_size (src->loci))
			{
				gdl_hashtable_itr * itr;
				
				itr = gdl_hashtable_iterator (src->loci);
				do
				{
					gdl_locus * locus = (gdl_locus *) gdl_hashtable_iterator_value (itr);
					gdl_hashtable_update (dest->loci, gdl_entity_get_name (locus), gdl_entity_clone (locus), 1);
				}
				while (gdl_hashtable_iterator_next (itr));
				gdl_hashtable_iterator_free (itr);
			}
			
			return GDL_SUCCESS;
		}			
	}
	
	return GDL_EINVAL;
}

static int
gdl_locus_type_entry_fprintf (FILE * stream, const gdl_locus_type_entry * e)
{
	if (stream && e)
	{
		if (gdl_hashtable_size (e->loci))
		{
			gdl_hashtable_itr * itr;
			
			itr = gdl_hashtable_iterator (e->loci);
			do
			{
				gdl_locus * locus = (gdl_locus *) gdl_hashtable_iterator_value (itr);
				fprintf (stream, "%s\t%s\n", e->type->name, gdl_entity_get_name (locus)); 
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
		}
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

static void
_gdl_locus_type_entry_free (void * ve)
{
	if (ve)
	{
		gdl_locus_type_entry * e = (gdl_locus_type_entry *) ve;
		gdl_locus_type_free (e->type);
		gdl_hashtable_free (e->loci);
		GDL_FREE (e);
	}
}

void *
_gdl_locus_type_entry_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_locus_type_entry * e;
		
		e = gdl_locus_type_entry_alloc ();
		
		e->type = gdl_locus_type_fread (stream);
		GDL_FREAD_STATUS (e->type!=0, 1);
		status  = gdl_hashtable_fread (stream, e->loci);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return e;
	}
	
	return NULL;	
}

int
_gdl_locus_type_entry_fwrite (FILE * stream, const void * ve)
{
	if (stream)
	{
		int status;
		gdl_locus_type_entry * e = (gdl_locus_type_entry *) ve;
		
		status = gdl_locus_type_fwrite (stream, e->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status  = gdl_hashtable_fwrite (stream, e->loci);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

static const gdl_data_interface _gdl_locus_type_entry_interface =
{
	_gdl_locus_type_entry_free,
	NULL,
	NULL,
	_gdl_locus_type_entry_fread,
	_gdl_locus_type_entry_fwrite,
}; 

struct _gdl_locus_type_registry
{
	gdl_hashtable * table;	
};

gdl_locus_type_registry *
gdl_locus_type_registry_alloc (void)
{
	gdl_locus_type_registry * r;
	
	r = GDL_MALLOC (gdl_locus_type_registry, 1);
	
	r->table = gdl_hashtable_alloc (&_gdl_locus_type_entry_interface, 0);
	
	return r;	
}

void
gdl_locus_type_registry_free (gdl_locus_type_registry * t)
{
	if (t)
	{
		gdl_hashtable_free (t->table);
		GDL_FREE (t);	
	}	
}

size_t
gdl_locus_type_registry_size (const gdl_locus_type_registry * r)
{
	return gdl_hashtable_size (r->table);	
}

const gdl_locus_type *
gdl_locus_type_registry_get (const gdl_locus_type_registry * r, size_t i)
{
	gdl_locus_type * type = NULL;
	
	if (gdl_hashtable_size (r->table))
	{
		size_t j=0;
		gdl_hashtable_itr * itr;
		
		itr = gdl_hashtable_iterator (r->table);
		
		do
		{
			gdl_locus_type_entry * e = (gdl_locus_type_entry *) gdl_hashtable_iterator_value (itr);
			type = e->type;
			j++;
		}
		while (gdl_hashtable_iterator_next (itr) && j < i);
		
		gdl_hashtable_iterator_free (itr);
		
		if (j < i)
		{
			type = NULL;	
		}
	}
	
	return type;
}

int
gdl_locus_type_registry_add (gdl_locus_type_registry * r, const gdl_locus_type * t, const gdl_locus * l)
{
	if (r && t && l)
	{
		gdl_locus_type_entry * e;
		
		e = (gdl_locus_type_entry *) gdl_hashtable_lookup (r->table, t->name);
		
		if (!e)
		{
			e = gdl_locus_type_entry_alloc ();
			e->type = gdl_locus_type_clone (t);
			gdl_hashtable_add (r->table, t->name, e, 1);
		}
		
		return gdl_hashtable_add (e->loci, gdl_entity_get_name (l), gdl_entity_clone (l), 1);
	}
	
	return -1;
}

int
gdl_locus_type_registry_update (gdl_locus_type_registry * r, const gdl_locus_type * t, const gdl_locus * l)
{
	if (r && t && l)
	{
		gdl_locus_type_entry * e;
		
		e = (gdl_locus_type_entry *) gdl_hashtable_lookup (r->table, t->name);
		
		if (!e)
		{
			e = gdl_locus_type_entry_alloc ();
			e->type = gdl_locus_type_clone (t);
			gdl_hashtable_add (r->table, t->name, e, 1);
		}
		
		return gdl_hashtable_update (e->loci, gdl_entity_get_name (l), gdl_entity_clone (l), 1);
	}
	
	return -1;
}

const gdl_locus_type *
gdl_locus_type_registry_search (const gdl_locus_type_registry * r, const gdl_locus * locus)
{
	if (gdl_hashtable_size (r->table))
	{
		gdl_hashtable_itr * itr;
		
		itr = gdl_hashtable_iterator (r->table);
		do
		{
			gdl_locus_type_entry * e = (gdl_locus_type_entry *) gdl_hashtable_iterator_value (itr);
			if (gdl_hashtable_lookup (e->loci, gdl_entity_get_name (locus)))
			{
				gdl_hashtable_iterator_free (itr);
				return e->type;
			}
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
		
		return NULL;
	}
	
	return NULL;	
}

int
gdl_locus_type_registry_merge (gdl_locus_type_registry * dest, const gdl_locus_type_registry * src)
{
	if (dest && src)
	{
		if (gdl_hashtable_size (src->table))
		{
			gdl_hashtable_itr * itr;
			gdl_locus_type_entry * e1, * e2;
			
			itr = gdl_hashtable_iterator (src->table);
			do
			{
				e2 = (gdl_locus_type_entry *) gdl_hashtable_iterator_value (itr);
				e1 = (gdl_locus_type_entry *) gdl_hashtable_lookup (dest->table, e2->type->name);
				if (e1)
				{
					gdl_locus_type_entry_merge (e1, e2);	
				}
				else
				{
					e1 = gdl_locus_type_entry_clone (e2);
					gdl_hashtable_add (dest->table, e1->type->name, e1, 1);
				}
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
		}
	}
}

gdl_locus_type_registry *
gdl_locus_type_registry_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_locus_type_registry * r;
		
		r = gdl_locus_type_registry_alloc ();
		
		status = gdl_hashtable_fread (stream, r->table);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return r;
	}
	
	return NULL;
}

int
gdl_locus_type_registry_fwrite (FILE * stream, const gdl_locus_type_registry * r)
{
	if (stream && r)
	{
		int status;
		
		status = gdl_hashtable_fwrite (stream, r->table);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

gdl_locus_type_registry *
gdl_locus_type_registry_fscanf (FILE * stream)
{
	if (stream)
	{
		size_t i, n;
		gdl_string * tmp, * line = NULL;
		gdl_locus_type * type;
		gdl_locus      * locus;
		gdl_locus_type_registry * r;
		
		r = gdl_locus_type_registry_alloc ();
		
		// ignore the header
		gdl_getline (&line, &n, stream);
		gdl_string_free (line);line=NULL;
		
		while (gdl_getline (&line, &n, stream)!=-1)
		{
			if (!strcmp (line, "--"))
			{
				gdl_string_free (line);
				break;	
			}
			for (i = 0; i < n && !isspace (line[i]); i++);
			if (i)
			{
				size_t j;
				
				tmp  = gdl_string_alloc (i);
				strncpy (tmp, line, i);
				type = gdl_locus_type_alloc (tmp);
				gdl_string_free (tmp);
				for (; i < n && isspace (line[i]); i++);
				for (j = i; j < n && !isspace (line[j]); j++);
				if (j)
				{
					tmp  = gdl_string_alloc (j-i+1);
					strncpy (tmp, &line[i], j-i+1);
					locus = gdl_locus_new (tmp);
					gdl_locus_type_registry_add (r, type, locus);
					gdl_string_free (tmp);
					gdl_locus_type_free (type);
					gdl_locus_free (locus);
				}
				else
				{
					gdl_locus_type_registry_free (r);
					gdl_string_free (line);
					GDL_ERROR_VAL (gdl_string_sprintf ("Invalid locus name for line [ %s ]", line), GDL_FAILURE, 0);
				}
			}
			else
			{
				gdl_locus_type_registry_free (r);
				gdl_string_free (line);
				GDL_ERROR_VAL (gdl_string_sprintf ("Invalid locus type name for line [ %s ]", line), GDL_FAILURE, 0);
			}
			gdl_string_free (line);line=NULL;
		}
		
		return r;
	}
	
	return NULL;
}

int
gdl_locus_type_registry_fprintf (FILE * stream, const gdl_locus_type_registry * r)
{
	if (stream && r)
	{
		if (gdl_hashtable_size (r->table))
		{
			gdl_hashtable_itr * itr;
			
			fprintf (stream, "type\tlocus\n");
			
			itr = gdl_hashtable_iterator (r->table);
			do
			{
				gdl_locus_type_entry * e = (gdl_locus_type_entry *) gdl_hashtable_iterator_value (itr);
				gdl_locus_type_entry_fprintf (stream, e);
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
			
			fprintf (stream, "--\n");
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

struct _gdl_locus_type_registry_itr
{
	const gdl_locus_type_entry * e;
	gdl_hashtable_itr * itr;
};

gdl_locus_type_registry_itr *
gdl_locus_type_registry_iterator (const gdl_locus_type_registry * r, const gdl_locus_type * type)
{
	gdl_locus_type_registry_itr * i;
	
	i = GDL_MALLOC (gdl_locus_type_registry_itr, 1);
	
	i->e = (gdl_locus_type_entry *) gdl_hashtable_lookup (r->table, type->name);
	
	if (!(i->e))
	{
		GDL_FREE (i);
		return NULL;	
	}
	
	i->itr = gdl_hashtable_iterator (i->e->loci);
	
	return i;
}

const gdl_locus *
gdl_locus_type_registry_iterator_get (gdl_locus_type_registry_itr * itr)
{
	return gdl_hashtable_iterator_value (itr->itr);
}

gdl_boolean
gdl_locus_type_registry_iterator_next (gdl_locus_type_registry_itr * itr)
{
	return gdl_hashtable_iterator_next (itr->itr);	
}

void
gdl_locus_type_registry_iterator_free (gdl_locus_type_registry_itr * itr)
{
	if (itr)
	{
		gdl_hashtable_iterator_free (itr->itr);
		GDL_FREE (itr);
	}	
}
