/*  
 *  mask/mask.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>

gdl_mask * 
gdl_mask_alloc ()
{
	gdl_mask * m = GDL_CALLOC (gdl_mask, 1);
	
	m->table = gdl_hashtable_alloc (gdl_entity_mask_interface, 2);
	
	return m;
}

gdl_mask *
gdl_mask_clone (const gdl_mask * m)
{
	if (m)
	{
		gdl_mask * c;
		
		c = gdl_mask_alloc ();
		
		c->table = gdl_hashtable_clone (m->table);
		
		return c;
	}
	return NULL;
}

void
gdl_mask_free (gdl_mask * m)
{
	if (m)
	{
		gdl_hashtable_free (m->table);
		GDL_FREE (m);
	}
}

void
gdl_mask_set (gdl_mask * m, const gdl_entity_type * T, gdl_entity_mask * em, size_t owner)
{
	if ((m && em) && T)
	{
		gdl_hashtable_update (m->table, T->name, em, owner);
	}
}

int
gdl_mask_add (gdl_mask * m, const gdl_entity * e)
{
	if (m && e)
	{
		gdl_entity_mask * em;
		const gdl_entity_type * T;
		
		T = gdl_entity_get_type (e);
		
		em = (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
		
		if (!em)
		{
			em = gdl_entity_mask_alloc ();
			gdl_hashtable_add (m->table, T->name, em, 1);
		}
		
		return gdl_entity_mask_add (em, e);
	}
}

int
gdl_mask_add_idx (gdl_mask * m, const gdl_entity_type * T, size_t idx)
{
	if (m && T)
	{
		gdl_entity_mask * em;
		
		em = (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
		
		if (!em)
		{
			em = gdl_entity_mask_alloc ();
			gdl_hashtable_add (m->table, T->name, em, 1);
		}
		
		return gdl_entity_mask_add_idx (em, idx);
	}	
}

const gdl_entity_mask *
gdl_mask_get (const gdl_mask * m, const gdl_entity_type * T)
{
	if (m && T)
	{
		return (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
	}
	
	return NULL;
}

gdl_entity_mask *
gdl_mask_get_clone (const gdl_mask * m, const gdl_entity_type * T)
{
	if (m && T)
	{
		return gdl_entity_mask_clone ((gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name));
	}
	
	return NULL;
}

size_t
gdl_mask_get_size (const gdl_mask * m, const gdl_entity_type * T)
{
	const gdl_entity_mask * em = gdl_mask_get (m, T);
	if (em)
	{
		return gdl_entity_mask_size (em);
	}
	return 0;
}

size_t
gdl_mask_get_idx (const gdl_mask * m, const gdl_entity_type * T, size_t i)
{
	const gdl_entity_mask * em = gdl_mask_get (m, T);
	if (em)
	{
		return gdl_entity_mask_idx (em, i);
	}
	return i;
}

void
gdl_mask_remove (const gdl_mask * m, const gdl_entity_type * T, size_t i)
{
	if (m && T)
	{
		gdl_entity_mask * em = (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
		gdl_entity_mask_remove (em, i);
	}
}

gdl_mask *
gdl_mask_clustering (const gdl_mask * m, const gdl_entity_type * T, const gdl_clustering * clustering)
{
	if (m)
	{
		size_t ic, idx, nc;
		const gdl_entity_mask * em;
		gdl_mask * cm;
		
		em = (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
		
		cm = gdl_mask_alloc ();
		
		nc = (em==0) ? 1 : 0;
		
		if (gdl_hashtable_size (m->table) > nc)
		{
			gdl_hashtable_itr * itr = gdl_hashtable_iterator (m->table);
			do
			{			
				const gdl_string * name = (gdl_string *) gdl_hashtable_iterator_key (itr);
				if (strcmp (name, T->name))
				{
					em = (gdl_entity_mask *) gdl_hashtable_iterator_value (itr);
					gdl_hashtable_add (cm->table, name, gdl_entity_mask_clone (em), 1);
				}
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
		}
		
		nc = gdl_clustering_nclust (clustering);
		for (ic = 0; ic < nc; ic++)
		{
			if (em)
			{
				idx = gdl_entity_mask_idx (em, gdl_clustering_clust_idx (clustering, ic));
			}
			else
			{
				idx = gdl_clustering_clust_idx (clustering, ic);
			}
			gdl_mask_add_idx (cm, T, idx);
		}
		
		return cm;
	}
	
	return NULL;
}

size_t
gdl_mask_size (const gdl_mask * m, const gdl_entity_type * T)
{
	if (m && T) 
	{
		gdl_entity_mask * em = (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
		if (em)
		{
			return gdl_entity_mask_size (em);
		}
		return 0;
	}	
}

void
gdl_mask_reset (gdl_mask * m, const gdl_entity_type * T)
{
	if (m && T) 
	{
		gdl_entity_mask * em = (gdl_entity_mask *) gdl_hashtable_lookup (m->table, T->name);
		if (em)
		{
			gdl_hashtable_update (m->table, T->name, gdl_entity_mask_alloc (), 1);
		}
	}
}

void
gdl_mask_set_identity (gdl_mask * m, const gdl_entity_type * T, size_t n)
{
	if (m && T)
	{
		gdl_entity_mask * em = gdl_entity_mask_ident (n);
		gdl_hashtable_update (m->table, T->name, em, 1);
	}	
}

gdl_mask *
gdl_mask_ligation (const gdl_mask * m1, 
                   const gdl_mask * m2,
                   const gdl_entity_type * ENTITY_LIGATION,
                   const gdl_entity_type * ENTITY_OVER,
                   gdl_entity_mask ** ma1,
                   gdl_entity_mask ** ma2)
{
    size_t nl;
	gdl_mask * m;
	gdl_entity_mask * em1;
	const gdl_entity_mask * em2;
	
	m = gdl_mask_alloc ();
	
	em1 = gdl_entity_mask_clone (gdl_mask_get (m1, ENTITY_LIGATION));
	
	if (em1 == 0)
	{
		GDL_ERROR_VAL ("Cannot ligate an empty mask",
		               GDL_FAILURE,
		               0);
	}
	
	nl  = gdl_entity_mask_size (em1);
	
	em2 = gdl_mask_get (m2, ENTITY_LIGATION);
	
	if (em2 == 0)
	{
		GDL_ERROR_VAL ("Cannot ligate an empty mask",
		               GDL_FAILURE,
		               0);
	}
	
	nl += gdl_entity_mask_size (em2);
	
	gdl_entity_mask_union (em1, em2);
	
	if (gdl_entity_mask_size (em1) != nl)
	{
		GDL_ERROR_VAL ("Cannot ligate entity masks when there are entities in common",
		               GDL_FAILURE,
		               0);
	}
	
	gdl_mask_set (m, GDL_LOCUS, em1, 1);
	
	em1 = gdl_mask_get_clone (m1, ENTITY_OVER);
	
	em2 = gdl_mask_get (m2, ENTITY_OVER);
	
	(*ma1) = (*ma2) = NULL;
	
	if (em1 == 0)
	{
		if (em2)
		{
			*ma1 = gdl_entity_mask_clone (em2);
			*ma2 = gdl_entity_mask_ident (em2->size);
			gdl_mask_set (m, ENTITY_OVER, gdl_entity_mask_clone(em2), 1);
		}
		return m;
	}
	else if (em2 == 0)
	{
		*ma1 = gdl_entity_mask_ident (em1->size);
		*ma2 = gdl_entity_mask_clone (em1);
		gdl_mask_set (m, ENTITY_OVER, em1, 1);
		return m;
	}
	else
	{
		*ma1 = gdl_entity_mask_alloc ();
		*ma2 = gdl_entity_mask_alloc ();
		gdl_entity_mask_inter_over (em1, em2, *ma1, *ma2);
		gdl_mask_set (m, ENTITY_OVER, em1, 1);
		return m;
	}

	return m;
}

int
gdl_mask_fprintf (FILE * stream, const gdl_mask * g)
{
	return GDL_SUCCESS;	
}

