/* matrix/gdl_matrix_long.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_MATRIX_LONG_H__
#define __GDL_MATRIX_LONG_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_vector.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size1;
  size_t size2;
  size_t tda;
  long * data;
  gdl_block_long * block;
  int owner;
} gdl_matrix_long;

typedef struct
{
  gdl_matrix_long matrix;
} _gdl_matrix_long_view;

typedef _gdl_matrix_long_view gdl_matrix_long_view;

typedef struct
{
  gdl_matrix_long matrix;
} _gdl_matrix_long_const_view;

typedef const _gdl_matrix_long_const_view gdl_matrix_long_const_view;

/* Allocation */

gdl_matrix_long * 
gdl_matrix_long_alloc (const size_t n1, const size_t n2);

gdl_matrix_long * 
gdl_matrix_long_calloc (const size_t n1, const size_t n2);

gdl_matrix_long * 
gdl_matrix_long_alloc_from_block (gdl_block_long * b, 
                                   const size_t offset, 
                                   const size_t n1, 
                                   const size_t n2, 
                                   const size_t d2);

gdl_matrix_long * 
gdl_matrix_long_alloc_from_matrix (gdl_matrix_long * m,
                                    const size_t k1, 
                                    const size_t k2,
                                    const size_t n1, 
                                    const size_t n2);

gdl_vector_long * 
gdl_vector_long_alloc_row_from_matrix (gdl_matrix_long * m,
                                        const size_t i);

gdl_vector_long * 
gdl_vector_long_alloc_col_from_matrix (gdl_matrix_long * m,
                                        const size_t j);

void gdl_matrix_long_free (gdl_matrix_long * m);

/* Views */

_gdl_matrix_long_view 
gdl_matrix_long_submatrix (gdl_matrix_long * m, 
                            const size_t i, const size_t j, 
                            const size_t n1, const size_t n2);

_gdl_vector_long_view 
gdl_matrix_long_row (gdl_matrix_long * m, const size_t i);

_gdl_vector_long_view 
gdl_matrix_long_column (gdl_matrix_long * m, const size_t j);

_gdl_vector_long_view 
gdl_matrix_long_diagonal (gdl_matrix_long * m);

_gdl_vector_long_view 
gdl_matrix_long_subdiagonal (gdl_matrix_long * m, const size_t k);

_gdl_vector_long_view 
gdl_matrix_long_superdiagonal (gdl_matrix_long * m, const size_t k);

_gdl_matrix_long_view
gdl_matrix_long_view_array (long * base,
                             const size_t n1, 
                             const size_t n2);

_gdl_matrix_long_view
gdl_matrix_long_view_array_with_tda (long * base, 
                                      const size_t n1, 
                                      const size_t n2,
                                      const size_t tda);


_gdl_matrix_long_view
gdl_matrix_long_view_vector (gdl_vector_long * v,
                              const size_t n1, 
                              const size_t n2);

_gdl_matrix_long_view
gdl_matrix_long_view_vector_with_tda (gdl_vector_long * v,
                                       const size_t n1, 
                                       const size_t n2,
                                       const size_t tda);


_gdl_matrix_long_const_view 
gdl_matrix_long_const_submatrix (const gdl_matrix_long * m, 
                                  const size_t i, const size_t j, 
                                  const size_t n1, const size_t n2);

_gdl_vector_long_const_view 
gdl_matrix_long_const_row (const gdl_matrix_long * m, 
                            const size_t i);

_gdl_vector_long_const_view 
gdl_matrix_long_const_column (const gdl_matrix_long * m, 
                               const size_t j);

_gdl_vector_long_const_view
gdl_matrix_long_const_diagonal (const gdl_matrix_long * m);

_gdl_vector_long_const_view 
gdl_matrix_long_const_subdiagonal (const gdl_matrix_long * m, 
                                    const size_t k);

_gdl_vector_long_const_view 
gdl_matrix_long_const_superdiagonal (const gdl_matrix_long * m, 
                                      const size_t k);

_gdl_matrix_long_const_view
gdl_matrix_long_const_view_array (const long * base,
                                   const size_t n1, 
                                   const size_t n2);

_gdl_matrix_long_const_view
gdl_matrix_long_const_view_array_with_tda (const long * base, 
                                            const size_t n1, 
                                            const size_t n2,
                                            const size_t tda);

_gdl_matrix_long_const_view
gdl_matrix_long_const_view_vector (const gdl_vector_long * v,
                                    const size_t n1, 
                                    const size_t n2);

_gdl_matrix_long_const_view
gdl_matrix_long_const_view_vector_with_tda (const gdl_vector_long * v,
                                             const size_t n1, 
                                             const size_t n2,
                                             const size_t tda);

/* Operations */

long   gdl_matrix_long_get(const gdl_matrix_long * m, const size_t i, const size_t j);
void    gdl_matrix_long_set(gdl_matrix_long * m, const size_t i, const size_t j, const long x);

long * gdl_matrix_long_ptr(gdl_matrix_long * m, const size_t i, const size_t j);
const long * gdl_matrix_long_const_ptr(const gdl_matrix_long * m, const size_t i, const size_t j);

void gdl_matrix_long_set_zero (gdl_matrix_long * m);
void gdl_matrix_long_set_identity (gdl_matrix_long * m);
void gdl_matrix_long_set_all (gdl_matrix_long * m, long x);

gdl_matrix_long * gdl_matrix_long_fread (FILE * stream) ;
int gdl_matrix_long_fwrite (FILE * stream, const gdl_matrix_long * m) ;
int gdl_matrix_long_fscanf (FILE * stream, gdl_matrix_long * m);
int gdl_matrix_long_fprintf (FILE * stream, const gdl_matrix_long * m, const char * format);
 
int gdl_matrix_long_memcpy(gdl_matrix_long * dest, const gdl_matrix_long * src);
int gdl_matrix_long_swap(gdl_matrix_long * m1, gdl_matrix_long * m2);

int gdl_matrix_long_swap_rows(gdl_matrix_long * m, const size_t i, const size_t j);
int gdl_matrix_long_swap_columns(gdl_matrix_long * m, const size_t i, const size_t j);
int gdl_matrix_long_swap_rowcol(gdl_matrix_long * m, const size_t i, const size_t j);
int gdl_matrix_long_transpose (gdl_matrix_long * m);
int gdl_matrix_long_transpose_memcpy (gdl_matrix_long * dest, const gdl_matrix_long * src);

long gdl_matrix_long_max (const gdl_matrix_long * m);
long gdl_matrix_long_min (const gdl_matrix_long * m);
void gdl_matrix_long_minmax (const gdl_matrix_long * m, long * min_out, long * max_out);

void gdl_matrix_long_max_index (const gdl_matrix_long * m, size_t * imax, size_t *jmax);
void gdl_matrix_long_min_index (const gdl_matrix_long * m, size_t * imin, size_t *jmin);
void gdl_matrix_long_minmax_index (const gdl_matrix_long * m, size_t * imin, size_t * jmin, size_t * imax, size_t * jmax);

int gdl_matrix_long_isnull (const gdl_matrix_long * m);

int gdl_matrix_long_add (gdl_matrix_long * a, const gdl_matrix_long * b);
int gdl_matrix_long_sub (gdl_matrix_long * a, const gdl_matrix_long * b);
int gdl_matrix_long_mul_elements (gdl_matrix_long * a, const gdl_matrix_long * b);
int gdl_matrix_long_div_elements (gdl_matrix_long * a, const gdl_matrix_long * b);
int gdl_matrix_long_scale (gdl_matrix_long * a, const double x);
int gdl_matrix_long_add_constant (gdl_matrix_long * a, const double x);
int gdl_matrix_long_add_diagonal (gdl_matrix_long * a, const double x);

/***********************************************************************/
/* The functions below are obsolete                                    */
/***********************************************************************/
int gdl_matrix_long_get_row(gdl_vector_long * v, const gdl_matrix_long * m, const size_t i);
int gdl_matrix_long_get_col(gdl_vector_long * v, const gdl_matrix_long * m, const size_t j);
int gdl_matrix_long_set_row(gdl_matrix_long * m, const size_t i, const gdl_vector_long * v);
int gdl_matrix_long_set_col(gdl_matrix_long * m, const size_t j, const gdl_vector_long * v);

/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline 
long
gdl_matrix_long_get(const gdl_matrix_long * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VAL("first index out of range", GDL_EINVAL, 0) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VAL("second index out of range", GDL_EINVAL, 0) ;
    }
#endif
  return m->data[i * m->tda + j] ;
} 

extern inline 
void
gdl_matrix_long_set(gdl_matrix_long * m, const size_t i, const size_t j, const long x)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VOID("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VOID("second index out of range", GDL_EINVAL) ;
    }
#endif
  m->data[i * m->tda + j] = x ;
}

extern inline 
long *
gdl_matrix_long_ptr(gdl_matrix_long * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (long *) (m->data + (i * m->tda + j)) ;
} 

extern inline 
const long *
gdl_matrix_long_const_ptr(const gdl_matrix_long * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (const long *) (m->data + (i * m->tda + j)) ;
} 

#endif

__END_DECLS

#endif /* __GDL_MATRIX_LONG_H__ */
